#!/bin/sh

##function: 自动获取ESXi(vmware)系统的设备硬件信息
##author:   10049@20221026
##params:   无
##example:  ./get_device_info_esxi.sh

# 要这么解析也是没办法的事，vmware没有正常的linux信息查询工具，vmware也不用bash，sh里连数组都不支持。
# 而且vmware的工具原始输出连个空行分隔都没有！！！
# 要解析的信息类似
# AAAA
#   xxx: www
#   yyy: zzz
# AAAA
#   xxx: sss
#   yyy: kkk
# BBBB
#   xxxxxxxxxxx
# 参数1：所有内容，上面的示例里就是整个这种信息
# 参数2：行数，上面的示例里就是两个AAAA的行数和一个BBBB的行数，也就是说上面的示例需要3个值在这个参数里
# 参数3：回调函数，其回调参数为解析到的结果，用来正儿八经的解析数据。每个硬件解析都会调用，所以可以实现调用一次回调获取一个硬件的所有信息
loop2parse()
{
    local message=$1
    local lines=$2
    local callback=$3
    local i
    local start=-1
    for i in ${lines}; do
        if [ "$start" -ne -1 ]; then
            $callback "$(echo "$message" | sed -n "${start},$((i - 1))p")"
        fi
        start=$i
    done
}

# 获取一个cpu信息
# 参数为一个cpu的详细输出
get_cpu_i_info()
{
    local message=$1
    if ! echo "$message" | sed -n '/Processor/=' | grep -x 1 >/dev/null; then
        return 1
    fi
    if echo "$message" | grep -E '^[\ ]*Socket Status' | grep -iw 'Empty' >/dev/null; then
        # 空的，不显示
        return 0
    fi
    local model=$(echo "$message" | grep -E '^[\ ]*Version' | awk -F ': ' '{print $2}' | sed 's/"//g' | head -n 1)
    local man=$(echo "$message" | grep -E '^[\ ]*Manufacturer' | awk -F ': ' '{print $2}' | sed 's/"//g' | head -n 1)
    local core=$(echo "$message" | grep -E '^[\ ]*Core Count' | awk -F ': ' '{print $2}' | sed 's/#//g' | head -n 1)
    local micro="N/A"
    [ x"$model" == x ] && return 1
    if [ "$format" == "parse" ]; then
        echo "CPU:                  $model;$man;$core;$micro"
    elif [ "$format" == "csv" ]; then
        echo "型号: ${model}, 厂商: ${man}, 核数: ${core}"
    fi
}

# 获取所有cpu信息
get_cpu_info()
{
    local message=$(smbiosDump)
    message="${message}"$'\n'"  MyEnd"   # 防止要读取的内容出现在最后一行，按照loop2parse的玩法，得提供结束的行，加上一行刚好解决这个问题
    local lines=$(echo "$message" | grep -nE '^[\ ]{2}[^\ ]' | grep -E -A1 '^[0-9]+:[\ ]*Processor' | awk -F ':' '$1~/^[0-9]+$/{print $1}')
    if [ "$format" == "parse" ]; then
        loop2parse "$message" "$lines" "get_cpu_i_info"
    elif [ "$format" == "csv" ]; then
        echo -n "$(escape_csv_cell "$(device_merge "$(loop2parse "$message" "$lines" "get_cpu_i_info")")")"
    fi
}

# 获取一张内存条信息
get_memory_i_info()
{
    local message=$1
    if ! echo "$message" | sed -n '/Memory Device/=' | grep -x 1 >/dev/null; then
        return 1
    fi
    local man=$(echo "$message" | grep -E '^[\ ]*Manufacturer' | awk -F ': ' '{print $2}' | sed 's/"//g' | head -n 1)
    local model=$(echo "$message" | grep -E '^[\ ]*Part Number' | awk -F ': ' '{print $2}' | sed 's/"//g' | head -n 1)
    local size=$(echo "$message" | grep -E '^[\ ]*Size' | awk -F ': ' '{print $2}' | head -n 1)
    local spec=$(echo "$message" | grep -E '^[\ ]*Type:' | awk -F ': ' '{print $2}' | sed -rn 's/.*\((.*)\)/\1/p' | head -n 1)
    local speed=$(echo "$message" | grep -E '^[\ ]*Speed' | awk -F ': ' '{print $2}' | head -n 1)
    if [ x"$size" == x"" ] || echo "$size" | grep -iw 'No' >/dev/null; then
        return 0
    fi
    if [ "$format" == "parse" ]; then
        echo "内存:                 $man;$model;$size;$spec;$speed"
    elif [ "$format" == "csv" ]; then
        echo "型号: ${model}, 厂商: ${man}, 大小: ${size}, 类型: ${spec}, 速度: ${speed}"
    fi
}

# 获取所有内存条信息
get_memory_info()
{
    local message=$(smbiosDump)
    message="${message}"$'\n'"  MyEnd"
    local lines=$(echo "$message" | grep -nE '^[\ ]{2}[^\ ]' | grep -E -A1 '^[0-9]+:[\ ]*Memory Device' | awk -F ':' '$1~/^[0-9]+$/{print $1}')
    if [ "$format" == "parse" ]; then
        loop2parse "$message" "$lines" "get_memory_i_info"
    elif [ "$format" == "csv" ]; then
        echo -n "$(escape_csv_cell "$(device_merge "$(loop2parse "$message" "$lines" "get_memory_i_info")")")"
    fi
}

# 获取一个硬盘信息
get_hd_i_info()
{
    local message=$1
    local model=$(echo "$message" | grep -E '^[\ ]*Model' | awk -F ': ' '{print $2}' | sed -r 's/[\ ]*$//g')
    local man=$(echo "$message" | grep -E '^[\ ]*Vendor' | awk -F ': ' '{print $2}' | sed -r 's/[\ ]*$//g')
    local serial="N/A"
    local cap=$(echo "$message" | grep -E '^[\ ]*Size' | awk -F ': ' '{print $2}')
    local fwversion="N/A"
    local iftype="N/A"
    local hd_type="N/A"
    if [ x"$cap" == x"" ] || [ "$cap" -eq 0 ]; then
        return 0
    fi
    cap="${cap} MB"
    if [ "$format" == "parse" ]; then
        echo "硬盘:                 $model;$man;$serial;$cap;$fwversion;$iftype;$hd_type"
    elif [ "$format" == "csv" ]; then
        echo "型号: ${model}, 厂商: ${man}, 大小: ${cap}, 固件版本: ${fwversion}, 接口类型: ${iftype}, 硬盘类型: ${hd_type}"
    fi
}

# 获取所有硬盘信息
# 由于vmware良心发现，获取到的每个硬盘之间，是有空行分隔的，所以不需要用loop2parse来解析硬盘了
get_hd_info()
{
    local hdlist=$(esxcli storage core device list | grep -Eo '^[^\ ].*$')
    local i
    if [ "$format" == "parse" ]; then
        for i in ${hdlist}; do
            get_hd_i_info "$(esxcli storage core device list | sed -n "/^${i}$/,/^$/p")"
        done
    elif [ "$format" == "csv" ]; then
        local data=""
        for i in ${hdlist}; do
            if [ x"$data" != x ]; then
                data="$data"$'\n'
            fi
            data="${data}$(get_hd_i_info "$(esxcli storage core device list | sed -n "/^${i}$/,/^$/p")")"
        done
        echo -n "$(escape_csv_cell "$(device_merge "$data")")"
    fi
}

# 根据卡槽编号获取PCI信息
# 参数1为卡槽编号，如：0000:01:00.0
# 返回示例：
#    |----Function........................................0x00
#    |----Runtime Owner...................................vmkernel
#    |----Has Configured Owner............................false
#    |----Configured Owner................................vmkernel
#    |----Vendor Id.......................................0x8086
#    |----Device Id.......................................0x1521
#    |----Sub-Vendor Id...................................0x1028
#    |----Sub-Device Id...................................0x1f9a
#    |----Vendor Name.....................................Intel Corporation
#    |----Device Name.....................................Gigabit 4P X710/I350 rNDC
#    |----Device Class....................................512
#    |----Device Class Name...............................Ethernet controller
#    |----PIC Line........................................11
#    |----Old IRQ.........................................255
#    |----Vector..........................................0
#    |----PCI Pin.........................................3
#    |----Spawned Bus.....................................0
#    |----Flags...........................................12801
#    \==+BAR Info :
get_pci_info()
{
    # 参数格式 Segment:Bus:Slot.Function 如：0000:01:00.0
    # 在esxcfg-info输出里的PCI Device里看起来是
    # \==+All Pci Devices :
    #    \==+PCI Device :
    #       |----Segment......................0x0000
    #       |----Bus..........................0x01
    #       |----Slot.........................0x00
    #       |----Function.....................0x00
    #       xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
    #       xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
    #       \==+BAR Info :
    #       xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
    #    \==+PCI Device :
    #    xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
    # \==+VmKernel Pci Devices :
    # xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
    # 获取Function到BAR Info
    
    local pci=$1
    local message=$(esxcfg-info 2>/dev/null | sed -rn '/\\==\+All Pci Devices :/,/\\==\+VmKernel Pci Devices :/p')
    local segment="0x${pci%%:*}"
    pci=${pci#*:}
    local bus="0x${pci%%:*}"
    pci=${pci#*:}
    local slot="0x${pci%.*}"
    local funct="0x0${pci#*.}"
    message=$(echo "$message" | sed -rn '/\\==\+PCI Device :/,/\==\+BAR Info :/p')
    echo "$message" |
        sed -rn "/[|]----Segment[.]+${segment}/,/\==\+BAR Info :/p" |
        sed -rn "/[|]----Bus[.]+${bus}/,/\==\+BAR Info :/p" |
        sed -rn "/[|]----Slot[.]+${slot}/,/\==\+BAR Info :/p" |
        sed -rn "/[|]----Function[.]+${funct}/,/\==\+BAR Info :/p"
}

# 获取所有网卡信息
get_netcard_info()
{
    # 这样写看起来高端，但是效率太低了，而且不支持远程调用（用到了$0这种自反射，当把命令传到远程环境执行的时候，$0是-bash，会获取不到脚本文件名）
    # esxcli network nic list | tail -n +3 |
    #     awk -F '[\ ]{2,}' '{
    #         pcidatacmd="'$0' -c get_pci_info "$2;
    #         printf "网卡:                 "$10";"$10";";
    #         system(pcidatacmd" | grep -- \"--Vendor Name\" | awk -F \"[.]+\" \"{printf \\\$2}\"");
    #         printf ";"
    #         system("esxcli network nic get -n "$1" | grep \"Firmware Version\" | awk -F \": \" \"{printf \\\$2}\"");
    #         printf ";"$3";[";
    #         system(pcidatacmd" | grep -- \"--Vendor Id\" | awk -F \"[.]+\" \"{printf \\\$2}\" | sed \"s/0x//g; s/[\\ ]//g\"");
    #         printf ":";
    #         system(pcidatacmd" | grep -- \"--Device Id\" | awk -F \"[.]+\" \"{printf \\\$2}\" | sed \"s/0x//g; s/[\\ ]//g\"");
    #         print "]";
    #     }'

    # 换成高效率且支持远程调用的写法。对于高手来说为了效率是完全不介意代码完全重写的
    local nics=$(esxcli network nic list | tail -n +3 | awk -F '[\ ]{2,}' '{print $1}')
    local slots=$(esxcli network nic list | tail -n +3 | awk -F '[\ ]{2,}' '{print $2}')
    local drivers=$(esxcli network nic list | tail -n +3 | awk -F '[\ ]{2,}' '{print $3}')
    local ctrls=$(esxcli network nic list | tail -n +3 | awk -F '[\ ]{2,}' '{print $10}')
    local idx=0
    local alldata=""
    for slot in $slots; do  # 注意：ESXi不支持数组，不要用分割然后"${xxx[@]}"的方法，没有用的
        let idx++   # ESXi不支持数组，只能用这种方法同步slots和ctrls遍历到的序号（或者说是我的水平不够，只能想到这种办法）
        local pci_info=$(get_pci_info $slot)
        
        local controller=$(echo "$ctrls" | sed -n "${idx}p")
        local subsystem=$(echo "$ctrls" | sed -n "${idx}p")
        local model=$(echo "$ctrls" | sed -n "${idx}p")
        local man=$(echo "$pci_info" | grep -- "--Vendor Name" | awk -F '[.]+' '{print $2}')
        local fw=$(esxcli network nic get -n "$(echo "$nics" | sed -n "${idx}p")" | grep 'Firmware Version' | awk -F ':' '{print $2}')
        local driver=$(echo "$drivers" | sed -n "${idx}p")
        local vendorid=$(echo "$pci_info" | grep -- '--Vendor Id' | awk -F '[.]+' '{print $2}' | sed 's/0x//g; s/[\ ]//g')
        local deviceid=$(echo "$pci_info" | grep -- '--Device Id' | awk -F '[.]+' '{print $2}' | sed 's/0x//g; s/[\ ]//g')

        if [ "$format" == "parse" ]; then
            echo "网卡:                 $controller;$subsystem;$model;$man;$fw;$driver;[${vendorid}:${deviceid}]"
        elif [ "$format" == "csv" ]; then
            if [ x"$alldata" != x ]; then
                alldata="$alldata"$'\n'
            fi
            alldata="${alldata}型号: ${model}, 厂商: ${man}, 固件版本: ${fw}, 驱动: ${driver}, 设备id: [$vendorid:$deviceid]"
        fi
    done
    if [ "$format" == "csv" ]; then
        echo -n "$(escape_csv_cell "$(device_merge "$alldata")")"
    fi
}

# 获取所有raid卡信息
get_raid_info()
{
    if [ "$format" == "parse" ]; then
        echo "RAID卡:               "
    elif [ "$format" == "csv" ]; then
        echo -n ""
    fi
}

get_gpu_info()
{
    if [ "$format" == "parse" ]; then
        echo "GPU:                  "
    elif [ "$format" == "csv" ]; then
        echo -n ""
    fi
}

# 获取所有hba卡信息
get_hba_info()
{
    # esxcfg-scsidevs -a | awk '{print $2,$5}' | sed -rn 's/^([^\ ]+).*\((.*)\).*$/\1 \2/p' |
    #     awk '{
    #         if (NF != 2) {
    #             return 0;
    #         }
    #         pcidatacmd="'$0' -c get_pci_info "$2;
    #         printf "HBA卡:                ";
    #         system(pcidatacmd" | grep -- \"--Device Name\" | awk -F \"[.]+\" \"{printf \\\$2}\"");
    #         printf ";";
    #         system(pcidatacmd" | grep -- \"--Device Name\" | awk -F \"[.]+\" \"{printf \\\$2}\"");
    #         printf ";";
    #         system(pcidatacmd" | grep -- \"--Vendor Name\" | awk -F \"[.]+\" \"{printf \\\$2}\"");
    #         printf ";N/A;"$1";[";
    #         system(pcidatacmd" | grep -- \"--Vendor Id\" | awk -F \"[.]+\" \"{printf \\\$2}\" | sed \"s/0x//g; s/[\\ ]//g\"");
    #         printf ":";
    #         system(pcidatacmd" | grep -- \"--Device Id\" | awk -F \"[.]+\" \"{printf \\\$2}\" | sed \"s/0x//g; s/[\\ ]//g\"");
    #         print "]";
    #     }'

    # 改成高效率且能够远程调用
    local drivers=$(esxcfg-scsidevs -a | awk '{print $2,$5}' | sed -rn 's/^([^\ ]+).*\((.*)\).*$/\1/p')
    local slots=$(esxcfg-scsidevs -a | awk '{print $2,$5}' | sed -rn 's/^([^\ ]+).*\((.*)\).*$/\2/p')
    local idx=0
    local alldata=""
    for slot in $slots; do
        let idx++
        local pci_info=$(get_pci_info $slot)

        local controller=$(echo "$pci_info" | grep -- '--Device Name' | awk -F '[.]+' '{print $2}')
        local subsystem=$(echo "$pci_info" | grep -- '--Device Name' | awk -F '[.]+' '{print $2}')
        local model=$(echo "$pci_info" | grep -- '--Device Name' | awk -F '[.]+' '{print $2}')
        local man=$(echo "$pci_info" | grep -- '--Vendor Name' | awk -F '[.]+' '{print $2}')
        local fw="N/A"
        local driver=$(echo "$drivers" | sed -n "${idx}p")
        local vendorid=$(echo "$pci_info" | grep -- '--Vendor Id' | awk -F '[.]+' '{print $2}' | sed 's/0x//g; s/[\ ]//g')
        local deviceid=$(echo "$pci_info" | grep -- '--Device Id' | awk -F '[.]+' '{print $2}' | sed 's/0x//g; s/[\ ]//g')

        if [ "$format" == "parse" ]; then
            echo "HBA卡:                $controller;$subsystem;$model;$man;$fw;$driver;[$vendorid:$deviceid]"
        elif [ "$format" == "csv" ]; then
            if [ x"$alldata" != x ]; then
                alldata="$alldata"$'\n'
            fi
            alldata="${alldata}型号: ${model}, 厂商: ${man}, 固件版本: ${fw}, 驱动: ${driver}, 设备id: [$vendorid:$deviceid]"
        fi
    done
    if [ "$format" == "csv" ]; then
        echo -n "$(escape_csv_cell "$(device_merge "$alldata")")"
    fi
}

print_csv_bar()
{
    echo "版本,服务器架构,BMC版本,主板型号,服务器厂商,系统管理,服务器型号,BIOS版本,CPU,内存,硬盘,RAID,GPU,HBA,网卡,外置存储"
    return 0
}

format=${format:-"parse"}
csv_server_name=${csv_server_name:-""}
csv_output_bar=${csv_output_bar:-"yes"}

usage()
{
cat << EOF
在ESXi系统下获取机器的硬件信息
使用方法 $0 [-f <格式>][-n <备注>][-b <是否>]
参数支持：
    -h | -? | --help                输出本信息并退出
    -c | --call <函数> [参数...]    直接调用本脚本里的函数
    -v | --version                  查看脚本版本
    -f | --format <格式>            设置输出格式，支持"parse"和"csv"
    -n | --csv-server-name <备注>   给获取到的信息设置备注，批量收集时效果拔群
    -b | --csv-output-bar <是否>    设置是否输出表头，"yes"或"no"，批量收集时效果拔群
EOF
}

# 解析参数
while [ $# -gt 0 ]; do
    case x"$1" in
        x"-h"|x"-?"|x"--help")
            usage
            exit 0
            ;;
        x"-c"|x"--call")
            shift
            funct=$1
            shift
            $funct "$@"
            exit $?
            ;;
        x"-v"|x"--version")
            cat "$0" | grep '^##author' | awk -F '@' '{print $2}' | sed 's/\ //g'
            exit 0
            ;;
        x"-f"|x"--format")
            if [ $# -lt 2 ]; then
                echo "选项\"$1\"需要指定硬件信息输出格式" >&2
                exit 1
            fi
            format=$2
            shift
            ;;
        x"-n"|x"--csv-server-name")
            if [ $# -lt 2 ]; then
                echo "选项\"$1\"需要设置服务器备注" >&2
                exit 1
            fi
            csv_server_name=$2
            shift
            ;;
        x"-b"|x"--csv-output-bar")
            if [ $# -lt 2 ]; then
                echo "选项\"$1\"需要设置是否显示csv表头" >&2
                exit 1
            fi
            csv_output_bar=$2
            shift
            ;;
        x"-p"|x"--print-bar")
            print_csv_bar
            exit 0
            ;;
        *)
            echo "无效的参数\"$1\"" >&2
            exit 1
            ;;
    esac
    shift
done

# 显示同名设备个数
# 参数1为搜索结果包含重复的情况下的设备列表，以换行符分隔
# 参数2为搜索结果除去重复的情况下的设备列表，以换行符分隔
get_device_count(){
    local result=""
    local old_IFS=$IFS
    IFS=$'\n'
    for i in $2; do
        local count=$(echo "$1" | grep "$i" -Fx | wc -l)
        result=${result}${i}' ['"${count}个"']'$'\n'
    done
    IFS=$old_IFS
    result=${result%$'\n'*}
    echo "$result"
    return 0
}

# 把重复的合并
# 参数示例：
# aaaa
# bbbb
# aaaa
# cccc
# 输出示例
# aaaa [2个]
# bbbb [1个]
# cccc [1个]
device_merge()
{
    get_device_count "$1" "$(echo "$1" | sort | uniq)"
}

# csv格式转义
# csv特殊字符有逗号、换行符、双引号
# 其中在两边加双引号可以解决逗号和换行符的问题
# 之后把内容里所有一个双引号替换成两个双引号即可
# 在最前面加上等号，可以强行使用文本模式，防止版本号等变成数字
escape_csv_cell(){
    if [ $(echo "$1" | wc -l) -eq 1 ] && echo "$1" | grep -Ex '[0-9.]+' >/dev/null; then
        echo '="'"${1//\"/\"\"}"'"'
    else
        echo '"'"${1//\"/\"\"}"'"'
    fi
    return 0
}

main()
{
    if [ "$format" == "parse" ]; then
        echo "系统信息:             $(uname -a)"
        echo "版本:                 $(vmware -v)"
        echo "平台类型:             $(uname -m)"
        echo "BMC版本:              $(esxcfg-info -a 2>/dev/null | grep -E '^[\ ]*\|-*BMC Version' | sed -rn 's/^[^.]*\.*(.*)/\1/p')"
        echo "主板型号:             $(esxcfg-info -a 2>/dev/null | grep -E '^[\ ]*\|-*Product Name' | sed -rn 's/^[^.]*\.*(.*)/\1/p')"
        echo "服务器厂商:           $(esxcfg-info -a 2>/dev/null | grep -E '^[\ ]*\|-*Vendor Name' | sed -rn 's/^[^.]*\.*(.*)/\1/p' | head -n 1)"
        echo "服务器系统管理:       N/A"
        echo "服务器型号:           $(smbiosDump | sed -n '/^  \(Base\)\{0,1\}[Bb]oard/,/^  [^\ ]/p' | awk '$1=="Product:"{print $2}' | sed 's/"//g')"
        echo "BIOS版本:             $(esxcfg-info -a 2>/dev/null | grep -E '^[\ ]*\|-*BIOS Version' | sed -rn 's/^[^.]*\.*(.*)/\1/p')"
        get_cpu_info
        get_memory_info
        get_hd_info
        get_raid_info
        get_gpu_info
        get_hba_info
        get_netcard_info
    elif [ "$format" == "csv" ]; then
        if [ "$csv_output_bar" == "yes" ]; then
            if [ x"$csv_server_name" != x ]; then
                echo -n ","
            fi
            print_csv_bar
        fi
        if [ x"$csv_server_name" != x ]; then
            echo -n "$(escape_csv_cell "$csv_server_name"),;"
        fi
        echo -n "$(escape_csv_cell "$(vmware -v)"),;"
        echo -n "$(escape_csv_cell "$(uname -m)"),;"
        echo -n "$(escape_csv_cell "$(esxcfg-info -a 2>/dev/null | grep -E '^[\ ]*\|-*BMC Version' | sed -rn 's/^[^.]*\.*(.*)/\1/p')"),;"
        echo -n "$(escape_csv_cell "$(esxcfg-info -a 2>/dev/null | grep -E '^[\ ]*\|-*Product Name' | sed -rn 's/^[^.]*\.*(.*)/\1/p')"),;"
        echo -n "$(escape_csv_cell "$(esxcfg-info -a 2>/dev/null | grep -E '^[\ ]*\|-*Vendor Name' | sed -rn 's/^[^.]*\.*(.*)/\1/p' | head -n 1)"),;"
        echo -n ",;"
        echo -n "$(escape_csv_cell "$(smbiosDump | sed -n '/^  \(Base\)\{0,1\}[Bb]oard/,/^  [^\ ]/p' | awk '$1=="Product:"{print $2}' | sed 's/"//g')"),;"
        echo -n "$(escape_csv_cell "$(esxcfg-info -a 2>/dev/null | grep -E '^[\ ]*\|-*BIOS Version' | sed -rn 's/^[^.]*\.*(.*)/\1/p')"),;"
        get_cpu_info
        echo -n ",;"
        get_memory_info
        echo -n ",;"
        get_hd_info
        echo -n ",;"
        get_raid_info
        echo -n ",;"
        get_gpu_info
        echo -n ",;"
        get_hba_info
        echo -n ",;"
        get_netcard_info
        echo -n ",;"
        echo -n ""
        echo
    fi
}

# 校验参数
if ! echo "$format" | grep -Ex 'parse|csv' >/dev/null; then
    echo "输出格式只支持\"parse\"和\"csv\"，不支持你设置的\"$format\"" >&2
    exit 1
fi

main
