#!/bin/bash

##function: 自动获取设备硬件信息
##author:   10049@20230117
##params:   -f或--format指定输出格式
##example:  ./get_device_info.sh

# 最后编辑于 2023-09-18

# 信息包括：
# 服务器厂商
# 系统管理
# 服务器型号
# BIOS版本
# CPU
# 内存
# 硬盘
# RAID卡
# GPU
# HBA卡
# 网卡

if [ "$0" == "${BASH_SOURCE}" ]; then
    THIS_FILE=$(readlink -f "$0")
    CUR_DIR=$(dirname "$THIS_FILE")
else
    CUR_DIR="/root"
fi

FRAMEWORK=x86
if uname -m | grep aarch >/dev/null; then
    FRAMEWORK=arm
fi

RAIDTOOLS_DIR="$CUR_DIR/get_device_info_tools/$FRAMEWORK/raidtools"

# LONGBIT=$(getconf LONG_BIT)

FAILED=no
TIMEOUTCMD="timeout"
TIMEOUTCODE=124
timeout -t 2>/dev/null
ret=$?
if [ "$ret" -eq 1 ]; then
    TIMEOUTCMD="timeout -t"
    TIMEOUTCODE=143
fi

# 操作系统类型
ostype=${ostype:-"HCI"}

# raidtools使用系统自带工具还是本脚本指定工具
raidtools_use=${raidtools_use:-"system"}

# 是否只加载函数
loadonly=${loadonly:-"no"}

# 输出格式
format=${format:-"parse"}

# csv格式选项，是否输出表头（其他格式此选项无效）
csv_output_bar=${csv_output_bar:-"yes"}

# csv格式选项，服务器名字（其他格式此选项无效）
csv_server_name=${csv_server_name:-""}

# 分隔符
split_symb=${split_symb:-";"}

# 是否输出ng需要的信息
extend_ng_info=${extend_ng_info:-"no"}

# ng需要的信息输出路径，如果是-则输出到标准输出
extend_ng_filename=${extend_ng_filename:-"ng_info.tar.gz"}

output_lang=${output_lang:-"zh"}    # zh表示中文，en表示英文

# 调试模式控制变量，默认为关闭
debug_main=${debug_main:-"no"}

# GPU信息字典，存储所有GPU的详细信息
declare -A GPU_INFO_DICT

# 缓存目录
RAIDSTAT_CACHE_DIR="/tmp/get_device_info_raidstat_cache_$$"
mkdir -p "$RAIDSTAT_CACHE_DIR"

# 清理函数，用于删除临时目录
cleanup_temp_dir() {
    if [ -d "$RAIDSTAT_CACHE_DIR" ]; then
        rm -rf "$RAIDSTAT_CACHE_DIR"
    fi
}

# 设置退出时自动清理临时目录
trap cleanup_temp_dir EXIT

# 需要获取的信息（默认为全部）
output_hardware_type=${output_hardware_type:-"cpu,memory,disk,raid,gpu,hba,network,ex_storage,power,usb,server"}

# 需要获取的硬件信息的每一项
OUTPUT_CPU_INFO=yes
OUTPUT_MEMORY_INFO=yes
OUTPUT_DISK_INFO=yes
OUTPUT_RAID_INFO=yes
OUTPUT_GPU_INFO=yes
OUTPUT_HBA_INFO=yes
OUTPUT_NETWORK_INFO=yes
OUTPUT_EX_STORAGE_INFO=yes
OUTPUT_POWET_INFO=yes
OUTPUT_USB_INFO=yes
OUTPUT_SERVER_INFO=yes

scan_output_hardware_type(){
    local dont_output_xxx_info=$(set | sed -rn 's/^OUTPUT_([A-Z_]+)_INFO=yes$/OUTPUT_\1_INFO=no/p')
    eval "$dont_output_xxx_info"
    local old_IFS=$IFS
    IFS=','
    local h=($output_hardware_type)
    IFS=$old_IFS
    local i
    for i in "${h[@]}"; do
        eval "OUTPUT_${i^^}_INFO=yes"
    done
}

# 获取服务器厂商
get_server_man(){
    local man=$(dmidecode -t system | grep 'Manufacturer:' | awk -F ': ' '{print $2}')
    man=$(echo $man)    # 去掉多个空格，全是空格也过滤掉
    if [ -z "$man" ]; then
        echo "N/A"
        return 1
    fi
    echo "$man"
    return 0
}

# 获取系统管理
get_system_manage(){
    if [ -e '/sys/firmware/efi' ]; then
        echo 'UEFI'
    else
        echo 'legacy'
    fi
    return 0
}

# 获取深信服设备的型号
get_server_model.sangfor(){
    local model=$(dmidecode -t system | grep 'Family:' | awk -F ': ' '{print $2}')
    if echo "$model" | grep -iEw 'Purley|Default string' >/dev/null; then
        model=$(dmidecode -t chassis | grep -E '[^_]+_[^_]+_[^_]+-' | awk -F ': ' '{print $2}')
    fi
    if [ -z "$model" ]; then
        return 1
    fi
    if echo "$model" | grep -iEw 'filled|O\.E\.M\.' >/dev/null; then
        return 1
    fi
    echo "$model"
    return 0
}

# 键为匹配关系（匹配到这个正则表达式就调用值指向的函数，不区分大小写），值为调用的函数
get_server_model_map=(
    "SANGFOR" "get_server_model.sangfor"
)

# 获取常规服务器型号
get_server_model.norm(){
    local model=$(dmidecode -t system | grep 'Product Name:' | awk -F ': ' '{print $2}' | sed -r 's/[.]{3,}//g')
    model=$(echo $model)
    if [ -z "$model" ]; then
        echo "N/A"
        return 1
    fi
    echo "$model"
    return 0
}

get_server_model(){
    local man=$(get_server_man)
    local i
    for ((i=0; i<${#get_server_model_map[@]}; i+=2)); do
        if echo "$man" | grep -Eix "${get_server_model_map[$i]}" >/dev/null; then
            if ${get_server_model_map[$(($i+1))]}; then     # 如果成功就这么获取，如果不成功就继续尝试其他方法，都不行就用回通常的方法
                return 0
            fi
        fi
    done
    get_server_model.norm
    return $?
}

get_serial_number(){
    if dmidecode -t 3 | grep "Version:" | grep -qi "HUAWEI_\|XFUSION_\|KUNLUN_"; then
        dmidecode -t 2 | grep "Asset Tag:" | awk -F ': ' '{print $2}'
        return 0
    fi
    dmidecode -t 2 | grep "Serial Number" | awk -F ': ' '{print $2}'
}

# 获取BIOS版本
get_bios_version(){
    local version=$(dmidecode -t bios | grep 'Version:' | awk -F ': ' '{print $2}')
    version=$(echo $version)
    if [ -z "$version" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$version"
    return 0
}

# 显示同名设备个数
# 参数1为搜索结果包含重复的情况下的设备列表，以换行符分隔
# 参数2为搜索结果除去重复的情况下的设备列表，以换行符分隔
get_device_count(){
    local result=""
    local old_IFS=$IFS
    IFS=$'\n'
    for i in $2; do
        local count=$(echo "$1" | grep "$i" -Fx | wc -l)
        result=${result}${i}' ['"${count}个"']'$'\n'
    done
    IFS=$old_IFS
    result=${result%$'\n'*}
    echo "$result"
    return 0
}

# 获取CPU型号
get_cpu_model(){
    local model=$(dmidecode -t processor | grep 'Version:' | grep -iv 'Not Specified\|Not Installed' | awk -F ': ' '{print $2}' | head -n 1)
    if [ -z "$model" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$model"
    return 0
}

# 获取CPU主频
get_cpu_clock_speed(){
    local speed=$(dmidecode -t processor | sed -rn 's/Current Speed: (.*) MHz/scale=2; \1 \/ 1000/p' | head -n 1 | bc)
    if [ -z "$speed" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$speed GHz"
    return 0
}

# 获取CPU厂商
get_cpu_man(){
    local man=$(dmidecode -t processor | grep 'Manufacturer:' | grep -iv 'Not Specified\|Unknown' | awk -F ': ' '{print $2}' | head -n 1)
    if [ -z "$man" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$man"
    return 0
}

# 获取CPU架构
get_cpu_frame(){
    local firmware=$(uname -m)
    if [ -z "$firmware" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$firmware"
    return 0
}

# 获取平台类型（基本就是获取CPU架构，但是对x86的情况下判断软件版本是否为c86）
get_hci_type(){
    local firmware=$(uname -m)
    if [ "$firmware" != "x86_64" ]; then
        echo "$firmware"
        return 0
    fi
    $TIMEOUTCMD 5 ssh -p 22 localhost ". /sf/bin/platform_check.sh 2>/dev/null && is_c86_platform && echo c86_64 || echo x86_64"
    local ret=$?
    if [ $ret -eq $TIMEOUTCODE ] || [ $ret -eq 255 ]; then
        echo "x86_64"
    fi
}

# 获取CPU微码
get_cpu_microcode(){
    local microcode=$(cat /proc/cpuinfo | grep 'microcode' | awk -F ': ' '{print $2}' | head -n 1)
    if [ -z "$microcode" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$microcode"
    return 0
}

# 获取cpu核数
get_cpu_core(){
    local x86=$(uname -m | grep x86)
    if [ -n "$x86" ]; then
        # 是x86
        local count=$(cat /proc/cpuinfo | grep 'cpu cores' | uniq | awk -F ': ' '{print $2}')
        if [ -z "$count" ]; then
            echo "N/A"
            return 1
        fi
        escape_parse_cell "$count"
    else
        # 是ARM
        echo $(( $(cat /proc/cpuinfo | grep 'processor' | wc -l) / $(dmidecode -t processor | grep '^Processor Information$' | wc -l) ))
    fi
    return 0
}

get_cpu_thread_count(){
    local count=$(dmidecode -t 4 | grep "Thread Count:" | awk -F":" 'NR == 1 {print $2}' | tr -d " ")
    if [ -z "$count" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$count"
}

get_cpu_max_speed(){
    local count=$(dmidecode -t 4 | grep "Max Speed:" | awk -F":" 'NR == 1 {print $2}' | tr -d " ")
    if [ -z "$count" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$count"
}

# 获取cpu线程数（每个cpu总的线程数）
get_cpu_thread(){
    local x86=$(uname -m | grep x86)
    if [ -n "$x86" ]; then
        # 是x86
        echo $(( $(cat /proc/cpuinfo | grep 'processor' | wc -l) / $(cat /proc/cpuinfo | grep 'physical id' | sort | uniq | wc -l) ))
        return 0
    else
        echo "N/A"
        return 1
    fi
}

# 获取cpu个数
get_cpu_count(){
    local x86=$(uname -m | grep x86)
    if [ -n "$x86" ]; then
        # 是x86
        cat '/proc/cpuinfo' | grep 'physical id' | sort | uniq | wc -l
    else
        # 是ARM
        dmidecode -t processor | grep '^Processor Information$' | wc -l
    fi
    return 0
}

# 获取cpu的l2 cache大小
get_cpu_l2_cache()
{
    if ! cat /sys/bus/cpu/devices/cpu0/cache/index2/size; then
        echo "N/A"
        return 1
    fi
    return 0
}

# 获取所有内存信息
get_memory_info(){
    local handles=$(dmidecode -t memory | grep '^Handle [0-9A-Fx]*' -o)
    local old_IFS=$IFS
    IFS=$'\n'
    local handle_array=($handles)
    IFS=$old_IFS
    local result=""
    for i in ${!handle_array[*]}; do
        local mem=$(dmidecode -t memory | sed -n '/^'"${handle_array[$i]}"'/,/^$/p' | sed -n '/^Memory Device$/,/^$/p')
        if [ -n "$mem" ]; then
            if [ -n "$(get_memory_size "$mem")" ]; then
                if [ "$format" == "human" ]; then
                    result=${result}"$(get_memory_model "$mem") (生产商: $(get_memory_man "$mem")) (大小: $(get_memory_size "$mem"), 规格: $(get_memory_spec "$mem"), 速度: $(get_memory_speed "$mem"))"$'\n'
                else
                    result=${result}"$(get_memory_man "$mem")${split_symb}$(get_memory_model "$mem")${split_symb}$(get_memory_size "$mem")${split_symb}$(get_memory_spec "$mem")${split_symb}$(get_memory_speed "$mem")"$'\n'
                fi
            fi
        fi
    done
    if [ -n "$result" ]; then
        if [ "$format" == "human" ]; then
            echo "内存数量: $(echo "$result" | grep -E '^.+$' | wc -l)"
            get_device_count "$result" "$(echo "$result" | sort | uniq)"
        else
            echo "$result"
        fi
        
        return 0
    else
        echo "N/A"
        return 1
    fi
}

# 获取内存速度
get_memory_speed(){
    local result=""
    if [ $# == 0 ]; then
        result=$(dmidecode -t memory | grep $'^[\ \t]*Speed:' | grep -v 'Unknown' | awk -F ': ' '{print $2}' | sort | uniq)
    else
        result=$(echo "$1" | grep $'^[\ \t]*Speed:' | grep -v 'Unknown' | awk -F ': ' '{print $2}')
    fi
    if [ -z "$result" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$result"
    return 0
}

# 获取内存槽位
get_memory_slot(){
    local result=""
    if [ $# == 0 ]; then
        result=$(dmidecode -t memory | grep $'^[\ \t]*Locator:' | awk -F ': ' '{print $2}' | sort | uniq)
    else
        result=$(echo "$1" | grep $'^[\ \t]*Locator:' | awk -F ': ' '{print $2}')
    fi
    if [ -z "$result" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$result"
    return 0
}

# 获取内存条高度
get_memory_height(){
    local result=""
    if [ $# == 0 ]; then
        result=$(dmidecode -t memory | grep $'^[\ \t]*Module Height:' | awk -F ': ' '{print $2}' | sort | uniq)
    else
        result=$(echo "$1" | grep $'^[\ \t]*Module Height:' | awk -F ': ' '{print $2}')
    fi
    if [ -z "$result" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$result"
    return 0
}

# 获取内存条日期
get_memory_date(){
    local result=""
    if [ $# == 0 ]; then
        result=$(dmidecode -t memory | grep $'^[\ \t]*Manufacturing Date:' | awk -F ': ' '{print $2}' | sort | uniq)
    else
        result=$(echo "$1" | grep $'^[\ \t]*Manufacturing Date:' | awk -F ': ' '{print $2}')
    fi
    if [ -z "$result" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$result"
    return 0
}

# 获取内存生产商，如果没有参数就获取所有的，如果有参数就按参数获取
get_memory_man(){
    local result=""
    if [ $# == 0 ]; then
        result=$(dmidecode -t memory | grep 'Manufacturer:' | grep -v 'NO DIMM\|Not Specified\|Undefined' | awk -F ': ' '{print $2}' | sort | uniq)
    else
        result=$(echo "$1" | grep 'Manufacturer:' | grep -v 'NO DIMM\|Not Specified\|Undefined' | awk -F ': ' '{print $2}')
    fi
    if [ -z "$result" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$result"
    return 0
}

# 获取内存大小，无参数则获取所有，有参数则按参数获取
# 当获取不到时输出空，不需要输出“无法获取XXX”这种东西
get_memory_size(){
    if [ $# == 0 ]; then
        get_device_count "$(dmidecode -t memory | grep -E '^[^A-Za-z]*Size: [0-9]+' | awk -F ': ' '{print $2}')" "$(dmidecode -t memory | grep -E '^[^A-Za-z]*Size: [0-9]+' | awk -F ': ' '{print $2}' | sort | uniq)"
    else
        escape_parse_cell "$(echo "$1" | grep -E '^[^A-Za-z]*Size: [0-9]+' | awk -F ': ' '{print $2}')"
    fi
    return $?
}

# 获取内存规格，无参数则获取所有，有参数则按参数获取
get_memory_spec(){
    if [ $# == 0 ]; then
        local tmp=$(dmidecode -t memory | grep -E '^[^A-Za-z]*Size: [0-9]+' -A5)
        if [ -z "$tmp" ]; then
            echo "N/A"
            return 1
        fi
        local p1=$(echo "$tmp" | grep 'Type:' | awk -F ': ' '{print $2}')
        local p2=$(echo "$p1" | sort | uniq)
        get_device_count "$p1" "$p2"
    else
        echo "$1" | grep 'Type:' | awk -F ': ' '{print $2}'
    fi
    return 0
}

# 获取内存型号
get_memory_model(){
    if [ $# == 0 ]; then
        local tmp=$(dmidecode -t memory | grep 'Part Number:' | grep -v 'NO DIMM\|Not Specified\|Undefined' | awk -F ': ' '{print $2}')
        if [ -z "$tmp" ]; then
            echo "N/A"
            return 1
        fi
        get_device_count "$tmp" "$(echo "$tmp" | sort | uniq)"
    else
        local tmp=$(echo "$1" | grep 'Part Number:' | grep -v 'NO DIMM\|Not Specified\|Undefined' | awk -F ': ' '{print $2}')
        if [ -z "$tmp" ]; then
            echo "N/A"
            return 1
        fi
        # 不加双引号是为了去除尾空
        echo $tmp
    fi
    return 0
}

# 获取网口的接口类型
get_net_interface_type(){
    dev_name=$1
    if [[ -z "${dev_name}" ]]; then
        echo "N/A"
        return 1
    fi
    support_ports=$(ethtool "$dev_name" | grep "Supported ports")
    electrical=$(echo $support_ports | grep "TP\|AUI\|BNC\|MII\|GMII\|RGMII\|SGMII\|QSGMII\|XGMII\|TBI\|RTBI\|XAUI\|CX4")
    light=$(echo $support_ports | grep "FIBRE\|KX4\|KR\|KR4\|CAUI")
    if [[ -n "$electrical" ]]; then
        echo "electrical"
    elif [[ -n "$light" ]]; then
        echo "light"
    else
        echo "N/A"
    fi
    return 0
}

# 获取电源厂商
get_power_vendor(){
    local result=""
    if [ $# == 0 ]; then
        result=$(dmidecode -t 39 | grep 'Manufacturer:' | grep -v 'NO DIMM\|Not Specified\|Undefined' | awk -F ': ' '{print $2}' | sort | uniq)
    else
        result=$(echo "$1" | grep 'Manufacturer:' | grep -v 'NO DIMM\|Not Specified\|Undefined' | awk -F ': ' '{print $2}')
    fi
    if [ -z "$result" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$result"
    return 0
}

# 获取电源功率
get_power_power(){
    local result=""
    if [ $# == 0 ]; then
        result=$(dmidecode -t 39 | grep 'Max Power Capacity:' | grep -v 'NO DIMM\|Not Specified\|Undefined' | awk -F ': ' '{print $2}' | sort | uniq)
    else
        result=$(echo "$1" | grep 'Max Power Capacity:' | grep -v 'NO DIMM\|Not Specified\|Undefined' | awk -F ': ' '{print $2}')
    fi
    if [ -z "$result" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$result"
    return 0
}

# 获取usb规格
get_usb_standard(){
    local result=""
    if [ $# == 0 ]; then
        result=$($TIMEOUTCMD 10 ssh -p 22 localhost lsusb -v | grep 'bcdUSB' | awk 'NR == 1 {print $2}' | sort | uniq)
    else
        result=$(echo "$1" | grep 'bcdUSB' | awk 'NR == 1 {print $2}')
    fi
    if [ -z "$result" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$result"
    return 0
}

# 获取硬盘接口类型
get_hd_iftype(){
    local hd=$1
    local iftype
    if [[ "$hd" =~ ^nvme[0-9]+n[0-9] ]]; then
        iftype="nvme"
    elif $SMARTCTL -i "/dev/$hd" | grep -q 'SATA Version is'; then
        iftype="sata"
    elif $SMARTCTL -i "/dev/$hd" | grep -q 'Transport protocol: *SAS'; then
        iftype="sas"
    fi
    if [ -z "$iftype" ]; then
        iftype=$(lsblk -d -o NAME,TRAN | awk "\$1==\"$hd\"{if (\$2 == \"\") print \"N/A\"; else print \$2;}")
    fi
    escape_parse_cell "$iftype"
}

# 获取硬盘是否可旋转（是HDD还是SSD）
get_hd_rota(){
    local hd=$1
    if use_raidstat -x | grep -q "^  disk_name: $hd\$"; then
        escape_parse_cell "logical"
        return 0
    fi
    escape_parse_cell "$(lsblk -d -o NAME,ROTA | awk "\$1==\"$hd\"{if (\$2 == \"0\") print \"ssd\"; else print \"hdd\";}")"
}

# 获取硬盘或外置存储信息
# 参数为是否外置存储（yes或no）
get_disk_info(){
    local hdinfo=""
    local hds=$(ls_hd)
    for i in $hds; do
        if get_hd_capacity "$i" >/dev/null && get_hd_model "$i" >/dev/null; then
            local tran=$(get_hd_tran "$i")
            if [[ "$1" == "no" && ( -z "$tran" || ( "$tran" != "iscsi" && "$tran" != "fc" ) ) ]]; then
                if [ "$format" == "human" ]; then
                    hdinfo=${hdinfo}"$(get_hd_model "$i") (厂商: $(get_hd_man "$i")) (容量: $(get_hd_capacity "$i"), 固件版本: $(get_hd_fwversion "$i"), 接口类型: $(get_hd_iftype "$i"), 硬盘类型: $(get_hd_rota "$i"))"$'\n'
                else
                    hdinfo=${hdinfo}"$(get_hd_model "$i")${split_symb}$(get_hd_man "$i")${split_symb}$(get_hd_serial "$i")${split_symb}$(get_hd_capacity "$i")${split_symb}$(get_hd_fwversion "$i")${split_symb}$(get_hd_iftype "$i")${split_symb}$(get_hd_rota "$i")"$'\n'
                fi
            elif [[ "$1" == "yes" && ( "$tran" == "iscsi" || "$tran" == "fc" ) ]]; then
                # 外置存储过滤掉vs的虚拟卷
                if get_hd_model "$i" | grep -qi virtual && get_hd_man "$i" | grep -qx SANGFOR; then
                    continue
                fi
                if [ "$format" == "human" ]; then
                    hdinfo=${hdinfo}"$(get_hd_model "$i") (厂商: $(get_hd_man "$i")) (容量: $(get_hd_capacity "$i"), 固件版本: $(get_hd_fwversion "$i"), 外置存储类型: $tran)"$'\n'
                else
                    hdinfo=${hdinfo}"$(get_hd_model "$i")${split_symb}$(get_hd_man "$i")${split_symb}$(get_hd_serial "$i")${split_symb}$(get_hd_capacity "$i")${split_symb}$(get_hd_fwversion "$i")${split_symb}$(escape_parse_cell "$tran")"$'\n'
                fi
            fi
        fi
    done
    if [ "$1" == "no" ]; then
        # 把组成raid逻辑卷的物理硬盘信息获取出来
        hdinfo+=$(get_disk_list@raidstat "txt")
    fi
    if [ -n "$hdinfo" ]; then
        if [ "$format" == "human" ]; then
            if [ "$1" == "yes" ]; then
                echo -n "外置存储数量: "
            else
                echo -n "硬盘数量: "
            fi
            echo "$(echo "$hdinfo" | grep -E '^.+$' | wc -l)"
            get_device_count "$hdinfo" "$(echo "$hdinfo" | sort | uniq)"
        else
            echo "$hdinfo"
        fi
        return 0
    else
        # if [ "$1" == "yes" ]; then
        #     # echo "无外置存储"
        #     echo "无"
        # else
        #     echo "N/A"
        # fi
        echo "无"
        return 1
    fi
}

# 获取所有硬盘信息
get_hd_info(){
    get_disk_info 'no'
}

# 获取所有外置存储
get_tran_info(){
    get_disk_info 'yes'
}

# 获取所有硬盘设备，以空格分隔
# 输出如：sda sdb sdc sdd
ls_hd(){
    #echo $(ls /dev | grep '^sd[a-z]$')
    echo $(lsblk | grep -E $'[\ \t]+disk ' | awk '{print $1}')
    return 0
}

# 获取硬盘型号，参数为硬盘设备
get_hd_model(){
    if [ "$FAILED" == "yes" ]; then
        return 1
    fi
    local sinfo=$(${TIMEOUTCMD} 10 $SMARTCTL -i /dev/$1)
    if [ "$?" -eq "$TIMEOUTCODE" ]; then
        FAILED="yes"
        return 1
    fi
    local model=$(echo "$sinfo" | grep -i 'Device Model:\|Model Number:\|Product:' | awk -F ':[\ ]*' '{print $2}')
    if [ -z "$model" ]; then
        local line=$(lsblk -S | grep -n $1 | awk -F ':' '{print $1}')
        if [ -z "$line" ]; then
            echo "N/A"
            return 1
        fi
        model=$(echo $(lsblk -S --output MODEL | sed -n "${line}p"))
        if [ -z "$model" ]; then
            echo "N/A"
            return 1
        fi
    fi
    escape_parse_cell "$model"
    return 0
}

# 获取硬盘容量，参数为硬盘设备
get_hd_capacity(){
    if [ "$FAILED" == "yes" ]; then
        return 1
    fi
    local sinfo=$(${TIMEOUTCMD} 10 $SMARTCTL -i /dev/$1)
    if [ "$?" -eq "$TIMEOUTCODE" ]; then
        FAILED="yes"
        return 1
    fi
    local tmp=$(echo "$sinfo" | grep -i 'User Capacity:\|Total NVM Capacity:' | sed -r 's/(^.*\[)(.*)(\].*)/\2/g')
    if [ -z "$tmp" ]; then
        tmp=$(lsblk -d | awk "\$1==\"$1\"{print \$4}")
        if [ -z "$tmp" ]; then
            echo "N/A"
            return 1
        fi
    fi
    escape_parse_cell "$tmp"
    return 0
}

get_hd_sector(){
    local tmp=$($TIMEOUTCMD 10 ssh -p 22 localhost /sbin/blockdev --getss /dev/$1)
    if [ -z "$tmp" ]; then
        tmp=$(lsblk -d | awk "\$1==\"$1\"{print \$4}")
        if [ -z "$tmp" ]; then
            echo "N/A"
            return 1
        fi
    fi
    escape_parse_cell "$tmp"
    return 0
}

# 获取硬盘固件版本，参数为硬盘设备
get_hd_fwversion(){
    if [ "$FAILED" == "yes" ]; then
        return 1
    fi
    local sinfo=$(${TIMEOUTCMD} 10 $SMARTCTL -i /dev/$1)
    if [ "$?" -eq "$TIMEOUTCODE" ]; then
        FAILED="yes"
        return 1
    fi
    local tmp=$(echo "$sinfo" | grep -i 'Firmware Version:' | awk -F ':[\ ]*' '{print $2}')
    if [ -z "$tmp" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$tmp"
    return 0
}

# 获取硬盘uuid，参数为硬盘设备
get_hd_uuid(){
    if [ "$FAILED" == "yes" ]; then
        return 1
    fi
    local s_uuid=$(${TIMEOUTCMD} 10 /lib/udev/scsi_id --whitelisted --replace-whitespace --device=/dev/$1)
    if [ "$?" -eq "$TIMEOUTCODE" ]; then
        FAILED="yes"
        return 1
    fi
    escape_parse_cell "$s_uuid"
    return 0
}

# 获取硬盘序列号
get_hd_serial(){
    if [ "$FAILED" == "yes" ]; then
        return 1
    fi
    local sinfo=$(${TIMEOUTCMD} 10 $SMARTCTL -i /dev/$1)
    if [ "$?" -eq "$TIMEOUTCODE" ]; then
        FAILED="yes"
        return 1
    fi
    local tmp=$(echo "$sinfo" | grep -i 'Serial Number:' | awk -F ':[\ ]*' '{print $2}')
    if [ -z "$tmp" ]; then
        local line=$(lsblk -S | grep -n $1 | awk -F ':' '{print $1}')
        if [ -z "$line" ]; then
            echo "N/A"
            return 1
        fi
        tmp=$(echo $(lsblk -S --output SERIAL | sed -n "${line}p"))
        if [ -z "$tmp" ]; then
            echo "N/A"
            return 1
        fi
    fi
    escape_parse_cell "$tmp"
    return 0
}

# 正规方法获取不到生产厂商时，使用这个函数再掰扯一下
dep_get_hd_man.alg(){
    local model=$1

    if [ "$model" == "N/A" ]; then
        return 1
    fi

    # 根据字典，建立型号到厂商的映射
    local model2man=(
            "SSDSCKKB[^\ ]*8R" "INTEL"
            "ST[^\ ]*" "SEAGATE"
            "Micron_[^\ ]*" "Micron"
            "WUS7212[^\ ]*" "Wd"
        )
    for ((i=0; i<${#model2man[*]}; i+=2)); do
        if echo "$model" | grep -x "${model2man[$i]}" >/dev/null; then
            echo "${model2man[$((i + 1))]}"
            return 0
        fi
    done

    # 字典里没有的再根据型号掰扯，有些硬盘的型号最前面可能会是它的生产厂商，如: INTEL SSDPE2KE032T8
    if [ $(echo "$model" | awk '{print NF}') -gt 1 ]; then
        echo "$model" | awk '{print $1}'
        return 0
    fi
}

dep_get_hd_man(){
    local model=$(get_hd_model "$1")
    dep_get_hd_man.alg "$model"
}

get_vendor_by_wwn() {
    local orig_wwn="$1"
    vendor=""
    declare -A VENDOR_MAP=(
        ["3a5a27"]="foresee"  # 直接匹配WWID中的第3字段
    )

    # 按空格分割WWID字段
    read -ra wwn_parts <<< "$orig_wwn"
    # 遍历所有字段查找匹配项
    for part in "${wwn_parts[@]}"; do
        part_lower=$(echo "$part" | tr '[:upper:]' '[:lower:]')
        [[ -n "${VENDOR_MAP[$part_lower]}" ]] && {
            vendor="${VENDOR_MAP[$part_lower]}"
            break
        }
    done

    echo "$vendor"
}

# 获取硬盘生产厂商
get_hd_man(){
    if [ "$FAILED" == "yes" ]; then
        return 1
    fi
    local sinfo=$(${TIMEOUTCMD} 10 $SMARTCTL -i /dev/$1)
    if [ "$?" -eq "$TIMEOUTCODE" ]; then
        FAILED="yes"
        return 1
    fi
    local tmp=$(echo "$sinfo" | grep -i 'Vendor:' | awk -F ':[\ ]*' '{print $2}')
    [[ -n "$tmp" ]] && escape_parse_cell "$tmp" && return 0

    local pos=$(lsblk -S | grep -nw $1 | awk -F ':' '{print $1}')
    if [[ -n "$pos" ]]; then
        tmp=$(lsblk -S --output VENDOR | sed -n "${pos}p" | grep -vx 'ATA[ ]*')
        [[ -n "$tmp" ]] && escape_parse_cell "$tmp" && return 0
    fi

    # 第三优先级：WWN查询
    tmp=$(awk -F ': +' '/LU WWN Device Id:/ {print $2; exit}' <<< "$sinfo")
    if [[ -n "$tmp" ]]; then
        tmp=$(get_vendor_by_wwn "$tmp")
        [[ -n "$tmp" ]] && escape_parse_cell "$tmp" && return 0
    fi

    # 从model里面来确实厂商
    tmp=$(dep_get_hd_man "$1")
    [[ -n "$tmp" ]] && escape_parse_cell "$tmp" && return 0

    # 如果还不行，则按照失败处理
    echo "N/A"
    return 1
}

# 获取外置存储。获取不到则什么都不输出，并返回1
get_hd_tran(){
    # echo 是为了删除尾空
    local tmp=$(echo $(lsblk -S --output TRAN | sed -n "$(lsblk -S | grep -n $1 | awk -F ':' '{print $1}')p"))
    if [ -z "$tmp" ]; then
        return 1
    fi
    escape_parse_cell "$tmp"
    return 0
}

ls_raid_number.HCI(){
    # sas3008型号分析
    local sasinfo=$(lspci | grep -i 'sas3008' | cut -d ' ' -f1)
    if [ -n "$sasinfo" ]; then
        echo $sasinfo
        return 0
    fi

    # adaptec型号分析
    local adaptecinfo=$(lspci | grep -i 'controller: Adaptec Smart Storage' | cut -d ' ' -f1)
    if [ -n "$adaptecinfo" ]; then
        echo $adaptecinfo
        return 0
    fi

    # LSI型号分析
    local lsiinfo=$(lspci | grep -i 'LSI Logic\|Logic MegaRAID\|LSI MegaRAID' | cut -d ' ' -f1)
    if [ -n "$lsiinfo" ]; then
        echo $lsiinfo
        return 0
    fi

    # 上面的都获取不到时
    local info=$(lspci | grep -i 'controller:' | grep -i 'raid')
    if [ -z "$info" ]; then
        # 获取不到有关raid的信息了
        return 1
    fi

    # adaptec型号分析
    adaptecinfo=$(echo "$info" | grep -i 'adaptec' | cut -d ' ' -f1)
    if [ -n "$adaptecinfo" ]; then
        echo $adaptecinfo
        return 0
    fi

    # HP型号分析
    local hpinfo=$(echo "$info" | grep -i 'Hewlett-Packard' | cut -d ' ' -f1)
    if [ -n "$hpinfo" ]; then
        echo $hpinfo
        return 0
    fi

    # 无法解析当前raid卡信息
    return 1
}

ls_raid_number.PXE(){
    echo $(ls -l /sys/block/ | sed -rn 's/.*([0-9a-f]{2}:[0-9a-f]{2}\.[0-9a-f])\/host.*/\1/p' | sort | uniq)
}

# 获取所有RAID卡编号，按空格分隔
ls_raid_number(){
    ls_raid_number.$ostype
}

# 获取所有GPU编号，按空格分隔
ls_gpu_number(){
    # 先筛选出可能的GPU设备，对于19e5设备使用更严格的匹配
    local gpu_numbers=$(lspci -nnv | grep -E '10de:|1e3e:|9999:|19e5:(d802|3000)|1d94:(6211|6210)' | grep -e $'^[^\ \t]' | while read line; do
        local pci_addr=$(echo "$line" | cut -d ' ' -f1)
        local device_info=$(lspci -nnvs "$pci_addr")
        # 对于NVIDIA显卡(10de)，只选择VGA控制器，排除音频控制器
        if echo "$device_info" | grep -q '10de:'; then
            if echo "$device_info" | grep -q 'VGA compatible controller'; then
                echo "$pci_addr"
            fi
        else
            # 对于其他通过筛选的GPU设备(1e3e, 9999, 19e5:d802, 19e5:3000)，直接输出
            echo "$pci_addr"
        fi
    done)
    echo $gpu_numbers
    return 0
}

# 初始化GPU工具和字典
init_gpu_tools(){
    local gpu_numbers=("$@")
    for gpu_num in "${gpu_numbers[@]}"; do
        local gpu_info=$(lspci -nnvs "$gpu_num")
        if echo "$gpu_info" | grep -q '10de:'; then
            GPU_INFO_DICT["$gpu_num,tool"]="nvidia-smi"
        elif echo "$gpu_info" | grep -q '1e3e:'; then
            GPU_INFO_DICT["$gpu_num,tool"]="ixsmi"
        elif echo "$gpu_info" | grep -q '9999:'; then
            GPU_INFO_DICT["$gpu_num,tool"]="mx-smi"
        elif echo "$gpu_info" | grep -q '19e5:'; then
            GPU_INFO_DICT["$gpu_num,tool"]="npu-smi"
        elif echo "$gpu_info" | grep -q '1d94:'; then
            GPU_INFO_DICT["$gpu_num,tool"]="hy-smi"
        fi
        
        # 初始化其他字段为空
        GPU_INFO_DICT["$gpu_num,model"]=""
        GPU_INFO_DICT["$gpu_num,memory_size"]=""
        GPU_INFO_DICT["$gpu_num,driver_version"]=""
        GPU_INFO_DICT["$gpu_num,firmware_version"]=""
        GPU_INFO_DICT["$gpu_num,controller_chip"]=""
        GPU_INFO_DICT["$gpu_num,vendor"]=""
        GPU_INFO_DICT["$gpu_num,product_name"]=""
        GPU_INFO_DICT["$gpu_num,pci_id"]=""
        GPU_INFO_DICT["$gpu_num,subsystem_pid"]=""
        GPU_INFO_DICT["$gpu_num,smi_version"]=""
        GPU_INFO_DICT["$gpu_num,nvml_version"]=""
        GPU_INFO_DICT["$gpu_num,cuda_version"]=""
        GPU_INFO_DICT["$gpu_num,vbios_version"]=""
    done
    
    return 0
}

# 获取所有HBA卡编号，按空格分隔
ls_hba_number(){
    echo $(lspci | grep 'Fibre' | cut -d ' ' -f1)
    return 0
}

# 获取所有网卡编号，按空格分隔
ls_netcard_number(){
    echo $(lspci | grep 'Ethernet' | cut -d ' ' -f1)
    return 0
}

# 从网口编号获取网口名字
# 如参数 0b:00.0 ，输出 eth0
get_ethname_from_number(){
    local num=${1//./\\.}
    local ethname=$(ls -l /sys/class/net | sed -rn "s/(.*${num}\/net\/)(.*)/\2/p")
    if [ -z "$ethname" ]; then
        { sed -n '/^# PCI/,/^$/p' /etc/udev/rules.d/70-persistent-net.rules; echo; } |
            sed -r 's/^# PCI.*\/([^ ]*) .*/\1/g; s/.* NAME="([^"]*)".*/\1/g' |
            sed 'N;N; s/\n/ /g' | awk "\$1~\"$num\"{print \$2}" | head -n 1
    else
        echo "${ethname/\//}"
    fi
}

# 从编号获取HBA卡host数字
# 如参数 3b:00.0 ，输出 host17
get_hbahostid_from_number(){
    local num=${1//./\\.}
    local hbahostid=$(ls -l /sys/class/scsi_host | sed -rn "s/(.*${num}\/host[0-9]+\/scsi_host\/)(.*)/\2/p")
    echo "${hbahostid/\//}"
}

get_netcard_model_from_number(){
    local model=$(lspci -vvvnn -s "$1" | grep 'Product Name:' | head -n 1 | awk -F ': ' '{print $2}')
    if [ -z "$model" ] || [ "$model" == "Example VPD" ]; then
        model=$(get_controller_from_number "$1")
    fi
    escape_parse_cell "$model"
    return 0
}

get_netcard_fwversion_from_number(){
    local ethname=$(get_ethname_from_number ${1})
    if [ -z "$ethname" ]; then
        # echo "无法获取网口名字 编号\"$1\""
        echo "N/A"
        return 1
    fi
    local tool="ethtool"
    if which "realethtool" >/dev/null; then
        tool="realethtool"
    fi
    local fwversion=$($tool -i "$ethname" | grep 'firmware-version:' | awk -F ': ' '{print $2}')
    if [ -z "$fwversion" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$fwversion"
    return 0
}

get_hbacard_fwversion_from_number(){
    local hostid=$(get_hbahostid_from_number $1)
    local fwversion=""
    for file in fwrev fw_version version_fw; do
        fwversion=$(cat /sys/class/scsi_host/${hostid}/$file)
        [ -n "$fwversion" ] && break
    done
    if [ -z "$fwversion" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$fwversion"
    return 0
}

get_hbacard_model_from_number(){
    local hostid=$(get_hbahostid_from_number $1)
    local model=""
    for file in modelname product_version version_product; do
        model=$(cat /sys/class/scsi_host/${hostid}/$file)
        [ -n "$model" ] && break
    done
    if [ -z "$model" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$model"
    return 0
}

# 获取HBA卡的上线状态，是否为可连接
get_hbacard_link_status_from_number(){
    local hostid=$(get_hbahostid_from_number $1)
    local link_status=""
    for file in port_state; do
        link_status=$(cat /sys/class/fc_host/${hostid}/$file)
        [ -n "$link_status" ] && break
    done
    if [ -z "$link_status" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$link_status"
    return 0
}

# 根据编号获取型号（子系统型号），参数为编号
get_model_from_number(){
    # local model=$(lspci -vvvnn -s "$1" | grep 'Subsystem:' | awk -F ': ' '{print $2}' | awk -F ' \[....:....\]' '{print $1}')
    local model=$(lspci -vvvnn -s "$1" | grep 'Subsystem:' | head -n 1 | awk -F ': ' '{print $2}')
    if [ -z "$model" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$model"
    return 0
}

# 根据编号获取控制器型号
get_controller_from_number(){
    # local cont=$(lspci -vvvnn -s "$1" | grep "^$1" | sed -r 's/(^.*\[....\]: )(.*)( (\[....:....\]).*)/\2/g')
    local cont=$(lspci -vvvnn -s "$1" | grep "^$1" | head -n 1 | sed -r 's/(^.*\[....\]: )(.*)/\2/g')
    if [ -z "$cont" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$cont"
    return 0
}

# 根据编号获取PCIe，参数为编号，按空格分隔主PCIe和副PCIe
get_pcie_from_number(){
    local pcieid=($(lspci -vvvnn -s "$1" | grep -E '\[[0-9a-f]{4}:[0-9a-f]{4}\]' -o))
    # if [ "$(echo "$pcieid" | awk '{print NF}')" -ne 2 ]; then
    #     echo "N/A"
    #     return 1
    # fi
    if [ "${#pcieid[@]}" == "0" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "${pcieid[*]}"
    return 0
}

# 获取厂商
get_man_from_number(){
    local man=$(lspci -vmm -s "$1" | grep '^Vendor:' | head -n 1 | awk -F ':[\t\ ]*' '{print $2}')
    if [ -z "$man" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$man"
    return 0
}

# 获取驱动
get_kernel_from_number(){
    local kernel=$(lspci -vvvnn -s "$1" | grep 'Kernel driver' | head -n 1 | awk -F ': ' '{print $2}')
    if [ -z "$kernel" ]; then
        echo "N/A"
        return 1
    fi
    local kernel_version=$(cat "/sys/module/$kernel/version")
    escape_parse_cell "$kernel $kernel_version"
    return 0
}

# 转义输出的;和\
# 把;转义为\;
# 把\转义为\\
escape_parse_cell(){
    if [ "$format" == "json" ] || [ "$split_symb" != ";" ]; then
        # 可以自定义分隔符，我只保证使用 ; 作为分隔符时，字段转义。如果不使用 ; 作为分隔符，出了问题，我不负责
        echo "$1"
        return 0
    fi
    local tmp=${1//;/$'\n'}
    tmp=${tmp//\\/\\\\}
    tmp=${tmp//$'\n'/\\;}
    echo "$tmp"
    return 0
}

# 转义为json字符串，两边不加双引号
jsonstr_no_dq(){
    local str=$1
    str=${str//\\/\\\\}
    str=${str//\"/\\\"}
    str=${str//$'\n'/\\n}
    echo -n "$str"
}

# 转义为json字符串
jsonstr(){
    local str=$1
    # if type jq >/dev/null 2>/dev/null; then
    #     echo -n "$str" | jq -sRM
    #     return 0
    # fi
    echo -n '"'
    jsonstr_no_dq "$str"
    echo '"'
}

# 类似表格的输出方式
# 参数为：
# 标题 内容 折行填充空格数量
output(){
    local old_IFS=$IFS
    IFS=$'\n'
    local pad=""
    echo -n $1
    if [ -z "$2" ]; then
        echo ""
    else
        for i in $2; do
            if [ -z "$pad" ]; then
                if [ "$format" == "human" ]; then
                    for ((j=0; j<$3; j++)); do
                        pad=${pad}' '
                    done
                else
                    pad=$1
                fi
            else
                echo -n $pad
            fi
            echo "$i"
        done
    fi
    IFS=$old_IFS
    if [ "$format" == "human" ]; then
        echo ""
    fi
    return 0
}

# raid工具映射表，键为匹配关系，值为调用的函数前缀
raid_tool_map=(
    "\[1028:....\]"                             "get_raid:perccli64"    # td2024012600085 只要是dell的卡，只用perccli64
    "sas3008"                                   "get_raid:sas3008"
    "\[1000:....\]|\[102a:....\]"               "get_raid:storcli64"
#    "LSI Logic|Logic MegaRAID|LSI MegaRAID"     "get_raid:lsi"         # td2024012600085 不再使用MegaCli64
    "adaptec"                                   "get_raid:adaptec"
    "Hewlett-Packard"                           "get_raid:hp"
)

get_raid:storcli64.model(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/storcli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/storcli64"
    "$RAIDTOOLS_DIR/storcli64" /c$(($idx-1)) show | grep '^Product Name' | awk -F '= ' '{print $2}'
}

get_raid:storcli64@arm.model(){
    get_raid:storcli64.model "$@"
}

get_raid:perccli64.model(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/perccli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/perccli64"
    "$RAIDTOOLS_DIR/perccli64" /c$(($idx-1)) show | grep '^Product Name' | awk -F '= ' '{print $2}'
}

get_raid:perccli64@arm.model(){
    get_raid:storcli64.model "$@"
}

get_raid:sas3008.model(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/sas3ircu" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/sas3ircu"
    "$RAIDTOOLS_DIR/sas3ircu" $(($idx-1)) display | grep 'Controller type' | awk -F ': ' '{print $2}'
}

get_raid:sas3008@arm.model(){
    get_raid:storcli64.model "$@"
}

get_raid:lsi.model(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/MegaCli64" ]; then
        return 1
    fi
    lspci -vvvnn | grep -q '\[1000:005d\]\|\[1000:9361\]' && return 1   # 9361-8i禁用
    lspci -vvvnn | grep -q '\[1000:10e[0-7]\]' && return 1              # H755、H750、H350、HBA350i等禁用
    chmod +x "$RAIDTOOLS_DIR/MegaCli64"
    "$RAIDTOOLS_DIR/MegaCli64" -ShowSummary -aALL -NoLog | grep 'ProductName' | awk -F ': ' '{print $2}' | sed -n "${idx}p" | sed -r 's/\(Bus [0-9]+, Dev [0-9]+\)//g'
}

get_raid:lsi@arm.model(){
    get_raid:storcli64.model "$@"
}

get_raid:adaptec.model(){
    local idx=$1
    local tool="$RAIDTOOLS_DIR/arcconf_pm822"
    if [ ! -f "$tool" ]; then
        tool="$RAIDTOOLS_DIR/arcconf64"
    fi
    if [ ! -f "$tool" ]; then
        return 1
    fi
    chmod +x "$tool"
    "$tool" getconfig $idx al | grep -i 'controller model' | awk -F ': ' '{print $2}'
}

get_raid:adaptec@arm.model(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/arcconf64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/arcconf64"
    "$RAIDTOOLS_DIR/arcconf64" getconfig $idx al | grep -i 'controller model' | awk -F ': ' '{print $2}'
}

get_raid:hp.model(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/hpssacli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/hpssacli64"
    "$RAIDTOOLS_DIR/hpssacli64" ctrl all show config detail | sed '/^[[:blank:]]*$/d' | grep -i 'in slot [0-9]*' | sed 's/in Slot [[:digit:]]* .*$//' | tr -t "\n" " "
}

get_raid:hp@arm.model(){
    echo "N/A"
}

get_raid:storcli64.fw(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/storcli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/storcli64"
    "$RAIDTOOLS_DIR/storcli64" /c$(($idx-1)) show | grep '^FW Version' | awk -F '= ' '{print $2}'
}

get_raid:storcli64@arm.fw(){
    get_raid:storcli64.fw "$@"
}

get_raid:perccli64.fw(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/perccli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/perccli64"
    "$RAIDTOOLS_DIR/perccli64" /c$(($idx-1)) show | grep '^FW Version' | awk -F '= ' '{print $2}'
}

get_raid:perccli64@arm.fw(){
    get_raid:storcli64.fw "$@"
}

get_raid:sas3008.fw(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/sas3ircu" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/sas3ircu"
    "$RAIDTOOLS_DIR/sas3ircu" $(($idx-1)) display | grep 'Firmware version' | awk -F ': ' '{print $2}'
}

get_raid:sas3008@arm.fw(){
    get_raid:storcli64.fw "$@"
}

get_raid:lsi.fw(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/MegaCli64" ]; then
        return 1
    fi
    lspci -vvvnn | grep -q '\[1000:005d\]\|\[1000:9361\]' && return 1   # 9361-8i禁用
    lspci -vvvnn | grep -q '\[1000:10e[0-7]\]' && return 1              # H755、H750、H350、HBA350i等禁用
    chmod +x "$RAIDTOOLS_DIR/MegaCli64"
    "$RAIDTOOLS_DIR/MegaCli64" -AdpAllInfo -aALL -NoLog | grep '^FW Version' | awk -F ': ' '{print $2}' | sed -n "${idx}p"
}

get_raid:lsi@arm.fw(){
    get_raid:storcli64.fw "$@"
}

get_raid:adaptec.fw(){
    local idx=$1
    local tool="$RAIDTOOLS_DIR/arcconf_pm822"
    if [ ! -f "$tool" ]; then
        tool="$RAIDTOOLS_DIR/arcconf64"
    fi
    if [ ! -f "$tool" ]; then
        return 1
    fi
    chmod +x "$tool"
    "$tool" getconfig $idx ad | grep 'Firmware' | awk -F ': ' '{print $2}'
}

get_raid:adaptec@arm.fw(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/arcconf64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/arcconf64"
    "$RAIDTOOLS_DIR/arcconf64" getconfig $idx ad | grep 'Firmware' | awk -F ': ' '{print $2}'
}

get_raid:hp.fw(){
    echo "N/A"
}

get_raid:hp@arm.fw(){
    echo "N/A"
}

get_raid:storcli64.fw_pkg(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/storcli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/storcli64"
    "$RAIDTOOLS_DIR/storcli64" /c$(($idx-1)) show | grep '^FW Package Build' | awk -F '= ' '{print $2}'
}

get_raid:storcli64@arm.fw_pkg(){
    get_raid:storcli64.fw_pkg "$@"
}

get_raid:perccli64.fw_pkg(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/perccli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/perccli64"
    "$RAIDTOOLS_DIR/perccli64" /c$(($idx-1)) show | grep '^FW Package Build' | awk -F '= ' '{print $2}'
}

get_raid:perccli64@arm.fw_pkg(){
    get_raid:storcli64.fw_pkg "$@"
}

get_raid:sas3008.fw_pkg(){
    echo "N/A"
}

get_raid:sas3008@arm.fw_pkg(){
    get_raid:storcli64.fw_pkg "$@"
}

get_raid:lsi.fw_pkg(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/MegaCli64" ]; then
        return 1
    fi
    lspci -vvvnn | grep -q '\[1000:005d\]\|\[1000:9361\]' && return 1   # 9361-8i禁用
    lspci -vvvnn | grep -q '\[1000:10e[0-7]\]' && return 1              # H755、H750、H350、HBA350i等禁用
    chmod +x "$RAIDTOOLS_DIR/MegaCli64"
    "$RAIDTOOLS_DIR/MegaCli64" -ShowSummary -aALL -NoLog | grep 'FW Package Version' | awk -F ': ' '{print $2}' | sed -n "${idx}p"
}

get_raid:lsi@arm.fw_pkg(){
    get_raid:storcli64.fw_pkg "$@"
}

get_raid:adaptec.fw_pkg(){
    echo "N/A"
}

get_raid:adaptec@arm.fw_pkg(){
    echo "N/A"
}

get_raid:hp.fw_pkg(){
    echo "N/A"
}

get_raid:hp@arm.fw_pkg(){
    echo "N/A"
}

get_raid:storcli64.enable_jbod(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/storcli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/storcli64"
    "$RAIDTOOLS_DIR/storcli64" /c$(($idx-1)) show all | grep 'Enable JBOD' | awk -F '= ' '{print $2}' | tr [:upper:] [:lower:]
}

get_raid:storcli64@arm.enable_jbod(){
    get_raid:storcli64.enable_jbod "$@"
}

get_raid:sas3008.enable_jbod(){
    echo "N/A"
}

get_raid:sas3008@arm.enable_jbod(){
    get_raid:storcli64.enable_jbod "$@"
}

get_raid:perccli64.enable_jbod(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/perccli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/perccli64"
    if "$RAIDTOOLS_DIR/perccli64" /c$(($idx-1)) show all | grep 'JBOD LIST' >/dev/null; then
        echo 'yes'
    else
        echo 'no'
    fi
}

get_raid:perccli64@arm.enable_jbod(){
    get_raid:storcli64.enable_jbod "$@"
}

get_raid:lsi.enable_jbod(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/MegaCli64" ]; then
        return 1
    fi
    lspci -vvvnn | grep -q '\[1000:005d\]\|\[1000:9361\]' && return 1   # 9361-8i禁用
    lspci -vvvnn | grep -q '\[1000:10e[0-7]\]' && return 1              # H755、H750、H350、HBA350i等禁用
    chmod +x "$RAIDTOOLS_DIR/MegaCli64"
    "$RAIDTOOLS_DIR/MegaCli64" -AdpAllInfo "-a$(($idx-1))" | grep '^Enable JBOD' | awk -F ': ' '{print $2}' | head -n 1 | tr [:upper:] [:lower:]
}

get_raid:lsi@arm.enable_jbod(){
    get_raid:storcli64.enable_jbod "$@"
}

get_raid:adaptec.enable_jbod(){
    local idx=$1
    local tool="$RAIDTOOLS_DIR/arcconf_pm822"
    if [ ! -f "$tool" ]; then
        tool="$RAIDTOOLS_DIR/arcconf64"
    fi
    if [ ! -f "$tool" ]; then
        return 1
    fi
    chmod +x "$tool"
    "$tool" getconfig $idx ad | grep 'Driver Supports SSD I/O Bypass' | awk -F ': ' '{print $2}' | tr [:upper:] [:lower:]
}

get_raid:adaptec@arm.enable_jbod(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/arcconf64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/arcconf64"
    "$RAIDTOOLS_DIR/arcconf64" getconfig $idx ad | grep 'Driver Supports SSD I/O Bypass' | awk -F ': ' '{print $2}' | tr [:upper:] [:lower:]
}

get_raid:hp.enable_jbod(){
    echo "N/A"
}

get_raid:hp@arm.enable_jbod(){
    echo "N/A"
}

get_raid:storcli64.bios(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/storcli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/storcli64"
    "$RAIDTOOLS_DIR/storcli64" /c$(($idx-1)) show | grep '^BIOS Version' | awk -F '= ' '{print $2}'
}

get_raid:storcli64@arm.bios(){
    get_raid:storcli64.bios "$@"
}

get_raid:sas3008.bios(){
    echo "N/A"
}

get_raid:sas3008@arm.bios(){
    get_raid:storcli64.bios "$@"
}

get_raid:perccli64.bios(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/perccli64" ]; then
        return 1
    fi
    chmod +x "$RAIDTOOLS_DIR/perccli64"
    "$RAIDTOOLS_DIR/perccli64" /c$(($idx-1)) show | grep '^BIOS Version' | awk -F '= ' '{print $2}'
}

get_raid:perccli64@arm.bios(){
    get_raid:storcli64.bios "$@"
}

get_raid:lsi.bios(){
    local idx=$1
    if [ ! -f "$RAIDTOOLS_DIR/MegaCli64" ]; then
        return 1
    fi
    lspci -vvvnn | grep -q '\[1000:005d\]\|\[1000:9361\]' && return 1   # 9361-8i禁用
    lspci -vvvnn | grep -q '\[1000:10e[0-7]\]' && return 1              # H755、H750、H350、HBA350i等禁用
    chmod +x "$RAIDTOOLS_DIR/MegaCli64"
    "$RAIDTOOLS_DIR/MegaCli64" -AdpAllInfo "-a$(($idx-1))" | grep '^BIOS Version' | awk -F ': ' '{print $2}'
}

get_raid:lsi@arm.bios(){
    get_raid:storcli64.bios "$@"
}

get_raid:adaptec.bios(){
    echo "N/A"
}

get_raid:adaptec@arm.bios(){
    echo "N/A"
}

get_raid:hp.bios(){
    echo "N/A"
}

get_raid:hp@arm.bios(){
    echo "N/A"
}

# 获取raid信息
# 参数1 类型，model、fw、enable_jbod、bios、fw_pkg
# 参数2 第几张raid卡，从1开始
get_raid_info(){
    local out=$1
    local idx=$2
    local slot=$(ls_raid_number | awk "{print \$${idx}}")
    local fw_magic=""
    if [ "$FRAMEWORK" == "arm" ]; then
        fw_magic="@arm"
    fi
    if [ -z "$slot" ]; then
        echo "N/A"
        return 1
    fi
    local i
    for ((i=0; i<${#raid_tool_map[@]}; i+=2)); do
        if lspci -vvvnns "${slot}" | grep -Eiw "${raid_tool_map[$i]}" >/dev/null; then
            local model=$(echo "$(${raid_tool_map[$(($i+1))]}${fw_magic}.$out "$idx")" | sed 's/^[ ]*$/N\/A/g')
            if [ "$model" == "N/A" ]; then
                if [ "$out" == "model" ]; then
                    model=$(get_hbacard_model_from_number "$slot")
                elif [ "$out" == "fw" ]; then
                    model=$(get_hbacard_fwversion_from_number "$slot")
                fi
            fi
            escape_parse_cell "$model"
            return 0
        fi
    done
    echo "N/A"
    return 1
}

# 用RAID卡厂商给出的工具查看RAID卡型号，第几个硬件
get_raid_model(){
    local idx=$1
    get_raid_info model $idx
}

get_raid_fw(){
    local idx=$1
    get_raid_info fw $idx
}

get_raid_fw_pkg(){
    local idx=$1
    get_raid_info fw_pkg $idx
}

get_raid_bios(){
    local idx=$1
    get_raid_info bios $idx
}

get_raid_enable_jbod(){
    local idx=$1
    get_raid_info enable_jbod $idx
}

nvidia_tool_cmd(){
    nvidia-smi "$@" ||
    /sf/data/local/sgax/sgax-chroot.sh nvidia-smi "$@" ||
    $TIMEOUTCMD 5 ssh -p 22346 localhost "/sf/data/local/sgax/sgax-chroot.sh nvidia-smi $*" ||
    return 1
    return 0
}

# 显示单个硬件信息，参数为硬件编号 标题（会根据标题进行特殊属性的获取） 第几个硬件
show_info(){
    local idx=$3
    local fwversion
    local extinfo
    local exthinfo
    if [ "${2}" == "RAID卡" ]; then
        fwversion=$(get_raid_fw "$idx")
        if [ -z "$fwversion" ]; then
            fwversion="N/A"
        fi
        # 扩展信息为 RAID型号 是否支持JBOD直通
        extinfo="$(get_raid_model $idx)${split_symb}$(get_raid_enable_jbod $idx)${split_symb}"
        exthinfo="型号: $(get_raid_model $idx), 是否支持JBOD: $(get_raid_enable_jbod $idx)"
    elif [ "${2}" == "GPU" ]; then
        fwversion="$(nvidia_tool_cmd -i $((idx - 1)) --query-gpu=vbios_version --format=csv,noheader || echo 'N/A')"
        # 扩展信息为 产品ID 简称 核心数 支持的guest_os 是否支持vGPU
        extinfo="$(get_pcie_from_number $1 | awk '{if ($1=="") print "N/A"; else print $1;}' | sed -r 's/\[....:(....)\]/\1/g')${split_symb}$(nvidia_tool_cmd -i $((idx - 1)) --query-gpu=name --format=csv,noheader || echo 'N/A')${split_symb}N/A${split_symb}N/A${split_symb}N/A${split_symb}"
        exthinfo=""
    elif [ "${2}" == "HBA卡" ]; then
        fwversion="$(get_hbacard_fwversion_from_number $1)"
        # 扩展信息为 HBA卡型号
        extinfo="$(get_hbacard_model_from_number $1)${split_symb}"
        exthinfo="型号: $(get_hbacard_model_from_number $1)"
    elif [ "${2}" == "网卡" ]; then
        fwversion="$(get_netcard_fwversion_from_number $1)"
        # 扩展信息为 网卡型号
        extinfo="$(get_netcard_model_from_number $1)${split_symb}"
        exthinfo="型号: $(get_netcard_model_from_number $1)"
    fi

    if [ "$format" == "human" ]; then
        echo "$(get_model_from_number $1) ${exthinfo} (PCIe: $(get_pcie_from_number $1), 固件版本: ${fwversion}, 驱动: $(get_kernel_from_number $1), 厂商: $(get_man_from_number $1)) (控制卡: $(get_controller_from_number $1))"
    else
        echo "$(get_controller_from_number $1)${split_symb}$(get_model_from_number $1)${split_symb}${extinfo}$(get_man_from_number $1)${split_symb}${fwversion}${split_symb}$(get_kernel_from_number $1)${split_symb}$(get_pcie_from_number $1 | awk '{if ($2=="") print "N/A"; else print $2;}' | sed -r 's/\[....:(....)\]/\1/g')${split_symb}$(get_pcie_from_number $1 | awk '{if ($2=="") print "N/A"; else print $2;}')"
    fi
    return 0
}

# 显示一组设备信息
# 参数1为标题
# 参数2为所有要显示信息的硬件编号，用空格隔开
show_device(){
    local numbers=($2)
    if [ "${#numbers[*]}" == 0 ]; then
        # echo "无$1"
        echo "无"
    else
        if [ "$format" == "human" ]; then
            if [ "$1" != "RAID卡" ]; then
                echo "${1}数量: ${#numbers[*]}"
            fi
        fi
        local result=""
        for i in ${!numbers[*]}; do
            result=${result}"$(show_info "${numbers[$i]}" "${1}" "$((i + 1))")"$'\n'
        done
        if [ "$1" == "RAID卡" ]; then
            result+=$(get_raid_list@raidstat "txt" "no" "${numbers[@]}")
        fi
        if [ "$format" == "human" ]; then
            if [ "$1" == "RAID卡" ]; then
                echo "${1}数量: $(echo "$result" | wc -l)"
            fi
            get_device_count "$result" "$(echo "${result}" | sort | uniq)"
        else
            echo "$result"
        fi
    fi
    # echo ""
    return 0
}

# 获取BMC版本
get_bmc_version(){
    # local version=$(ipmitool -V | awk -F 'version ' '{print $2}')
    # 加载 ipmitool
    modprobe ipmi_devintf 2>/dev/null && modprobe ipmi_si 2>/dev/null
    if [ $? -ne 0 ]; then
        echo "N/A"
        return 1
    fi
    local version=$(ipmitool mc info | grep 'Firmware Revision' |  awk -F ': ' '{print $2}')
    version=$(echo $version)
    if [ -z "$version" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$version"
    return 0
}

# 获取ipmitool工具版本
get_ipmitool_version(){
    # 加载 ipmitool
    modprobe ipmi_devintf 2>/dev/null && modprobe ipmi_si 2>/dev/null
    if [ $? -ne 0 ]; then
        echo "N/A"
        return 1
    fi
    local version=$(ipmitool -V | awk '{print $3}')
    version=$(echo $version)
    if [ -z "$version" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$version"
    return 0
}

# 检查numa节点数
get_numa_number(){
    local numa_number=""
    if numactl >/dev/null 2>&1; then
        numa_number=$(numactl --hardware | grep 'available' | awk '{print \$2}')
        numa_number=$(echo $numa_number)
        if [ -z "$numa_number" ]; then
            echo "N/A"
            return 1
        fi
    else
        # 只有vt容器里面才有该命令，需要进行尝试性的写法。最后不行再N/A
        if container_exec >/dev/null 2>&1; then
            numa_number=$(container_exec -n asv-c -c "numactl --hardware | grep 'available' | awk '{print \$2}'" | sed 's/\r//g')
            numa_number=$(echo $numa_number)
            if [ -z "$numa_number" ]; then
                echo "N/A"
                return 1
            fi
        else
            echo "N/A"
            return 1
        fi
    fi
    escape_parse_cell "$numa_number"
    return 0
}

# 获取主板插槽情况
get_board_slot(){
    local board_slot=$(lspci -vvv | grep 'LnkCap' | grep -v "ASPM not supported" | awk -F"," '{print $2,$3}' | uniq | tr '\n' ';')
    board_slot=$(echo $board_slot)
    if [ -z "$board_slot" ]; then
        echo "N/A"
        return 1
    fi
    escape_parse_cell "$board_slot"
    return 0
}

# 检查BMC是否支持电源检测
get_bmc_power_monitor(){
    # 加载 ipmitool
    modprobe ipmi_devintf 2>/dev/null && modprobe ipmi_si 2>/dev/null
    if [ $? -ne 0 ]; then
        echo "N/A"
        return 1
    fi
    if ipmitool chassis power status >/dev/null 2>&1; then
        echo "yes"
    else
        echo "no"
    fi
    return 0
}

# 获取主板型号
get_baseboard_model(){
    local name=$(dmidecode -t baseboard | grep 'Product Name:' | awk -F ': ' '{print $2}')
    name=$(echo $name)
    if [ -z "$name" ]; then
        echo "N/A"
        return 1
    fi
    local version=$(dmidecode -t baseboard | grep 'Version:' | awk -F ': ' '{print $2}')
    if [ -z "$version" ]; then
        escape_parse_cell "$name"
        return 0
    fi
    escape_parse_cell "${name} (version: ${version})"
    return 0
}

# csv格式转义
# csv特殊字符有逗号、换行符、双引号
# 其中在两边加双引号可以解决逗号和换行符的问题
# 之后把内容里所有一个双引号替换成两个双引号即可
# 在最前面加上等号，可以强行使用文本模式，防止版本号等变成数字
escape_csv_cell(){
    if [ $(echo "$1" | wc -l) -eq 1 ] && echo "$1" | grep -Ex '[0-9.]+' >/dev/null; then
        echo '="'"${1//\"/\"\"}"'"'
    else
        echo '"'"${1//\"/\"\"}"'"'
    fi
    return 0
}

get_cpu_list.json(){
    echo '['
    local cpu_cnt=$(get_cpu_count)
    local i
    for ((i=0; i<cpu_cnt; i++)); do
        ((i > 0)) && echo ','
cat << EOF
        {
            "model": $(jsonstr "$(get_cpu_model)"),
            "vendor": $(jsonstr "$(get_cpu_man)"),
            "core_count": $(jsonstr "$(get_cpu_core)"),
            "thread_count": $(jsonstr "$(get_cpu_thread_count)"),
            "max_speed": $(jsonstr "$(get_cpu_max_speed)"),
            "microcode": $(jsonstr "$(get_cpu_microcode)"),
            "arch": $(jsonstr "$(get_cpu_frame)"),
            "cpu_clock_speed": $(jsonstr "$(get_cpu_clock_speed)"),
            "l2_cache": $(jsonstr "$(get_cpu_l2_cache)")
EOF
        echo -n '        }'
    done
    echo
    echo -n '    ]'
}

get_power_list.json(){
    echo '['
    local handles=$(dmidecode -t 39 | grep '^Handle [0-9A-Fx]*' -o)
    local old_IFS=$IFS
    IFS=$'\n'
    local handle_array=($handles)
    IFS=$old_IFS
    local is_first=yes
    for i in ${!handle_array[*]}; do
        local power=$(dmidecode -t 39 | sed -n '/^'"${handle_array[$i]}"'/,/^$/p' | sed -n '/^System Power Supply$/,/^$/p')
        if [ -n "$power" ]; then
            if [ -n "$(get_power_vendor "$power")" ]; then
                if [ "$is_first" == "yes" ]; then
                    is_first=no
                else
                    echo ','
                fi
cat << EOF
        {
            "vendor": $(jsonstr "$(get_power_vendor "$power")"),
            "power": $(jsonstr "$(get_power_power "$power")")
EOF
                echo -n '        }'
            fi
        fi
    done
    echo
    echo -n '    ]'
}

get_usb_list.json(){
    echo '['
    local handles=$($TIMEOUTCMD 10 ssh -p 22 localhost lsusb | awk '{print $2 ":" $4}' | sed 's/:$//g')
    local old_IFS=$IFS
    IFS=$'\n'
    local handle_array=($handles)
    IFS=$old_IFS
    local is_first=yes
    for i in ${!handle_array[*]}; do
        local usb=$($TIMEOUTCMD 10 ssh -p 22 localhost lsusb -s ${handle_array[$i]} -v)
        if [ -n "$usb" ]; then
            if [ -n "$(get_usb_standard "$usb")" ]; then
                if [ "$is_first" == "yes" ]; then
                    is_first=no
                else
                    echo ','
                fi
cat << EOF
        {
            "standard": $(jsonstr "$(get_usb_standard "$usb")")
EOF
                echo -n '        }'
            fi
        fi
    done
    echo
    echo -n '    ]'
}

get_memory_list.json(){
    echo '['
    local handles=$(dmidecode -t memory | grep '^Handle [0-9A-Fx]*' -o)
    local old_IFS=$IFS
    IFS=$'\n'
    local handle_array=($handles)
    IFS=$old_IFS
    local is_first=yes
    for i in ${!handle_array[*]}; do
        local mem=$(dmidecode -t memory | sed -n '/^'"${handle_array[$i]}"'/,/^$/p' | sed -n '/^Memory Device$/,/^$/p')
        if [ -n "$mem" ]; then
            if [ -n "$(get_memory_size "$mem")" ]; then
                if [ "$is_first" == "yes" ]; then
                    is_first=no
                else
                    echo ','
                fi
cat << EOF
        {
            "dev_name": $(jsonstr "$(get_memory_slot "$mem")"),
            "vendor": $(jsonstr "$(get_memory_man "$mem")"),
            "model": $(jsonstr "$(get_memory_model "$mem")"),
            "capacity": $(jsonstr "$(get_memory_size "$mem")"),
            "specification": $(jsonstr "$(get_memory_spec "$mem")"),
            "speed": $(jsonstr "$(get_memory_speed "$mem")"),
            "slot": $(jsonstr "$(get_memory_slot "$mem")"),
            "height": $(jsonstr "$(get_memory_height "$mem")"),
            "date": $(jsonstr "$(get_memory_date "$mem")")
EOF
                echo -n '        }'
            fi
        fi
    done
    echo
    echo -n '    ]'
}

use_raidstat(){
    # 生成安全的缓存键，处理特殊字符
    local cache_key
    if [ $# -eq 0 ]; then
        cache_key="no_args"
    else
        # 使用更安全的方式生成缓存键，避免特殊字符问题
        cache_key=$(printf "%s" "$*" | md5sum | cut -d' ' -f1)
    fi
    
    # 计算缓存文件路径（只计算一次）
    local cache_file="$RAIDSTAT_CACHE_DIR/$cache_key"
    
    # 检查缓存中是否已有结果
    if [ -f "$cache_file" ]; then
        cat "$cache_file"
        return 0
    fi
    
    # 缓存中没有，执行实际命令
    local ret=1
    
    # 直接执行命令并将输出重定向到缓存文件
    if [ -x "$CUR_DIR/raidstat/raidstat" ]; then
        "$CUR_DIR/raidstat/raidstat" -z "$@" > "$cache_file"
        ret=$?
    elif [ -x "/sf/bin/raidstat" ]; then
        /sf/bin/raidstat "$@" > "$cache_file"
        ret=$?
    else
        # 创建空文件以避免重复尝试
        touch "$cache_file"
    fi
    
    # 输出缓存内容并返回状态码
    cat "$cache_file"
    return $ret
}

disk_sgs=""

get_disk_sgs()
{
    if [ -z "$disk_sgs" ]; then
        disk_sgs=$(
            \ls /sys/class/scsi_generic/ | while read sg; do
                echo "$sg $($SMARTCTL -i /dev/$sg | grep -i 'Serial Number:' | awk '{print $NF}')"
            done
        )
    fi
    echo "$disk_sgs"
}

get_disk_list@raidstat_ci(){
    local ci=$1
    local callback=$2
    
    # 一次性获取控制器和设备的完整信息
    local controller_full_info
    if ! controller_full_info=$(use_raidstat -kmnrapxk -c "$ci"); then
        return 1
    fi
    
    # 从完整信息中提取逻辑卷信息
    local logical_volume_info
    logical_volume_info=$(echo "$controller_full_info" | grep -E '^controller_driver:|^device_[0-9]+_model:|^device_[0-9]+_serial:|^device_[0-9]+_slot:|^device_[0-9]+_vendor:|^  disk_name:|^  [0-9]*_[0-9]*$|^  [0-9]*:[A-Za-z]*:[A-Za-z]*$')
    
    # 检查是否为特殊RAID卡
    local is_sg_raid="no"
    if echo "$controller_full_info" | grep -qFx 'controller_model: SAS3008'; then
        is_sg_raid="yes"    # 专门适配SAS3008这款卡
    elif echo "$controller_full_info" | grep -qF 'controller_driver: ps3stor '; then
        is_sg_raid="yes"
    fi
    local ctrl_driver=$(echo "$logical_volume_info" | grep '^controller_driver:' | awk -F ': ' '{print $2}')
    local ctrl_driver_name=$(echo "$ctrl_driver" | awk '{print $1}')
    local ctrl_driver_version=$(echo "$ctrl_driver" | awk '{print $2}')
    # 如果逻辑卷盘符获取不到，就用当前logical_volume_info里这个逻辑卷出现的行号代替
    local logical_disk_names=$(echo "$logical_volume_info" | grep -n '^  disk_name:' | grep -v ': md' | awk -F ':  disk_name:[ ]*' '{ if ($2=="") {print $1} else {print $2} }')
    local disk
    for disk in $logical_disk_names; do
        local real_disk disk_did disk_dids real_disks
        if echo "$disk" | grep -q '[^0-9]'; then
            real_disks=$(echo "$logical_volume_info" | sed -rn "/^  disk_name: $disk\$/,/^  disk_name:|^[^ ]/p")
        else
            real_disks=$(echo "$logical_volume_info" | sed -rn "$disk,/^  disk_name:|^[^ ]/p")
            disk=""
        fi
        real_disks=$(echo "$real_disks" | sed '1d' | sed 's/^  //g')
        if echo "$real_disks" | sed -n '$p' | grep -q 'disk_name'; then
            real_disks=$(echo "$real_disks" | sed '$d')
        fi
        if (($(echo "$real_disks" | wc -l) % 2 != 0)); then
            # 这个时候的real_disks是real_disks和disk_dids的合体，肯定是偶数，不是偶数就是异常了
            continue
        fi
        disk_dids=$(echo "$real_disks" | tail -n $(($(echo "$real_disks" | wc -l) / 2)))
        real_disks=$(echo "$real_disks" | head -n $(($(echo "$real_disks" | wc -l) / 2)))
        real_disks=($real_disks)
        disk_dids=($disk_dids)
        local real_disk_idx
        for real_disk_idx in "${!real_disks[@]}"; do
            real_disk=${real_disks[real_disk_idx]}
            disk_did=${disk_dids[real_disk_idx]}
            local model vendor serial_number capacity firmware_version interface_type disk_type d_arg smt_arg
            model=$(echo "$logical_volume_info" | grep -E -A1 -B2 "^device_[0-9]+_slot: $real_disk\$" | grep -E 'device_[0-9]+_model:' | awk -F ': ' '{print $2}')
            vendor=$(echo "$logical_volume_info" | grep -E -A1 -B2 "^device_[0-9]+_slot: $real_disk\$" | grep -E 'device_[0-9]+_vendor:' | awk -F ': ' '{print $2}')
            if [ -z "$vendor" ] || [ "$vendor" == "ATA" ] || [[ "$vendor" =~ ^ATA\  ]]; then
                vendor=$(dep_get_hd_man.alg "$model")
            fi
            serial_number=$(echo "$logical_volume_info" | grep -E -A1 -B2 "^device_[0-9]+_slot: $real_disk\$" | grep -E 'device_[0-9]+_serial:' | awk -F ': ' '{print $2}')
            interface_type=$(echo "$disk_did" | awk -F ':' '{print $2}' | sed -r 's/(.*)/\L\1/g')
            disk_type=$(echo "$disk_did" | awk -F ':' '{print $3}' | sed -r 's/(.*)/\L\1/g')
            local d_arg_map=(
                "megaraid_sas"  "megaraid"
                "mpt3sas"       "megaraid"
                "smartpqi"      "cciss"
            )
            d_arg=""
            smt_arg=""
            local i
            for ((i=0; i<${#d_arg_map[@]}; i+=2)); do
                if [ "$ctrl_driver_name" == "${d_arg_map[i]}" ]; then
                    d_arg=${d_arg_map[i+1]}
                    break
                fi
            done
            if [ -n "$d_arg" ]; then
                local N
                if [ "$d_arg" == "cciss" ]; then
                    local serial_map=$(for i in {0..50}; do echo "$i $(smartctl -i "/dev/$disk" -d "cciss,$i" | grep -i 'Serial Number:' | awk '{print $NF}')"; done)
                    N=$(echo "$serial_map" | grep -Fw "$serial_number" | awk '{print $1}')
                else
                    N=$(echo "$disk_did" | awk -F ':' '{print $1}')
                fi
                if [ -n "$N" ]; then
                    d_arg+=",$N"
                fi
                if [ -n "$disk" ]; then
                    capacity=$($SMARTCTL -i "/dev/$disk" -d "$d_arg" | grep -i 'User Capacity:\|Total NVM Capacity:' | sed -r 's/(^.*\[)(.*)(\].*)/\2/g')
                    firmware_version=$($SMARTCTL -i "/dev/$disk" -d "$d_arg" | grep -i 'Firmware Version:' | awk -F ':[\ ]*' '{print $2}')
                fi
                smt_arg="/dev/$disk -d $d_arg"
            fi
            if [ "$is_sg_raid" == "yes" ]; then
                # SAS3008这款卡拿型号和固件的方法有点特别，下面请看我的表演
                local sg=$(get_disk_sgs | grep -Fw "$serial_number" | awk '{print $1}')
                # smartctl，但是sg（对没错，就是用smartctl来读，但是要传/dev/sgX而不是盘符）
                model=$($SMARTCTL -i /dev/$sg | grep 'Device Model\|Model Number' | awk -F ': *' '{print $2}')
                capacity=$($SMARTCTL -i /dev/$sg | grep -i 'User Capacity:\|Total NVM Capacity:' | sed -r 's/(^.*\[)(.*)(\].*)/\2/g')
                firmware_version=$($SMARTCTL -i /dev/$sg | grep 'Firmware Version\|Revision' | awk -F ': *' '{print $2}')
                [ -z "$interface_type" ] && interface_type=$(get_hd_iftype "$sg")
                [ -z "$disk_type" ] && disk_type=$(lsblk -Sno ROTA /dev/$disk | head -n 1 | awk '{if ($1 == "0") print "ssd"; else print "hdd";}')
                d_arg=""
                smt_arg="/dev/$sg"
            fi
            "$callback" "$disk" "$model" "$vendor" "$serial_number" "$capacity" "$firmware_version" "$interface_type" "$disk_type" "$d_arg" "$smt_arg"
        done
    done
}

get_disk_list@raidstat_ci.json:output(){
    local disk=$1
    local model=$2
    local vendor=$3
    local serial_number=$4
    local capacity=$5
    local firmware_version=$6
    local interface_type=$7
    local disk_type=$8
    local d_arg=$9
    local smt_arg=${10}

    echo ','
cat << EOF
        {
            "dev_name": $(jsonstr "@$disk"),
            "model": $(jsonstr "$model"),
            "vendor": $(jsonstr "$vendor"),
            "serial_number": $(jsonstr "$serial_number"),
            "capacity": $(jsonstr "$capacity"),
            "firmware_version": $(jsonstr "$firmware_version"),
            "interface_type": $(jsonstr "$interface_type"),
            "type": $(jsonstr "$disk_type"),
            "d_arg": $(jsonstr "$d_arg"),
            "smt_arg": $(jsonstr "$smt_arg"),
            "smart": $(jsonstr "$($SMARTCTL -A $smt_arg | awk '$1==1||$1==5||$1==7||$1==9||$1==10||$1==11||$1==161||$1==167||$1==187||$1==196||$1==197||$1==198||$1==199||$1==233')")
EOF
    echo -n '        }'
}

get_disk_list@raidstat_ci.json(){
    local ci=$1
    get_disk_list@raidstat_ci "$ci" "get_disk_list@raidstat_ci.json:output"
}

get_disk_list@raidstat_ci.txt:output(){
    local disk=$1
    local model=$2
    local vendor=$3
    local serial_number=$4
    local capacity=$5
    local firmware_version=$6
    local interface_type=$7
    local disk_type=$8
    local d_arg=$9

    if [ "$format" == "human" ]; then
        echo "${model} (厂商: ${vendor}) (容量: ${capacity}, 固件版本: ${firmware_version}, 接口类型: ${interface_type}, 硬盘类型: ${disk_type})"
    else
        echo "${model}${split_symb}${vendor}${split_symb}${serial_number}${split_symb}${capacity}${split_symb}${firmware_version}${split_symb}${interface_type}${split_symb}${disk_type}"
    fi
}

get_disk_list@raidstat_ci.txt(){
    local ci=$1
    get_disk_list@raidstat_ci "$ci" "get_disk_list@raidstat_ci.txt:output"
}

get_disk_list@raidstat(){
    local list_format=$1
    local ctrler_num
    if ! ctrler_num=$(use_raidstat | grep -Ec '^controller[0-9]+$'); then
        return 1
    fi
    local ci
    for ((ci = 0; ci < ctrler_num; ci++)); do
        get_disk_list@raidstat_ci.$list_format "$ci"
    done
}

get_disk_list.json(){
    echo '['
    local hds=$(ls_hd)
    local is_first=yes
    local i
    local cur_info_file
    local each_info_file=()
    for i in $hds; do
        cur_info_file=$(mktemp)
        each_info_file+=("$cur_info_file")
        (
        if get_hd_capacity "$i" >/dev/null && get_hd_model "$i" >/dev/null; then
            local tran=$(get_hd_tran "$i")
            if [[ -z "$tran" || ( "$tran" != "iscsi" && "$tran" != "fc" ) ]]; then
                # if [ "$is_first" == "yes" ]; then
                #     is_first=no
                # else
                #     echo ','
                # fi
cat << EOF
        {
            "dev_name": $(jsonstr "$i"),
            "model": $(jsonstr "$(get_hd_model "$i")"),
            "vendor": $(jsonstr "$(get_hd_man "$i")"),
            "serial_number": $(jsonstr "$(get_hd_serial "$i")"),
            "capacity": $(jsonstr "$(get_hd_capacity "$i")"),
            "sector": $(jsonstr "$(get_hd_sector "$i")"),
            "firmware_version": $(jsonstr "$(get_hd_fwversion "$i")"),
            "uuid": $(jsonstr "$(get_hd_uuid "$i")"),
            "interface_type": $(jsonstr "$(get_hd_iftype "$i")"),
            "type": $(jsonstr "$(get_hd_rota "$i")"),
            "smart": $(jsonstr "$($SMARTCTL -A "/dev/$i" | awk '$1==1||$1==5||$1==7||$1==9||$1==10||$1==11||$1==161||$1==167||$1==187||$1==196||$1==197||$1==198||$1==199||$1==233')")
EOF
                echo -n '        }'
            fi
        fi
        ) >"$cur_info_file"
    done
    for i in "${each_info_file[@]}"; do
        if [ ! -s "$i" ]; then
            continue
        fi
        if [ "$is_first" == "yes" ]; then
            is_first=no
        else
            echo ','
        fi
        cat "$i"
    done
    rm -f "${each_info_file[@]}"
    get_disk_list@raidstat "json"
    echo
    echo -n '    ]'
}

get_raid_list@raidstat_ci(){
    local ci=$1
    local callback=$2
    local is_first=$3

    local raid_info
    raid_info=$(use_raidstat -slfkjv -c "$ci")

    local pci_address
    local model
    local enable_jbod
    local fw
    local fw_pkg
    local bios_version
    local driver
    local vendor

    pci_address=$(echo "$raid_info" | awk -F ': ' '$1=="controller_pci_slot"{print $2}')
    model=$(echo "$raid_info" | awk -F ': ' '$1=="controller_model"{print $2}')
    enable_jbod=$(echo "$raid_info" | awk -F ': ' '$1=="controller_sup_jbod"{print $2}')
    fw=$(echo "$raid_info" | awk -F ': ' '$1=="controller_fw_version"{print $2}')
    fw_pkg=$(echo "$raid_info" | awk -F ': ' '$1=="controller_fw_package_build"{print $2}')
    bios_version="N/A"
    driver=$(echo "$raid_info" | awk -F ': ' '$1=="controller_driver"{print $2}')
    vendor=$(echo "$raid_info" | awk -F ': ' '$1=="controller_vendor"{print $2}')

    "$callback" "$is_first" "$pci_address" "$model" "$enable_jbod" "$fw" "$fw_pkg" "$bios_version" "$driver" "$vendor"
}

get_raid_list@raidstat_ci.txt:output(){
    local is_first=$1

    local pci_address=$2
    local model=$3
    local enable_jbod=$4
    local fw=$5
    local fw_pkg=$6
    local bios_version=$7
    local driver=$8
    local vendor=$9

    if [ "$format" == "human" ]; then
        echo "$(get_model_from_number $pci_address) 型号: $model, 是否支持JBOD: $enable_jbod (PCIe: $(get_pcie_from_number $pci_address), 固件版本: ${fw}, 驱动: ${driver}, 厂商: ${vendor}) (控制卡: $(get_controller_from_number $pci_address))"
    else
        echo "$(get_controller_from_number $pci_address)${split_symb}$(get_model_from_number $pci_address)${split_symb}${model}${split_symb}${enable_jbod}${split_symb}${vendor}${split_symb}${fw}${split_symb}${driver}${split_symb}$(get_pcie_from_number $pci_address | awk '{if ($2=="") print "N/A"; else print $2;}' | sed -r 's/\[....:(....)\]/\1/g')${split_symb}$(get_pcie_from_number $pci_address | awk '{if ($2=="") print "N/A"; else print $2;}')"
    fi
}

get_raid_list@raidstat_ci.txt(){
    local ci=$1
    local is_first=$2
    get_raid_list@raidstat_ci "$ci" "get_raid_list@raidstat_ci.txt:output" "$is_first"
}

get_raid_list@raidstat_ci.json:output(){
    local is_first=$1

    local pci_address=$2
    local model=$3
    local enable_jbod=$4
    local fw=$5
    local fw_pkg=$6
    local bios_version=$7
    local driver=$8
    local vendor=$9

    if [ "$is_first" == "no" ]; then
        echo ','
    fi
cat << EOF
        {
            "dev_name": $(jsonstr "$pci_address"),
            "model": $(jsonstr "$model"),
            "controller_chip_model": $(jsonstr "$(get_controller_from_number "$pci_address")"),
            "card_model": $(jsonstr "$(get_model_from_number "$pci_address")"),
            "is_jbod": $(jsonstr "$enable_jbod"),
            "firmware_version": $(jsonstr "$fw"),
            "fw_pkg_version": $(jsonstr "$fw_pkg"),
            "bios_version": $(jsonstr "$bios_version"),
            "driver_version": $(jsonstr "$driver"),
            "pcieid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($1=="") print "N/A"; else print $1;}')"),
            "subsystem_pid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($2=="") print "N/A"; else print $2;}')"),
            "vendor": $(jsonstr "$vendor")
EOF
    echo -n '        }'
}

get_raid_list@raidstat_ci.json(){
    local ci=$1
    local is_first=$2
    get_raid_list@raidstat_ci "$ci" "get_raid_list@raidstat_ci.json:output" "$is_first"
}

get_raid_list@raidstat(){
    local list_format=$1
    local is_first=$2
    shift
    shift
    local skip_pci_address_list=("$@")

    local ctrler_num
    ctrler_num=$(use_raidstat -l | grep '^controller_pci_slot: ' | awk -F ': ' '{print $2}' | cat -n | awk '{$1=$1-1; print $1,$2}')

    local address_i
    for address_i in "${skip_pci_address_list[@]}"; do
        address_i=$(lspci -Ds "$address_i" | awk '{print $1}' | head -n 1 | sed 's/\./\\./g')
        ctrler_num=$(echo "$ctrler_num" | grep -Ev "^[0-9]+ ${address_i}\$")
    done

    ctrler_num=$(echo "$ctrler_num" | awk '{print $1}')

    local ci
    for ci in $ctrler_num; do
        get_raid_list@raidstat_ci.$list_format "$ci" "$is_first"
        is_first=no
    done
}

get_raid_list.json(){
    echo '['
    local pci_address_list=($(ls_raid_number))
    local is_first=yes
    local idx
    local pci_address
    for idx in "${!pci_address_list[@]}"; do
        if [ "$is_first" == "yes" ]; then
            is_first=no
        else
            echo ','
        fi
        pci_address=${pci_address_list[$idx]}
cat << EOF
        {
            "dev_name": $(jsonstr "$pci_address"),
            "model": $(jsonstr "$(get_raid_model "$(($idx + 1))")"),
            "controller_chip_model": $(jsonstr "$(get_controller_from_number "$pci_address")"),
            "card_model": $(jsonstr "$(get_model_from_number "$pci_address")"),
            "is_jbod": $(jsonstr "$(get_raid_enable_jbod "$(($idx + 1))")"),
            "firmware_version": $(jsonstr "$(get_raid_fw "$(($idx + 1))")"),
            "fw_pkg_version": $(jsonstr "$(get_raid_fw_pkg "$(($idx + 1))")"),
            "bios_version": $(jsonstr "$(get_raid_bios "$(($idx + 1))")"),
            "driver_version": $(jsonstr "$(get_kernel_from_number "$pci_address")"),
            "pcieid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($1=="") print "N/A"; else print $1;}')"),
            "subsystem_pid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($2=="") print "N/A"; else print $2;}')"),
            "vendor": $(jsonstr "$(get_man_from_number "$pci_address")")
EOF
        echo -n '        }'
    done
    get_raid_list@raidstat "json" "$is_first" "${pci_address_list[@]}"
    echo
    echo -n '    ]'
}

# 初始化GPU信息字典，获取所有GPU的详细信息
# 参数: GPU编号列表
init_gpu_info_dict(){
    local pci_address_list=("$@")
    local idx
    local pci_address
    
    # 遍历每个GPU设备
    for idx in "${!pci_address_list[@]}"; do
        pci_address=${pci_address_list[$idx]}
        local tool=${GPU_INFO_DICT["$pci_address,tool"]}
        
        # 根据不同的GPU工具获取信息
        case "$tool" in
            "nvidia-smi")
                # 获取NVIDIA GPU信息
                GPU_INFO_DICT["$pci_address,memory_size"]=$(nvidia-smi -i $idx --query-gpu=memory.total --format=csv,noheader,nounits 2>/dev/null || echo "N/A")
                GPU_INFO_DICT["$pci_address,firmware_version"]=$(nvidia-smi -i $idx --query-gpu=vbios_version --format=csv,noheader,nounits 2>/dev/null || echo "N/A")
                GPU_INFO_DICT["$pci_address,abbreviation"]=$(nvidia-smi -i $idx --query-gpu=name --format=csv,noheader,nounits 2>/dev/null || echo "N/A")
                GPU_INFO_DICT["$pci_address,smi_version"]=$(nvidia-smi --version 2>/dev/null | head -n 1 | awk '{print $NF}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,nvml_version"]=$(nvidia-smi --version 2>/dev/null | grep "NVML version" | awk '{print $NF}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,cuda_version"]=$(nvidia-smi --version 2>/dev/null | grep "CUDA Version" | awk '{print $NF}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,vbios_version"]=$(nvidia-smi -i $idx --query-gpu=vbios_version --format=csv,noheader,nounits 2>/dev/null || echo "N/A")
                ;;
            "ixsmi")
                # 获取Ixsmi GPU信息 - 根据正确的命令格式修改
                # 获取基本信息（型号、驱动版本、cuda版本、显存）
                local basic_info=$(ixsmi --query-gpu=gpu_name,driver_version,cuda_version,memory.total --format=csv,noheader 2>/dev/null | head -n $(($idx+1)) | tail -n 1)
                GPU_INFO_DICT["$pci_address,abbreviation"]=$(echo "$basic_info" | awk -F',' '{print $1}' | tr -d '"')
                GPU_INFO_DICT["$pci_address,driver_version"]=$(echo "$basic_info" | awk -F',' '{print $2}' | tr -d '"')
                GPU_INFO_DICT["$pci_address,cuda_version"]=$(echo "$basic_info" | awk -F',' '{print $3}' | tr -d '"')
                GPU_INFO_DICT["$pci_address,memory_size"]=$(echo "$basic_info" | awk -F',' '{print $4}' | tr -d '"')
                GPU_INFO_DICT["$pci_address,firmware_version"]=$(ixsmi -q -d hardware_monitor 2>/dev/null | grep "Firmware" | head -n 1 | awk '{print $NF}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,smi_version"]=""
                GPU_INFO_DICT["$pci_address,nvml_version"]=""
                GPU_INFO_DICT["$pci_address,vbios_version"]=""
                ;;
            "mx-smi")
                # 获取Mellanox GPU信息
                # 获取版本信息（maca, bios, driver, firmware version）
                local version_info=$(mx-smi --show-version 2>/dev/null)
                GPU_INFO_DICT["$pci_address,smi_version"]=$(echo "$version_info" | grep -i "driver" | head -n 1 | awk '{print $NF}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,firmware_version"]=$(echo "$version_info" | grep -i "firmware" | head -n 1 | awk '{print $NF}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,vbios_version"]=$(echo "$version_info" | grep -i "bios" | head -n 1 | awk '{print $NF}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,driver_version"]=$(echo "$version_info" | grep -i "driver" | head -n 1 | awk '{print $NF}' || echo "N/A")
                # 获取显存信息
                local memory_info=$(mx-smi --show-memory 2>/dev/null)
                GPU_INFO_DICT["$pci_address,memory_size"]=$(echo "$memory_info" | grep -i "memory\|显存" | head -n 1 | grep -o '[0-9]\+[[:space:]]*[GM]B\?' || echo "N/A")                
                # 其他信息设为N/A或尝试从版本信息中提取
                GPU_INFO_DICT["$pci_address,abbreviation"]=$(echo "$version_info" | grep -i "maca\|gpu" | head -n 1 | awk '{print $NF}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,cuda_version"]=""  # mx-smi 不支持CUDA版本
                GPU_INFO_DICT["$pci_address,nvml_version"]=""  # mx-smi 不支持NVML版本
                ;;
            "hy-smi")
                # 获取hy-smi GPU信息
                # 获取vbios版本（CSV格式，跳过表头，取第一行的第三列）
                GPU_INFO_DICT["$pci_address,vbios_version"]=$(hy-smi -v --csv 2>/dev/null | tail -n +2 | head -n 1 | awk -F',' '{print $3}' | tr -d '"' || echo "N/A")
                # 获取驱动版本（从Driver Version:后面提取）
                GPU_INFO_DICT["$pci_address,driver_version"]=$(hy-smi --showdriverversion 2>/dev/null | awk -F ': ' '/Driver Version/{print $2}' || echo "N/A")
                # 获取固件版本（CSV格式，跳过表头，取第一行的第8列 SMC Firmware Version）
                GPU_INFO_DICT["$pci_address,firmware_version"]=$(hy-smi --showfwinfo --csv 2>/dev/null | tail -n +2 | head -n 1 | awk -F',' '{print $8}' | tr -d '"' || echo "N/A")
                # 获取显卡型号（CSV格式，跳过表头，取第一行的第三列Card Series）
                GPU_INFO_DICT["$pci_address,abbreviation"]=$(hy-smi --showproductname --csv 2>/dev/null | tail -n +2 | head -n 1 | awk -F',' '{print $3}' | tr -d '"' || echo "N/A")
                # 获取smi版本（从版本行提取版本号，格式：Version 1.10.1 (Mar  4 2025 ...)）
                GPU_INFO_DICT["$pci_address,smi_version"]=$(hy-smi --version 2>/dev/null | head -n 1 | awk '{print $2}' || echo "N/A")
                # 获取显存大小（只取当前GPU的显存大小）
                GPU_INFO_DICT["$pci_address,memory_size"]=$(hy-smi --showmeminfo vram 2>/dev/null | grep "vram Total Memory" | awk '{print $NF}' | head -n 1 || echo "N/A")
                # hy-smi不支持的字段设置为空
                GPU_INFO_DICT["$pci_address,nvml_version"]=""
                GPU_INFO_DICT["$pci_address,cuda_version"]=""
                ;;
            "npu-smi")
                # 获取NPU GPU信息
                # 从board信息获取版本和固件信息
                local board_info=$(npu-smi info -i 0 -t board 2>/dev/null)
                GPU_INFO_DICT["$pci_address,smi_version"]=$(echo "$board_info" | grep "Software Version" | awk -F': ' '{print $2}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,firmware_version"]=$(echo "$board_info" | grep "Firmware Version" | awk -F': ' '{print $2}' || echo "N/A")
                
                # 从model信息获取芯片名称
                local model_info=$(npu-smi info -m 2>/dev/null)
                GPU_INFO_DICT["$pci_address,abbreviation"]=$(echo "$model_info" | grep "^[[:space:]]*$idx[[:space:]]\+0[[:space:]]" | awk '{print $NF}' || echo "N/A")
                
                # 从memory信息获取显存大小
                GPU_INFO_DICT["$pci_address,memory_size"]=$(npu-smi info -i 0 -t memory 2>/dev/null | grep "HBM Capacity" | awk -F': ' '{print $2}')
                
                # npu-smi不支持的字段设置为空
                GPU_INFO_DICT["$pci_address,vbios_version"]=""
                GPU_INFO_DICT["$pci_address,driver_version"]=$(echo "$board_info" | grep "Software Version" | awk -F': ' '{print $2}' || echo "N/A")
                GPU_INFO_DICT["$pci_address,cuda_version"]=""
                GPU_INFO_DICT["$pci_address,nvml_version"]=""
                ;;
            *)
                # 默认情况，使用通用方法获取信息
                GPU_INFO_DICT["$pci_address,memory_size"]=""
                GPU_INFO_DICT["$pci_address,abbreviation"]=""
                GPU_INFO_DICT["$pci_address,firmware_version"]=""
                GPU_INFO_DICT["$pci_address,smi_version"]=""
                GPU_INFO_DICT["$pci_address,nvml_version"]=""
                GPU_INFO_DICT["$pci_address,cuda_version"]=""
                GPU_INFO_DICT["$pci_address,vbios_version"]=""
                ;;
        esac
    done
}

# 从GPU信息字典中获取指定字段的值
get_gpu_info_from_dict(){
    local pci_address=$1
    local field=$2
    echo "${GPU_INFO_DICT["$pci_address,$field"]}"
}

get_gpu_list.json(){
    echo '['
    local pci_address_list=($(ls_gpu_number))
    local is_first=yes
    local idx
    local pci_address
    # 初始化GPU工具和字典
    init_gpu_tools "${pci_address_list[@]}"
    init_gpu_info_dict "${pci_address_list[@]}"
    
    # 遍历每个GPU设备并输出JSON格式信息
    for idx in "${!pci_address_list[@]}"; do
        if [ "$is_first" == "yes" ]; then
            is_first=no
        else
            echo ','
        fi
        pci_address=${pci_address_list[$idx]}
cat << EOF
        {
            "model": $(jsonstr "$(get_model_from_number "$pci_address")"),
            "controller_chip_model": $(jsonstr "$(get_controller_from_number "$pci_address")"),
            "firmware_version": $(jsonstr "$(get_gpu_info_from_dict "$pci_address" "firmware_version")"),
            "abbreviation": $(jsonstr "$(get_gpu_info_from_dict "$pci_address" "abbreviation")"),
            "driver_version": $(jsonstr "$(get_kernel_from_number "$pci_address")"),
            "pid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($1=="") print "N/A"; else print $1;}' | sed -r 's/\[....:(....)\]/\1/g')"),
            "pcieid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($2=="") print "N/A"; else print $2;}' | sed -r 's/\[....:(....)\]/\1/g')"),
            "pci_id": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($1=="") print "N/A"; else print $1;}')"),
            "subsystem_pid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($2=="") print "N/A"; else print $2;}')"),
            "vendor": $(jsonstr "$(get_man_from_number "$pci_address")"),
            "gpu_memory_size": $(jsonstr "$(get_gpu_info_from_dict "$pci_address" "memory_size")"),
            "smi_version": $(jsonstr "$(get_gpu_info_from_dict "$pci_address" "smi_version")"),
            "nvml_version": $(jsonstr "$(get_gpu_info_from_dict "$pci_address" "nvml_version")"),
            "cuda_version": $(jsonstr "$(get_gpu_info_from_dict "$pci_address" "cuda_version")"),
            "vbios_version": $(jsonstr "$(get_gpu_info_from_dict "$pci_address" "vbios_version")")
EOF
        echo -n '        }'
    done
    echo
    echo -n '    ]'
}

get_hba_list.json(){
    echo '['
    local pci_address_list=($(ls_hba_number))
    local is_first=yes
    local idx
    local pci_address
    for idx in "${!pci_address_list[@]}"; do
        if [ "$is_first" == "yes" ]; then
            is_first=no
        else
            echo ','
        fi
        pci_address=${pci_address_list[$idx]}
cat << EOF
        {
            "dev_name": $(jsonstr "$pci_address"),
            "model": $(jsonstr "$(get_hbacard_model_from_number "$pci_address")"),
            "link_status": $(jsonstr "$(get_hbacard_link_status_from_number "$pci_address")"),
            "bus_controller": $(jsonstr "$(get_controller_from_number "$pci_address")"),
            "subsystem": $(jsonstr "$(get_model_from_number "$pci_address")"),
            "firmware_version": $(jsonstr "$(get_hbacard_fwversion_from_number "$pci_address")"),
            "driver_version": $(jsonstr "$(get_kernel_from_number "$pci_address")"),
            "pcieid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($1=="") print "N/A"; else print $1;}')"),
            "subsystem_pid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($2=="") print "N/A"; else print $2;}')"),
            "vendor": $(jsonstr "$(get_man_from_number "$pci_address")")
EOF
        echo -n '        }'
    done
    echo
    echo -n '    ]'
}

get_optical_module_list.json(){
    echo '[]'
#     local netcard_pci_address=$1
#     local eth=$(get_ethname_from_number "$netcard_pci_address")
#     local tool="ethtool"
#     if which "realethtool" >/dev/null; then
#         tool="realethtool"
#     fi
#     echo '['
#     if [ -n "$eth" ] && "$tool" -m "$eth" >/dev/null 2>&1 ; then
#         local o_info=$("$tool" -m "$eth")
# cat << EOF
#                 {
#                     "vendor_name": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Vendor name[\ \t]' | awk -F ': ' '{print $2}')"),
#                     "vendor_sn": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Vendor SN[\ \t]' | awk -F ': ' '{print $2}')"),
#                     "vendor_pn": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Vendor PN[\ \t]' | awk -F ': ' '{print $2}')"),
#                     "vendor_oui": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Vendor OUI[\ \t]' | awk -F ': ' '{print $2}')"),
#                     "vendor_rev": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Vendor rev[\ \t]' | awk -F ': ' '{print $2}')"),
#                     "transceiver_type": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Transceiver type[\ \t]*:' | awk -F 'Transceiver type[\ \t]*: ' '{print $2}')"),
#                     "optical_diagnostics_support": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Optical diagnostics support[\ \t]' | awk -F ': ' '{print $2}')"),
#                     "module_temperature": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Module temperature[\ \t]*:' | awk -F ': ' '{print $2}')"),
#                     "module_voltage": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Module voltage[\ \t]*:' | awk -F ': ' '{print $2}')"),
#                     "laser_bias_current": $(jsonstr "$(echo "$o_info" | grep $'[\ \t]Laser bias current[\ \t]*:' | awk -F ': ' '{print $2}')")
#                 }
# EOF
#     fi
#     echo -n '            ]'
}

get_network_list.json(){
    echo '['
    local pci_address_list=($(ls_netcard_number))
    local is_first=yes
    local idx
    local pci_address
    local dev_name
    local max_speed
    local cur_info_file
    local each_info_file=()
    for idx in "${!pci_address_list[@]}"; do
        cur_info_file=$(mktemp)
        each_info_file+=("$cur_info_file")
        (
        # if [ "$is_first" == "yes" ]; then
        #     is_first=no
        # else
        #     echo ','
        # fi
        pci_address=${pci_address_list[$idx]}
        dev_name=$(get_ethname_from_number "$pci_address")
        max_speed=$([ -n "$dev_name" ] && ethtool "$dev_name" | sed -rn '/Supported link modes:/,/^\t[^ \t]/p' | grep base | grep -Eo '[^ ]+base[^ ]+' | sort -rn | head -n 1 | grep -Eo '[0-9]+')
        link_status=$([ -n "$dev_name" ] && ethtool "$dev_name" | grep "Link detected" | awk -F":" '{print $2}' | tr -d " ")
cat << EOF
        {
            "dev_name": $(jsonstr "$dev_name"),
            "model": $(jsonstr "$(get_netcard_model_from_number "$pci_address")"),
            "controller_chip_model": $(jsonstr "$(get_controller_from_number "$pci_address")"),
            "subsystem": $(jsonstr "$(get_model_from_number "$pci_address")"),
            "firmware_version": $(jsonstr "$(get_netcard_fwversion_from_number "$pci_address")"),
            "driver_version": $(jsonstr "$(get_kernel_from_number "$pci_address")"),
            "pcieid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($1=="") print "N/A"; else print $1;}')"),
            "subsystem_pid": $(jsonstr "$(get_pcie_from_number "$pci_address" | awk '{if ($2=="") print "N/A"; else print $2;}')"),
            "vendor": $(jsonstr "$(get_man_from_number "$pci_address")"),
            "interface_type": $(jsonstr "$(get_net_interface_type "$dev_name")"),
            "link_status": $(jsonstr "$link_status"),
            "max_speed": $(jsonstr "$max_speed"),
            "optical_module": $(get_optical_module_list.json "$pci_address")
EOF
        echo -n '        }'
        ) >"$cur_info_file"
    done
    for i in "${each_info_file[@]}"; do
        if [ ! -s "$i" ]; then
            continue
        fi
        if [ "$is_first" == "yes" ]; then
            is_first=no
        else
            echo ','
        fi
        cat "$i"
    done
    rm -f "${each_info_file[@]}"
    echo
    echo -n '    ]'
}

get_ex_storage_list.json(){
    echo '['
    local hds=$(ls_hd)
    local is_first=yes
    local i
    local cur_info_file
    local each_info_file=()
    for i in $hds; do
        cur_info_file=$(mktemp)
        each_info_file+=("$cur_info_file")
        (
        if get_hd_capacity "$i" >/dev/null && get_hd_model "$i" >/dev/null; then
            local tran=$(get_hd_tran "$i")
            if [[ "$tran" == "iscsi" || "$tran" == "fc" ]]; then
                # if [ "$is_first" == "yes" ]; then
                #     is_first=no
                # else
                #     echo ','
                # fi
cat << EOF
        {
            "dev_name": $(jsonstr "$i"),
            "model": $(jsonstr "$(get_hd_model "$i")"),
            "vendor": $(jsonstr "$(get_hd_man "$i")"),
            "serial_number": $(jsonstr "$(get_hd_serial "$i")"),
            "capacity": $(jsonstr "$(get_hd_capacity "$i")"),
            "uuid": $(jsonstr "$(get_hd_uuid "$i")"),
            "firmware_version": $(jsonstr "$(get_hd_fwversion "$i")"),
            "es_type": $(jsonstr "$tran")
EOF
                echo -n '        }'
            fi
        fi
        ) >"$cur_info_file"
    done
    for i in "${each_info_file[@]}"; do
        if [ ! -s "$i" ]; then
            continue
        fi
        if [ "$is_first" == "yes" ]; then
            is_first=no
        else
            echo ','
        fi
        cat "$i"
    done
    rm -f "${each_info_file[@]}"
    echo
    echo -n '    ]'
}

# 单独获取server硬件类型
get_server_list.json(){
cat << EOF
[
        {
            "hci_version": $(jsonstr "$(cat /sf/version | head -n 1)"),
            "mainboard_model": $(jsonstr "$(get_baseboard_model)"),
            "platform_type": $(jsonstr "$(get_hci_type)"),
            "vendor": $(jsonstr "$(get_server_man)"),
            "model": $(jsonstr "$(get_server_model)"),
            "bios_version": $(jsonstr "$(get_bios_version)"),
            "bmc_version": $(jsonstr "$(get_bmc_version)"),
            "ipmitool_version": $(jsonstr "$(get_ipmitool_version)"),
            "bmc_power_monitor": $(jsonstr "$(get_bmc_power_monitor)"),
            "numa_number": $(jsonstr "$(get_numa_number)"),
            "board_slot": $(jsonstr "$(get_board_slot)"),
            "system_manager": $(jsonstr "$(get_system_manage)"),
            "product_name": $(jsonstr "$(get_baseboard_model)"),
            "serial_number": $(jsonstr "$(get_serial_number)"),
            "env": $(jsonstr "$csv_server_name")
        }
    ]
EOF
}

main(){
    local padcount=22

    if [[ "$format" == "human" || "$format" == "parse" ]]; then
        if [ -n "$csv_server_name" ]; then
            echo "$csv_server_name"
        fi
        output "系统信息:             " "$(uname -a)" padcount
        if [ "$format" == "human" ]; then
            output "版本:                 " "$(cat /sf/version)" padcount
        else
            output "版本:                 " "$(cat /sf/version | head -n 1)" padcount
        fi
        output "平台类型:             " "$(get_hci_type)" padcount
        output "BMC版本:              " "$(get_bmc_version)" padcount
        output "主板型号:             " "$(get_baseboard_model)" padcount
        output "服务器厂商:           " "$(get_server_man)" padcount
        output "服务器系统管理:       " "$(get_system_manage)" padcount
        output "服务器型号:           " "$(get_server_model)" padcount
        output "BIOS版本:             " "$(get_bios_version)" padcount
        output "产品名称:             " "$(get_baseboard_model)" padcount
        if [ "$format" == "human" ]; then
            output "CPU型号:              " "$(get_cpu_model)" padcount
            output "CPU厂商:              " "$(get_cpu_man)" padcount
            output "CPU架构:              " "$(get_cpu_frame)" padcount
            if get_cpu_microcode >/dev/null; then
                output "CPU微码:              " "$(get_cpu_microcode)" padcount
            fi
            output "CPU核数:              " "$(get_cpu_core)" padcount
            if get_cpu_thread >/dev/null; then
                output "CPU线程数:            " "$(get_cpu_thread)" padcount
            fi
            output "CPU个数:              " "$(get_cpu_count)" padcount
        else
            for ((cpui=0; cpui<$(get_cpu_count); cpui++)); do
                output "CPU:                  " "$(get_cpu_model)${split_symb}$(get_cpu_man)${split_symb}$(get_cpu_core)${split_symb}$(get_cpu_microcode)" padcount
            done
        fi
        output "内存:                 " "$(get_memory_info)" padcount
        output "硬盘:                 " "$(get_hd_info)" padcount

        output "RAID卡:               " "$(show_device "RAID卡" "$(ls_raid_number)")" padcount
        output "GPU:                  " "$(show_device "GPU" "$(ls_gpu_number)")" padcount
        output "HBA卡:                " "$(show_device "HBA卡" "$(ls_hba_number)")" padcount
        output "网卡:                 " "$(show_device "网卡" "$(ls_netcard_number)")" padcount
        output "外置存储:             " "$(get_tran_info)" padcount
    elif [ "$format" == "csv" ]; then
        format="human"
        if [ "$csv_output_bar" == "yes" ]; then
            if [ -n "$csv_server_name" ]; then
                echo -n ","
            fi
            print_csv_bar
        fi
        if [ -n "$csv_server_name" ]; then
            echo -n "$(escape_csv_cell "$csv_server_name"),"
        fi
        echo -n "$(escape_csv_cell "$(cat /sf/version)"),;"
        echo -n "$(escape_csv_cell "$(get_hci_type)"),"
        echo -n "$(escape_csv_cell "$(get_bmc_version)"),;"
        echo -n "$(escape_csv_cell "$(get_baseboard_model)"),;"
        echo -n "$(escape_csv_cell "$(get_server_man)"),;"
        echo -n "$(escape_csv_cell "$(get_system_manage)"),;"
        echo -n "$(escape_csv_cell "$(get_server_model)"),;"
        echo -n "$(escape_csv_cell "$(get_bios_version)"),;"
        # echo -n "$(escape_csv_cell "$(get_cpu_model)"),"
        # echo -n "$(escape_csv_cell "$(get_cpu_frame)"),"
        # echo -n "$(escape_csv_cell "$(get_cpu_core)"),"
        # echo -n "$(escape_csv_cell "$(get_cpu_count)"),"
        # echo -n "$(escape_csv_cell "$(get_cpu_microcode)"),"
        echo -n "$(escape_csv_cell "型号: $(get_cpu_model), 架构: $(get_cpu_frame), 核数: $(get_cpu_core), 微码: $(get_cpu_microcode) [$(get_cpu_count)个]"),;"
        echo -n "$(escape_csv_cell "$(get_memory_info)"),;"
        echo -n "$(escape_csv_cell "$(get_hd_info)"),;"
        echo -n "$(escape_csv_cell "$(show_device "RAID卡" "$(ls_raid_number)")"),;"
        echo -n "$(escape_csv_cell "$(show_device "GPU" "$(ls_gpu_number)")"),;"
        echo -n "$(escape_csv_cell "$(show_device "HBA卡" "$(ls_hba_number)")"),;"
        echo -n "$(escape_csv_cell "$(show_device "网卡" "$(ls_netcard_number)")"),;"
        echo "$(escape_csv_cell "$(get_tran_info)")"
        format="csv"
    elif [ "$format" == "json" ]; then
        local cpu_info_file=$(mktemp)
        local memory_info_file=$(mktemp)
        local disk_info_file=$(mktemp)
        local raid_info_file=$(mktemp)
        local gpu_info_file=$(mktemp)
        local hba_info_file=$(mktemp)
        local network_info_file=$(mktemp)
        local ex_storage_info_file=$(mktemp)
        local power_info_file=$(mktemp)
        local usb_info_file=$(mktemp)
        local server_info_file=$(mktemp)

        local cpu_info_pid=""
        local memory_info_pid=""
        local disk_info_pid=""
        local raid_info_pid=""
        local gpu_info_pid=""
        local hba_info_pid=""
        local network_info_pid=""
        local ex_storage_info_pid=""
        local power_info_pid=""
        local usb_info_pid=""
        local server_info_pid=""

        local ht
        for ht in $(set | sed -rn 's/^OUTPUT_([A-Z_]+)_INFO=yes$/\1/p'); do
            if ! echo "${ht,,}" | grep -qE 'cpu|memory|disk|raid|gpu|hba|network|ex_storage|power|usb|server'; then
                continue
            fi
            eval "get_${ht,,}_list.json >\"\$${ht,,}_info_file\" &"
            eval "${ht,,}_info_pid=$!"
        done

        wait $cpu_info_pid $memory_info_pid $disk_info_pid $raid_info_pid $gpu_info_pid $hba_info_pid $network_info_pid $ex_storage_info_pid $power_info_pid $usb_info_pid $server_info_pid

        # 获取符合条件的变量名列表
        ht_list=($(set | sed -rn 's/^OUTPUT_([A-Z_]+)_INFO=yes$/\1/p'))
        # 获取列表长度
        ht_count=${#ht_list[@]}
        echo "{"
        # 使用数组下标循环
        for ((i=0; i<ht_count; i++)); do
            ht=${ht_list[i]}
            # 检查是否符合条件
            if ! echo "${ht,,}" | grep -qE 'cpu|memory|disk|raid|gpu|hba|network|ex_storage|power|usb|server'; then
                continue
            fi
            echo -n "    \"${ht,,}\": $(eval "cat \"\$${ht,,}_info_file\"")"
            # 判断是否是最后一个元素
            if ((i != ht_count - 1)); then
                echo ","
            else
                echo 
            fi
        done
        echo "}"

        rm -f "$cpu_info_file" "$memory_info_file" "$disk_info_file" "$raid_info_file" "$gpu_info_file" "$hba_info_file" "$network_info_file" "$ex_storage_info_file" "$power_info_file" "$usb_info_file" "$server_info_file"
    else
        echo "无效的格式\"${format}\""
        return 1
    fi

    return 0
}

ng_get_cpu_cache_size(){
    local save_dir=$1
    local base_dir="/sys/devices/system/cpu"
    local cpuid
    for cpuid in $(find $base_dir -regex "$base_dir/cpu[0-9]+"); do
        local index
        for index in $(find $cpuid -regex "$cpuid/cache/index[0-9]+"); do
            ng_run_cmd "cat $index/size" "$save_dir"
        done
    done
}

ng_turbostat(){
    local save_dir=$1
    local tool=turbostat
    type sfturbostat >/dev/null 2>&1 && tool=sfturbostat
    ${TIMEOUTCMD} 8 $tool >"$save_dir/turbostat.txt"
    ${TIMEOUTCMD} 8 cgexec -g cpu:/ $tool >"$save_dir/cgexec_-g_cpu:+_turbostat.txt"
}

type4smartctl_map=(
    "sas3008"                               "sat"
    "Adaptec"                               "cciss"
    "LSI Logic|Logic MegaRAID|LSI MegaRAID" "megaraid"
)

# 获取smartctl -d 需要传入的raid类型参数
# 参数为盘符，如sda
# 输出为需要传给smartctl -d 的参数，如megaraid,0
# 如果硬盘不在raid上，返回值非0，并且输出为空
ng_get_raid_type4smartctl(){
    local disk=$1
    local raid_slot=$(ls -l "/sys/block/$disk" | sed -rn 's/.*(..:..\..)\/host[0-9].*/\1/p')
    if [ -z "$raid_slot" ]; then
        return 1
    fi

    local d_type="megaraid"
    local disk_idx=0

    # 对应关系
    local i
    for ((i=0; i<${#type4smartctl_map[@]}; i+=2)); do
        if lspci -s "$raid_slot" | grep -Eiq "${type4smartctl_map[$i]}"; then
            d_type=${type4smartctl_map[$((i+1))]}
            break
        fi
    done

    echo "$d_type,$disk_idx"

    return 0
}

ng_get_disk_smartctl(){
    local save_dir=$1
    local disks=$(lsblk -S -p | tail -n +2 | awk '{print $1}')
    local disk
    for disk in $disks; do
        local d_type=""
        local sctl_type=$(ng_get_raid_type4smartctl "$(basename "$disk")")
        if [ -n "$sctl_type" ]; then
            d_type=" -d $sctl_type"
        fi
        if [ -n "$d_type" ]; then
            ng_run_cmd "smartctl -i$d_type $disk" "$save_dir"
            ng_run_cmd "smartctl -H$d_type $disk" "$save_dir"
            ng_run_cmd "smartctl -A$d_type $disk" "$save_dir"
        fi
        ng_run_cmd "smartctl -i $disk" "$save_dir"
        ng_run_cmd "smartctl -H $disk" "$save_dir"
        ng_run_cmd "smartctl -A $disk" "$save_dir"
        ng_run_cmd "/sbin/blockdev --getss $disk" "$save_dir"
        ng_run_cmd "/lib/udev/scsi_id --whitelisted --page=0x80 --replace-whitespace --device=$disk" "$save_dir"
        ng_run_cmd "udevadm info --query=all --name=$disk" "$save_dir"
        ng_run_cmd "udevadm info --query=all --name=$disk |grep ID_SERIAL=" "$save_dir"
        ng_run_cmd "cat /sys/block/$(basename "$disk")/queue/discard_granularity" "$save_dir"
        if lsblk -S $disk -o ROTA | tail -n +2 | grep -q 1; then
            if [ -n "$d_type" ]; then
                ng_run_cmd "smartctl -l$d_type error $disk" "$save_dir"
                ng_run_cmd "smartctl -l$d_type xerror $disk" "$save_dir"
            fi
            ng_run_cmd "smartctl -l error $disk" "$save_dir"
            ng_run_cmd "smartctl -l xerror $disk" "$save_dir"
        fi
        smartctl --scan | sed 's/#.*$//g' | while read d_cmd; do
            ng_run_cmd "smartctl -i $d_cmd" "$save_dir"
            ng_run_cmd "smartctl -A $d_cmd" "$save_dir"
        done
        ng_run_cmd "/usr/bin/sg_verify --lba=0 --count=10007 $disk -r" "$save_dir"
        ng_run_cmd "/sf/bin/smartctl -i $disk" "$save_dir"
    done
}

ng_get_block_info(){
    local save_dir=$1
    local md
    for md in $(\ls -d /sys/block/md*); do
        ng_run_cmd "cat $md/md/level" "$save_dir"
    done
    local device
    for device in $(find /sys/bus/scsi/devices/ -regex '/sys/bus/scsi/devices/.+:.+:.+:.+'); do
        ng_run_cmd "cat $device/state" "$save_dir"
        local disk_name
        disk_name=$(\ls $device/block/) &&
        ng_run_cmd "cat $device/block/$disk_name/size" "$save_dir"
    done
}

ng_get_raid_controller(){
    local save_dir=$1
    local ctrler_num
    local i
    for i in $(lspci | awk '{print $1}'); do
        ng_run_cmd "lspci -v -s $i" "$save_dir"
    done
    if [ -x "$RAIDTOOLS_DIR/perccli64" ]; then
        ctrler_num=$($RAIDTOOLS_DIR/perccli64 show ctrlcount | grep '^Controller Count' | awk -F '= ' '{print $2}')
        ng_run_cmd "$RAIDTOOLS_DIR/perccli64 show all" "$save_dir"
        ng_run_cmd "$RAIDTOOLS_DIR/perccli64 show" "$save_dir"
        for ((i=0; i<ctrler_num; i++)); do
            ng_run_cmd "$RAIDTOOLS_DIR/perccli64 /c${i} show all" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/perccli64 /c${i} /bbu show all" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/perccli64 /c${i}/eall/sall show" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/perccli64 /c${i}/eall/sall show all" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/perccli64 /c${i}/vall show all" "$save_dir"

            local disk_info=$("$RAIDTOOLS_DIR/perccli64" /c"$i"/eall/sall show | sed -n '/^Drive Information/,$p' | sed -n '/^----/,/^$/p' | sed '1d;2d;3d;$d' | sed '$d')
            local disk_cnt=$(echo "$disk_info" | wc -l)
            [ -z "$disk_info" ] && disk_cnt=0
            local j
            for ((j=0; j<disk_cnt; j++)); do
                local EID
                local Slt
                EID=$(echo "$disk_info" | sed -nr "$((j+1)) s/^([^:]*):.*/\1/p")
                Slt=$(echo "$disk_info" | sed -nr "$((j+1)) s/^([^:]*):([^ ]*) .*/\2/p")
                ng_run_cmd "$RAIDTOOLS_DIR/perccli64 /c${i}/e${EID}/s${Slt} show" "$save_dir"
                ng_run_cmd "$RAIDTOOLS_DIR/perccli64 /c${i}/e${EID}/s${Slt} show all" "$save_dir"
            done
        done
    fi
    if [ -x "$RAIDTOOLS_DIR/storcli64" ]; then
        ctrler_num=$($RAIDTOOLS_DIR/storcli64 show ctrlcount | grep '^Controller Count' | awk -F '= ' '{print $2}')
        ng_run_cmd "$RAIDTOOLS_DIR/storcli64 show all" "$save_dir"
        ng_run_cmd "$RAIDTOOLS_DIR/storcli64 show" "$save_dir"
        for ((i=0; i<ctrler_num; i++)); do
            ng_run_cmd "$RAIDTOOLS_DIR/storcli64 /c${i} show all" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/storcli64 /c${i} /bbu show all" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/storcli64 /c${i}/eall/sall show" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/storcli64 /c${i}/eall/sall show all" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/storcli64 /c${i}/vall show all" "$save_dir"

            local disk_info=$("$RAIDTOOLS_DIR/storcli64" /c"$i"/eall/sall show | sed -n '/^Drive Information/,$p' | sed -n '/^----/,/^$/p' | sed '1d;2d;3d;$d' | sed '$d')
            local disk_cnt=$(echo "$disk_info" | wc -l)
            [ -z "$disk_info" ] && disk_cnt=0
            local j
            for ((j=0; j<disk_cnt; j++)); do
                local EID
                local Slt
                EID=$(echo "$disk_info" | sed -nr "$((j+1)) s/^([^:]*):.*/\1/p")
                Slt=$(echo "$disk_info" | sed -nr "$((j+1)) s/^([^:]*):([^ ]*) .*/\2/p")
                ng_run_cmd "$RAIDTOOLS_DIR/storcli64 /c${i}/e${EID}/s${Slt} show" "$save_dir"
                ng_run_cmd "$RAIDTOOLS_DIR/storcli64 /c${i}/e${EID}/s${Slt} show all" "$save_dir"
            done
        done
    fi
    # if [ -x "$RAIDTOOLS_DIR/MegaCli64" ]; then
    #     if ! lspci -vvvnn | grep -q '\[1000:005d\]\|\[1000:9361\]\|\[1000:10e[0-7]\]'; then
    #         ctrler_num=$($RAIDTOOLS_DIR/MegaCli64 -AdpCount | grep '^Controller Count' | awk -F ': ' '{print $2}' | tr -d '.')
    #         ng_run_cmd "$RAIDTOOLS_DIR/MegaCli64 -AdpAllInfo -aALL" "$save_dir"
    #         for ((i=0; i<ctrler_num; i++)); do
    #             ng_run_cmd "$RAIDTOOLS_DIR/MegaCli64 -AdpAllInfo -a${i}" "$save_dir"
    #             ng_run_cmd "$RAIDTOOLS_DIR/MegaCli64 -AdpBbuCmd -GetBbuStatus -a${i}" "$save_dir"
    #             ng_run_cmd "$RAIDTOOLS_DIR/MegaCli64 -PDList -a${i}" "$save_dir"
    #             ng_run_cmd "$RAIDTOOLS_DIR/MegaCli64 -LDInfo -Lall -a${i}" "$save_dir"
    #         done
    #     fi
    # fi
    if [ -x "$RAIDTOOLS_DIR/arcconf64" ]; then
        ctrler_num=$($RAIDTOOLS_DIR/arcconf64 list | grep -i '^Controllers found' | awk -F ': ' '{print $2}' | head -n 1)
        ng_run_cmd "$RAIDTOOLS_DIR/arcconf64 list" "$save_dir"
        for ((i=1; i<=ctrler_num; i++)); do
            ng_run_cmd "$RAIDTOOLS_DIR/arcconf64 getconfig $i al" "$save_dir"
        done
    fi
    if [ -x "$RAIDTOOLS_DIR/arcconf_pm822" ]; then
        ctrler_num=$($RAIDTOOLS_DIR/arcconf_pm822 list | grep -i '^Controllers found' | awk -F ': ' '{print $2}' | head -n 1)
        ng_run_cmd "$RAIDTOOLS_DIR/arcconf_pm822 list" "$save_dir"
        for ((i=1; i<=ctrler_num; i++)); do
            ng_run_cmd "$RAIDTOOLS_DIR/arcconf_pm822 list $i" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/arcconf_pm822 getconfig $i LD" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/arcconf_pm822 getconfig $i pd" "$save_dir"
            ng_run_cmd "$RAIDTOOLS_DIR/arcconf_pm822 getconfig $i ad" "$save_dir"
        done
    fi
    if [ -x "$RAIDTOOLS_DIR/ssacli" ]; then
        local old_PATH=$PATH
        PATH="${PATH}:${RAIDTOOLS_DIR}"
        ctrler_num=$(ssacli ctrl all show | grep -ioE 'Slot [^ ]+' | awk '{print $2}')
        ng_run_cmd "ssacli ctrl all show" "$save_dir"
        for i in $ctrler_num; do
            ng_run_cmd "ssacli ctrl slot=$i show detail" "$save_dir"
        done
        PATH=$old_PATH
    fi
    if [ -x "$RAIDTOOLS_DIR/sas2ircu" ]; then
        ctrler_num=$($RAIDTOOLS_DIR/sas2ircu list | grep -Ec '^[\ ]+[0-9]')
        ng_run_cmd "$RAIDTOOLS_DIR/sas2ircu list" "$save_dir"
        for ((i=0; i<ctrler_num; i++)); do
            ng_run_cmd "$RAIDTOOLS_DIR/sas2ircu $i display" "$save_dir"
        done
    fi
    if [ -x "$RAIDTOOLS_DIR/sas3ircu" ]; then
        ctrler_num=$($RAIDTOOLS_DIR/sas3ircu list | grep -Ec '^[\ ]+[0-9]')
        ng_run_cmd "$RAIDTOOLS_DIR/sas3ircu list" "$save_dir"
        for ((i=0; i<ctrler_num; i++)); do
            ng_run_cmd "$RAIDTOOLS_DIR/sas3ircu $i display" "$save_dir"
        done
    fi
}

ng_get_nic_ethtool(){
    local save_dir=$1
    local nics=$(ifconfig | grep -Eo '^[^\ ]+')
    for nic in $nics; do
        ng_run_cmd "ethtool $nic" "$save_dir"
        ng_run_cmd "ethtool -i $nic" "$save_dir"
    done
}

ng_hw_list=(
    "cpu"
    "memory"
    "disk"
    "raid"
    "nic"
    "host"
)

ng_info_normal_cpu=(
    "uname -m"
    "dmidecode -t processor"
    "cat /proc/stat"
    "cat /proc/cpuinfo"
    "sensors"
    "ipmitool -v sdr list"
    "dmidecode -t system"
)

ng_info_normal_memory=(
    "dmidecode -t memory"
    "cat /proc/meminfo"
)

ng_info_normal_disk=(
    "dmidecode -t chassis"
    "lsblk -S -p"
    "lsblk -S -P"
    "ls -l /dev/disk/by-id/"
    "ls -l /sys/bus/scsi/devices/"
    "lspci"
    "smartctl --scan"
    "ls -l /sys/block/"
    "cat /proc/diskstats"
)

ng_info_normal_raid=(
    "dmidecode -t chassis"
    "lspci"
    "lspci -v"
    "ls -l /sys/class/scsi_generic/"
)

ng_info_normal_nic=(
    "ifconfig"
)

ng_info_normal_host=(
    "df /boot"
    "cat /sf/etc/host_down_history"
    "/bin/mount"
    "hostname"
    "df"
)

ng_info_special_cpu=(
    "ng_get_cpu_cache_size"
    "ng_turbostat"
)

ng_info_special_disk=(
    "ng_get_disk_smartctl"
    "ng_get_block_info"
)

ng_info_special_raid=(
    "ng_get_raid_controller"
)

ng_info_special_nic=(
    "ng_get_nic_ethtool"
)

ng_cmd2filename(){
    local cmd=$1
    local endwith=".txt"
    echo "$cmd" | grep -q '^cat ' && endwith=""
    echo "$cmd" | sed "s/${RAIDTOOLS_DIR//\//\\/}/\/sf\/bin\/raidtools\/bin/g" | sed "s/^cat //g; s/\//+/g; s/ /_/g; s/\$/$endwith/g";
}

# 执行命令后自动保存到对应文件
# 参数1为命令，参数2为存储路径
ng_run_cmd(){
    local cmd=$1
    local dir=$2

    eval "$cmd" >"$dir/$(ng_cmd2filename "$cmd")"
}

output_ng_info.normal(){
    local info_dir=$1
    local dir
    for dir in "${ng_hw_list[@]}"; do
        # mkdir -p "$info_dir/$dir"
        eval "
        if (set -u; : \${ng_info_normal_${dir}}) 2>/dev/null; then
            for cmd in \"\${ng_info_normal_${dir}[@]}\"; do
                # ng_run_cmd \"\$cmd\" '$info_dir/$dir'
                ng_run_cmd \"\$cmd\" '$info_dir'
            done
        fi
        "
    done
}

output_ng_info.special(){
    local info_dir=$1
    local dir
    for dir in "${ng_hw_list[@]}"; do
        # mkdir -p "$info_dir/$dir"
        eval "
        if (set -u; : \${ng_info_special_${dir}}) 2>/dev/null; then
            for callback in \"\${ng_info_special_${dir}[@]}\"; do
                # \$callback '$info_dir/$dir'
                \$callback '$info_dir'
            done
        fi
        "
    done
}

output_ng_info(){
    local info_dir=$(mktemp -d)

    output_ng_info.normal "$info_dir"
    output_ng_info.special "$info_dir"

    cd "$info_dir"
    
    if [ "$extend_ng_filename" == "-" ]; then
        tar -zc *
    else
        tar -zcf "$extend_ng_filename" --exclude="$extend_ng_filename" *
    fi

    cd - >/dev/null

    if [ "$extend_ng_filename" != "-" ]; then
        cp "$info_dir/$extend_ng_filename" .
    fi

    rm -rf "$info_dir"
}

print_csv_bar(){
    echo "版本,服务器架构,BMC版本,主板型号,服务器厂商,系统管理,服务器型号,BIOS版本,CPU,内存,硬盘,RAID,GPU,HBA,网卡,外置存储"
    return 0
}

usage(){
cat << EOF
使用说明:
功能: 获取硬件信息
参数说明:
  --help | -h | -?                 本使用说明

  --format 格式 | -f 格式          设置输出格式, 支持的格式有"parse", "human", "csv", "json"
                                   parse为便于其他脚本提取信息的格式, human为看着方便的格式, csv为逗号分隔符表格的格式
                                   默认为parse

  --split-symb 分隔符 | -s 分隔符  设置分隔符, 默认为";", 当使用";"作为分隔符时, 会有转义机制, 其他分隔符没有转义机制
                                   当你选用其他分隔符, 出了问题, 我不负责

  --version | -v                   显示版本

  --loadonly | -l                  作为库使用

  csv格式选项 (csv格式控制输出表格使用, 其他格式这些选项无效):
  --csv-output-bar <yes/no> | -b <yes/no>    是否输出表头, 默认为yes, 可以通过修改这个选项不输出表头, 往已有的csv表格追加写数据
  --csv-server-name <备注>  | -n <备注>      在csv表格左侧添加获取到硬件信息的设备备注, 备注可自行设置

  --ostype <HCI/PXE> | -t <HCI/PXE>          操作系统类型，目前支持"HCI"和"PXE"，默认为HCI

  --raidtools-use <system/local> | -r <system/local>  查询raid信息的工具使用系统自带还是本脚本指定的工具，system表示系统自带

  --extend <输出文件名> | -e <输出文件名>    包含ng需要的内容，输出是一个.tar.gz文件，默认为ng_info.tar.gz，如果文件名设置为"-"则输出掉标准输出

  --guide | -g                    输出各个字段收集方法

  --lang <语言>                   修改输出的语言，支持zh和en

  --hardware-type | -T <类型>      指定要输出的硬件类型（仅json格式有效）。如果要指定多个硬件类型，每个硬件类型使用英文逗号隔开

  --debug | -d                    启用调试模式，显示main函数下所有子步骤的执行过程

为了方便远程调用, 所有的参数, 都可以通过环境变量来设置
如: $0 --format human 也可以这样使用 export format=human; $0 效果是一样的
也可以通过 format=human; eval "\$(cat $0)" 达到一样的效果
format=human $0 效果也是一样的
所以可以通过如: echo "format=human; \$(cat $0)" | ssh 设备IP地址 这样的方式来获取远程设备的信息, 并可以通过变量定义的方式设置参数
EOF
}

guide(){
cat << EOF
{
    "server": [
        {
            "hci_version": $(jsonstr "使用shell命令cat /sf/version"$'\n'"只需要第一行"),
            "mainboard_model": $(jsonstr "使用shell命令dmidecode -t baseboard看Product Name字段"),
            "platform_type": $(jsonstr "使用shell命令uname -m"),
            "vendor": $(jsonstr "使用shell命令dmidecode -t system看Manufacturer字段"),
            "model": $(jsonstr "分两种情况："$'\n'"1）是深信服一体机"$'\n'"  dmidecode -t system看Family字段"$'\n'"  如果Family字段明显不对则看"$'\n'"  dmidecode -t chassis里有好几个下划线的那个字段"$'\n'"2）不是深信服一体机"$'\n'"  dmidecode -t system看Product Name字段"),
            "bios_version": $(jsonstr "使用shell命令dmidecode -t bios看Version字段"),
            "bmc_version": $(jsonstr "使用shell命令modprobe ipmi_devintf && modprobe ipmi_si先加载需要的驱动"$'\n'"然后使用shell命令ipmitool mc info看Firmware Revision字段"),
            "system_manager": $(jsonstr "看文件（文件夹）/sys/firmware/efi是否存在"$'\n'"存在则为UEFI，否则为legacy"),
            "product_name": $(jsonstr "使用shell命令dmidecode -t baseboard看Product Name字段")
        }
    ],
    "cpu": [
        {
            "model": $(jsonstr "使用shell命令dmidecode -t processor看Version字段"),
            "vendor": $(jsonstr "使用shell命令dmidecode -t processor看Manufacturer字段"),
            "core_count": $(jsonstr "分两种情况："$'\n'"1）x86或者之类的架构（C86基本上就是x86，懂的都懂）"$'\n'"  使用shell命令cat /proc/cpuinfo | grep 'cpu cores' | uniq | awk -F ': ' '{print \$2}'"$'\n'"2）arm或者之类的架构"$'\n'"  使用shell命令echo \$(( \$(cat /proc/cpuinfo | grep 'processor' | wc -l) / \$(dmidecode -t processor | grep '^Processor Information\$' | wc -l) ))"),
            "microcode": $(jsonstr "使用shell命令cat /proc/cpuinfo | grep 'microcode' | awk -F ': ' '{print $2}' | head -n 1"),
            "cpu_clock_speed": $(jsonstr "使用shell命令dmidecode -t processor | sed -rn 's/Current Speed: (.*) MHz/scale=2; \\1 \\/ 1000/p' | head -n 1 | bc")
        }
    ],
    "memory": [
        {
            "vendor": $(jsonstr "使用shell命令dmidecode -t memory看Manufacturer字段"$'\n'"注意其Size字段不能是异常的值（否则不是一张安装的内存条）"),
            "model": $(jsonstr "使用shell命令dmidecode -t memory看Part Number字段"$'\n'"注意其Size字段不能是异常的值（否则不是一张安装的内存条）"),
            "capacity": $(jsonstr "使用shell命令dmidecode -t memory看Size字段"),
            "specification": $(jsonstr "使用shell命令dmidecode -t memory看Type字段"$'\n'"注意其Size字段不能是异常的值（否则不是一张安装的内存条）"),
            "speed": $(jsonstr "使用shell命令dmidecode -t memory看speed字段"$'\n'"注意其Size字段不能是异常的值（否则不是一张安装的内存条）")
        }
    ],
    "disk": [
        {
            "model": $(jsonstr "使用shell命令smartctl -i {盘符}看Device Model、Model Number、Product任意一个字段"),
            "vendor": $(jsonstr "使用shell命令smartctl -i {盘符}看Vendor字段"),
            "serial_number": $(jsonstr "使用shell命令smartctl -i {盘符}看Serial Number字段"),
            "capacity": $(jsonstr "使用shell命令smartctl -i {盘符}看User Capacity或Total NVM Capacity字段"),
            "firmware_version": $(jsonstr "使用shell命令smartclt -i {盘符}看Firmware Version字段"),
            "interface_type": $(jsonstr "使用shell命令lsblk {盘符} -o TRAN"),
            "type": $(jsonstr "使用shell命令lsblk {盘符} -o TOTA")
        }
    ],
    "raid": [
        {
            "model": $(jsonstr "用对应的raid管理工具查看"$'\n'"例如LSI厂商卡可以这样storcli64 /c0 show然后关注和model、product name这些相关的字段"),
            "controller_chip_model": $(jsonstr "使用shell命令lspci -nns {PCI槽位}"),
            "card_model": $(jsonstr "使用shell命令lspci -vnns {PCI槽位}，看Subsystem那行"),
            "is_jbod": $(jsonstr "用对应的raid管理工具查看"$'\n'"例如LSI厂商的卡可以这样storcli64 /c0 show然后关注support jobd相关的字段"),
            "firmware_version": $(jsonstr "用对应的raid管理工具查看"$'\n'"例如LSI厂商的卡可以这样storcli64 /c0 show然后关注firmware相关的字段"),
            "fw_pkg_version": $(jsonstr "固件包版本，博通raid卡能获取到这个信息，不重要，但是有时候可能可以排查问题"),
            "bios_version": $(jsonstr "用对应raid管理工具查看"$'\n'"例如LSI厂商的卡可以这样storcli64 /c0 show然后关注BIOS相关的字段"),
            "driver_version": $(jsonstr "使用shell命令lspci -vs {PCI槽位}，看Kernel driver in use那行"),
            "pcieid": $(jsonstr "使用shell命令lspci -nns {PCI槽位}，看[xxxx:xxxx]这个格式的信息"),
            "subsystem_pid": $(jsonstr "使用shell命令lspci -vnns {PCI槽位}，看Subsystem那行的[xxxx:xxxx]信息"),
            "vendor": $(jsonstr "使用shell命令lspci -mvs {PCI槽位}，看Vendor字段")
        }
    ],
    "gpu": [
        {
            "model": $(jsonstr "使用shell命令lspci -vnns {PCI槽位}，看Subsystem那行"),
            "controller_chip_model": $(jsonstr "使用shell命令lspci -nns {PCI槽位}"),
            "driver_version": $(jsonstr "使用shell命令lspci -vs {PCI槽位}，看Kernel driver in use那行"),
            "pid": $(jsonstr "我们自己定义的产品id，看的是pci_id的后面半段（也就是设备id）"),
            "pcieid": $(jsonstr "我们自己定义的设备id，看的是subsystem_pid的后面半段（也就是子系统设备id）"),
            "pci_id": $(jsonstr "使用shell命令lspci -nns {PCI槽位}，看[xxxx:xxxx]这个格式的信息"),
            "subsystem_pid": $(jsonstr "使用shell命令lspci -vnns {PCI槽位}，看Subsystem那行的[xxxx:xxxx]信息"),
            "vendor": $(jsonstr "使用shell命令lspci -mvs {PCI槽位}，看Vendor字段")
        }
    ],
    "hba": [
        {
            "model": $(jsonstr "使用shell命令cat /sys/class/scsi_host/host{索引}/{modelname,product_version,version_product}"),
            "bus_controller": $(jsonstr "使用shell命令lspci -nns {PCI槽位}"),
            "subsystem": $(jsonstr "使用shell命令lspci -vnns {PCI槽位}，看Subsystem那行"),
            "firmware_version": $(jsonstr "使用shell命令cat /sys/class/scsi_host/host{索引}/{fwrev,fw_version,version_fw}"),
            "driver_version": $(jsonstr "使用shell命令lspci -vs {PCI槽位}，看Kernel driver in use那行"),
            "pcieid": $(jsonstr "使用shell命令lspci -nns {PCI槽位}，看[xxxx:xxxx]这个格式的信息"),
            "subsystem_pid": $(jsonstr "使用shell命令lspci -vnns {PCI槽位}，看Subsystem那行的[xxxx:xxxx]信息"),
            "vendor": $(jsonstr "使用shell命令lspci -mvs {PCI槽位}，看Vendor字段")
        }
    ],
    "network": [
        {
            "model": $(jsonstr "使用shell命令lspci -vvvnn -s {PCI槽位}，看Product Name字段"),
            "controller_chip_model": $(jsonstr "使用shell命令lspci -nns {PCI槽位}"),
            "subsystem": $(jsonstr "使用shell命令lspci -vnns {PCI槽位}，看Subsystem那行"),
            "firmware_version": $(jsonstr "分两种情况："$'\n'"1）是HCI"$'\n'"  使用realethtool -i {网口名字}"$'\n'"2）非HCI"$'\n'"  使用ethtool -i {网口名字}"$'\n'""$'\n'"看firmware-version字段"),
            "driver_version": $(jsonstr "使用shell命令lspci -vs {PCI槽位}，看Kernel driver in use那行"),
            "pcieid": $(jsonstr "使用shell命令lspci -nns {PCI槽位}，看[xxxx:xxxx]这个格式的信息"),
            "subsystem_pid": $(jsonstr "使用shell命令lspci -vnns {PCI槽位}，看Subsystem那行的[xxxx:xxxx]信息"),
            "vendor": $(jsonstr "使用shell命令lspci -mvs {PCI槽位}，看Vendor字段"),
            "optical_module": [
                {
                    "vendor_name": $(jsonstr "分两种情况："$'\n'"1）是HCI"$'\n'"  使用realethtool -m {网口名字}"$'\n'"2）非HCI"$'\n'"  使用ethtool -m {网口名字}"$'\n'""$'\n'"看Vendor name字段"),
                    "vendor_sn": $(jsonstr "分两种情况："$'\n'"1）是HCI"$'\n'"  使用realethtool -m {网口名字}"$'\n'"2）非HCI"$'\n'"  使用ethtool -m {网口名字}"$'\n'""$'\n'"看Vendor SN字段"),
                    "vendor_pn": $(jsonstr "分两种情况："$'\n'"1）是HCI"$'\n'"  使用realethtool -m {网口名字}"$'\n'"2）非HCI"$'\n'"  使用ethtool -m {网口名字}"$'\n'""$'\n'"看Vendor PN字段"),
                    "vendor_oui": $(jsonstr "分两种情况："$'\n'"1）是HCI"$'\n'"  使用realethtool -m {网口名字}"$'\n'"2）非HCI"$'\n'"  使用ethtool -m {网口名字}"$'\n'""$'\n'"看Vendor OUI字段"),
                    "vendor_rev": $(jsonstr "分两种情况："$'\n'"1）是HCI"$'\n'"  使用realethtool -m {网口名字}"$'\n'"2）非HCI"$'\n'"  使用ethtool -m {网口名字}"$'\n'""$'\n'"看Vendor rev字段")
                }
            ]
        }
    ],
    "ex_storage": [
        {
            "model": $(jsonstr "使用shell命令smartctl -i {盘符}看Device Model、Model Number、Product任意一个字段"),
            "vendor": $(jsonstr "使用shell命令smartctl -i {盘符}看Vendor字段"),
            "serial_number": $(jsonstr "使用shell命令smartctl -i {盘符}看Serial Number字段"),
            "capacity": $(jsonstr "使用shell命令smartctl -i {盘符}看User Capacity或Total NVM Capacity字段"),
            "firmware_version": $(jsonstr "使用shell命令smartclt -i {盘符}看Firmware Version字段"),
            "es_type": $(jsonstr "使用shell命令lsblk {盘符} -o TRAN")
        }
    ]
}
EOF
}

while [ $# -gt 0 ]; do
    case "$1" in
        "--help"|"-h"|"-?")
            usage
            exit 0
            ;;
        "--format"|"-f")
            if [ $# -lt 2 ]; then
                echo "参数\"${1}\"需要指定格式类型"
                echo "支持的格式类型有\"parse\", \"human\", \"csv\""
                echo "parse为便于其他脚本提取信息的格式，human为人类看着方便的格式，默认格式为parse"
                exit 1
            fi
            case "$2" in
                "parse"|"human"|"csv"|"json")
                    format="$2"
                    ;;
                "p")
                    format=parse
                    ;;
                "h")
                    format=human
                    ;;
                "c")
                    format=csv
                    ;;
                "j")
                    format=json
                    ;;
                *)
                    echo "参数\"${1}\"无效的格式\"${2}\""
                    exit 1
                    ;;
            esac
            shift
            ;;
        "--csv-output-bar"|"-b")
            if [ $# -lt 2 ]; then
                echo "参数\"${1}\"需要指定csv格式是否输出表头"
                echo "支持的选项有\"yes\", \"no\""
                echo "yes为csv格式输出表头，no为csv格式不输出表头，默认选项为yes"
                exit 1
            fi
            case "$2" in
                "yes"|"no")
                    csv_output_bar="$2"
                    ;;
                *)
                    echo "参数\"${1}\"无效的选项\"${2}\""
                    exit 1
                    ;;
            esac
            shift
            ;;
        "--csv-server-name"|"-n")
            if [ $# -lt 2 ]; then
                echo "参数\"${1}\"需要备注服务器名字"
                echo "服务器名字可以随意备注，用于csv表格显示"
                exit 1
            fi
            csv_server_name="$2"
            shift
            ;;
        "--print-bar"|"-p")
            print_csv_bar
            exit 0
            ;;
        "--version"|"-v")
            cat "$0" | grep '^##author' | awk -F '@' '{print $2}' | sed 's/\ //g'
            exit 0
            ;;
        "--split"| "--split-symb" |"-s")
            if [ $# -lt 2 ]; then
                echo "参数\"${1}\"需要输入你想要的分隔符"
                echo "分隔符可以随便定义，但是记住，如果不使用 ; 作为分隔符，出了问题，我不负责"
                exit 1
            fi
            split_symb="$2"
            shift
            ;;
        "--loadonly" | "-l")
            loadonly=yes
            ;;
        "--ostype" | "-t")
            if [ $# -lt 2 ]; then
                echo "参数\"$1\"需要指定操作系统类型"
                echo "支持的类型有\"HCI\", \"PXE\""
                exit 1
            fi
            ostype=$2
            shift
            ;;
        "--raidtools-use" | "-r")
            if [ $# -lt 2 ]; then
                echo "参数\"$1\"需要指定使用的raid工具"
                echo "支持的有\"system\"和\"local\""
                echo "system表示使用系统自带的工具（通常为HCI自带的查询工具），local表示本脚本指定的工具"
                exit 1
            fi
            raidtools_use=$2
            shift
            ;;
        "--extend" | "-e")
            extend_ng_info="yes"
            if [ $# -ge 2 ]; then
                extend_ng_filename=$2
                shift
            fi
            ;;
        "--guide" | "-g")
            guide
            exit 0
            ;;
        "--lang")
            if [ $# -lt 2 ]; then
                echo "参数\"${1}\"需要指定语言"
                echo "支持的语言有zh和en"
                exit 1
            fi
            output_lang=$2
            shift
            ;;
        "--hardware-type" | "-T")
            if [ $# -lt 2 ]; then
                echo "参数\"${1}\"需要指定硬件类型"
                exit 1
            fi
            output_hardware_type=$2
            shift
            ;;
        "--debug"|"-d")
            debug_main="yes"
            set -x
            ;;
        *)
            echo "无效的参数\"${1}\""
            usage
            exit 1
            ;;
    esac
    shift
done

if [ -z "$(echo "$format" | grep -w 'parse\|human\|csv\|json')" ]; then
    echo "无效的格式\"${format}\""
    usage
    exit 1
fi

if [ -z "$(echo "$ostype" | grep -w 'HCI\|PXE')" ]; then
    echo "无效的操作系统类型\"${ostype}\""
    exit 1
fi

if [ -z "$(echo "$raidtools_use" | grep -w 'system\|local')" ]; then
    echo "无效的raid工具\"$raidtools_use\""
    exit 1
fi

case "$raidtools_use" in
    "system")
        RAIDTOOLS_DIR="/sf/bin/raidtools/bin"
        ;;
    "local")
        RAIDTOOLS_DIR="$CUR_DIR/get_device_info_tools/$FRAMEWORK/raidtools"
        ;;
esac

if echo "$extend_ng_filename" | grep -q "/"; then
    echo "ng输出文件不能指定路径，只能输出到当前路径（文件名不支持/）" >&2
    exit 1
fi

scan_output_hardware_type

zh2en(){
    sed 's/服务器厂商/server vendor/g; s/大小\|容量/size/g; s/规格/form/g; s/速度/speed/g;
        s/内存数量/memory count/g; s/固件版本/firmware version/g; s/接口类型/interface type/g;
        s/硬盘类型/disk type/g; s/外置存储类型/external storage type/g; s/外置存储数量/external storage count/g;
        s/无内存/no memory/g; s/无硬盘/no disk/g; s/无RAID卡/no raid card/g; s/无HBA卡/no hba card/g;
        s/无网卡/no network card/g;
        s/RAID卡数量/raid card count/g; s/HBA卡数量/hba card count/g;
        s/网卡数量/network card count/g;
        s/硬盘数量/disk count/g; s/无外置存储/no external storage/g; s/RAID卡/raid card/g;
        s/HBA卡/hba card/g; s/网卡/network card/g; s/驱动/driver/g;
        s/内存/memory/g; s/硬盘/disk/g; s/外置存储/external storage/g;
        s/控制卡/controller/g; s/是否支持JBOD/support jbod/g;
        s/系统信息/system information/g; s/平台类型/platform type/g; s/BMC版本/bmc version/g;
        s/主板型号/baseboard model/g; s/服务器系统管理/server boot manager/g; s/系统管理/boot manager/g; s/服务器型号/server model/g; s/服务器架构/server frame/g;
        s/BIOS版本/bios version/g; s/产品名称/product name/g; s/CPU型号/cpu model/g; s/CPU厂商/cpu vendor/g;
        s/CPU架构/cpu frame/g; s/CPU微码/cpu microcode/g; s/CPU核数/cpu core/g; s/CPU线程数/cpu thread/g;
        s/CPU个数/cpu count/g;
        s/架构/frame/g; s/微码/microcode/g; s/核数/core/g;
        s/生产商\|厂商/vendor/g; s/型号/model/g; s/版本/version/g; s/无/no/g;
        s/\([0-9]\{1,\}\)个/count: \1/g;'
}

i18n(){
    if [ "$output_lang" == "zh" ]; then
        cat
    elif [ "$output_lang" == "en" ]; then
        zh2en
    else
        echo "Invalid language '$output_lang'" >&2
    fi
}

# 判断环境是否有smartctl.real，有的话说明环境里smartctl是一个缓存工具
SMARTCTL="smartctl"
if type smartctl.real >/dev/null 2>&1; then
    SMARTCTL="smartctl.real"
fi

if [ "$loadonly" != "yes" ]; then
    # 调试模式下不再重定向，正常打印出来
    if [ "$debug_main" == "yes" ]; then
        main_redirect=""
    else
        main_redirect="2>/dev/null"
    fi

    # 根据语言设置执行main函数
    if [ "$output_lang" == "zh" ]; then
        main $main_redirect
    else
        main $main_redirect | i18n     # 这种写法会导致main无法修改环境变量（因为此时main已经是i18n的子进程），下面的FAILED变量已经无法被main修改
    fi

    if [ "$FAILED" == "yes" ]; then
        echo "failed" >&2
        exit 1
    fi
fi

if [ "$extend_ng_info" == "yes" ]; then
    output_ng_info
fi
