#!/bin/bash

# UEFI x86 环境

# 2024/09/12 更新如下
# 1 无需重新制作引导，从物理设备引导启动
# 2 boot分区下增加soft_raid文件标识

main()
{
    # 存放执行日志
    CLOG_PATH="./install.log"
    # boot分区临时挂载点
    BOOT_DIR="/home/boot"
    
    # 判断当前是UEFI还是BIOS
    if [ -d /sys/firmware/efi ]; then
        echo "INFO:System is UEFI" | tee -a ${CLOG_PATH}
    else
        echo "ERROR:System is BIOS, exiting..." | tee -a ${CLOG_PATH}
        exit 1
    fi
    
    # 组建RAID，获取设备名
    mdadm --assemble --scan >>${CLOG_PATH} 2>&1
    
    # 使用blkid命令查找所有设备，然后使用grep筛选出包含vtpboot的行
    device_info=$(blkid | grep 'vtpboot')
    
    # 使用awk打印出设备名称
    device_name_with_partition=$(readlink -f /dev/disk/by-label/vtpboot)
    
    # 检查设备名称是否以/dev/md开头并以p2结尾
    if [[ $device_name_with_partition =~ ^(/dev/md[0-9]+)p2$ ]]; then
        md_device=${BASH_REMATCH[1]}
        echo "INFO: Found Soft RAID device: ${md_device}" | tee -a ${CLOG_PATH}
    else
        echo "ERROR: vtpboot is not on a Soft RAID device, exiting..." | tee -a ${CLOG_PATH}
        exit 1
    fi
    
    # 创建临时目录
    mkdir -p ${BOOT_DIR}
    
    # 挂载
    mount ${device_name_with_partition} ${BOOT_DIR}
    if [ $? -ne 0 ]; then
        echo "ERROR: Failed to mount ${device_name_with_partition} to ${BOOT_DIR}, exiting..." | tee -a ${CLOG_PATH}
        exit 1
    fi
    echo "INFO: Mounted ${device_name_with_partition} to ${BOOT_DIR}" | tee -a ${CLOG_PATH}
    
    # 创建文件标识
    touch "${BOOT_DIR}/boot/soft_raid"
    if [ $? -ne 0 ]; then
        echo "ERROR: Failed to create boot flag file, exiting..." | tee -a ${CLOG_PATH}
        exit 1
    fi
    umount ${device_name_with_partition} 
    echo "success to exec post.sh" | tee -a ${CLOG_PATH}
    return 0
}

main "$@"