#!/bin/bash

# 恢复删除lv的配置

set -e
set -u

if [ $# != 1 ] || [ "$1" = "-h" ] || [ "$1"  = "--help" ] ; then
        echo "Usage: $0 <gfid>  (eg: $0 abcdef)"
        exit 1
fi

g_size=256  # raw vg size (MB)

# color
RED='\033[0;31m'
BLUE='\033[0;34m'
NC='\033[0m'

function do_recover_lv_cfg()
{
        local gfid="$1"
        local raw_vgcfg="$2"

        cat "$raw_vgcfg" | awk "BEGIN {
                                        FS=\"\";
                                        stack_top = 0;
                                        start = 0;
                                        shard = 0;
                                        lv_index = 0;
                                        lv_max_index = 0;
                                    }

                                    function max(a, b) {
                                        if (a > b)
                                            return a;
                                        else
                                            return b;
                                    }

                                    {
                                        # 匹配配置起始位置, 同样的配置使用最后的
                                        if (\$0 ~ \"$gfid.*rubbish.*{\") {
                                            for (i = 0; i < lv_max_index; i = i + 2) {
                                                tmp = lv[i];
                                                if (\$0 == lvcfg[tmp]) {
                                                    lv_index = i;
                                                    break;
                                                }
                                            }
                                            start = 1;
                                            lv[lv_index++] = shard;
                                        }

                                        if (start) {
                                            for (i = 1; i <= NF; ++i) {
                                                if (\$i == \"{\") {
                                                    ++stack_top;
                                                } else if (\$i == \"}\") {
                                                    --stack_top;
                                                }
                                            }

                                            lvcfg[shard++] = \$0;
                                        }

                                        if (stack_top == 0 && start) {
                                            lv[lv_index++] = shard;
                                            shard++;
                                            start = 0;

                                            lv_max_index = max(lv_max_index, lv_index)
                                        }
                                    }

                                    END {
                                        if (!shard) {
                                            print \"configure not found\"

                                        } else if (stack_top != 0) {
                                            print \"configure corrupt\"
                                        } else {
                                            for (i = 0; i < lv_max_index; i = i + 2) {
                                                for (j = lv[i]; j < lv[i + 1]; ++j) {
                                                    print lvcfg[j]
                                                }

                                                print \"\n\n\"
                                            }
                                        }
                                    }"
}

function save_raw_cfg()
{
        local dev="$1"
        local vg="$2"
        local raw_vgcfg="$3"

        if ! [ -f "$raw_vgcfg" ]; then
                dd if="$dev" iflag=direct bs=1M count=$g_size 2>/dev/null | tr -cd '[[:print:]\n\]]' > "$raw_vgcfg"
        fi
}

function find_insert_line()
{
        local file=$1

        cat $file | awk 'BEGIN {
                                FS = "";
                                stack_top = -1;
                                start = 0
                                line=0
                        }

                        {
                                if ($0 ~ "logical_volumes") {
                                        stack_top = 0;
                                        start = 1;
                                }
                                if (start) {
                                        for (i = 1; i <= NF; ++i) {
                                                if ($i == "{") {
                                                        ++stack_top;
                                                } else if ($i == "}") {
                                                        --stack_top;
                                                }
                                        }
                                }

                                if (start && stack_top == 0) {
                                        start = 0;
                                        line  = NR - 1;
                                }
                        }

                        END {
                                print line
                        }'
}

function try_to_recovercfg()
{
        local vg="$1"
        local cfgfile="$2"
        local line
        local vgfile=/etc/lvm/backup/$vg

        cp "$vgfile" /root/$vg

        line=$(find_insert_line /root/$vg)

        if ((line == 0)); then
                echo "** recover configure on vg $vg failed **"
                exit 1
        fi

        sed -i "${line}r $cfgfile" /root/$vg

        vgcfgrestore --file /root/$vg $vg
        vgscan --ignorelockingfailure
        vgchange -ay --ignorelockingfailure
        vgchange --refresh

        printf "${BLUE}Restored configure file of vg $vg ${NC}\n"
}

function recover_lv_cfg()
{
        local gfid="$1"
        local old_IFS="$IFS"
        local pvinfo
        local dev
        local vg
        local cfg
        local raw_vgcfg
        local cfgfile
        local found=0
        local today=$(date +'%Y-%m-%d')

        IFS=$'\n'

        for pvinfo in $(/sf/vs/bin/vs_pvinfo.sh); do
                dev=$(echo $pvinfo | awk '{print $1}')
                vg=$(echo $pvinfo | awk '{print $2}')


                printf "Finding configure file of $gfid on $vg... "

                # 过滤掉缓存盘
                if lsblk "$dev" | grep -q cache; then
                        echo "ssd cache disk, skip"
                        continue
                fi

                raw_vgcfg="/root/$today/$vg/vgcfg.raw"
                cfgfile="/root/$today/$vg/$gfid.recover"

                mkdir -p /root/$today/$vg

                save_raw_cfg "$dev" "$vg" "$raw_vgcfg"

                cfg=$(do_recover_lv_cfg "$gfid" "$raw_vgcfg")

                if  echo $cfg | grep -q 'not found'; then
                        printf "${RED}not found${NC}\n"
                        continue
                elif echo $cfg | grep -q 'corrupt'; then
                        echo -e "${RED}configure corrupt, exit!!${NC}"
                        exit 1
                fi

                found=1

                echo "$cfg" | sed 's/rubbish/tmp./' > $cfgfile
                printf "${BLUE}found, configure file was save at $cfgfile ${NC}\n"
                printf "${BLUE}try to recover lv configure: $vg $gfid ${NC}\n"

                try_to_recovercfg $vg $cfgfile
        done

        IFS="$old_IFS"

        if ((!found)); then
                echo -e "${RED}No configure file was found for gfid: $gfid !!!${NC}"
                exit 1
        fi
}

recover_lv_cfg "$1"
