#!/usr/bin/env python
# -- coding: utf-8 --
# 与vs rpc服务相关接口调用
import json
from shard import ShardService
import pylib.utils.utiltools as common

shard_service = ShardService.ShardService()


def shard_gfid_to_base_gfid(gfid):
    base_gfid = shard_service.gfid2basegfid([gfid])[0]
    if base_gfid.op_ret:
        return None
    return base_gfid.bgfid


def shard_gfid_to_path(gfid):
    shard_path = shard_service.gfid2path([gfid])[0]
    if shard_path.op_ret:
        return None
    return shard_path.path


def shard_gfid_to_file_path(gfid):
    base_gfid = shard_gfid_to_base_gfid(gfid)
    if not base_gfid:
        return None
    base_path = shard_gfid_to_path(base_gfid)
    if not base_path:
        return None
    if base_path.startswith('/vs/snapshot/'):
        # /vs/snapshot/29/299c747e-76fc-418f-a548-572e955cf82c/4961.qcow2
        if len(base_path.split('/')) <= 4:
            return base_path

        base_gfid = base_path.split('/')[4]
        base_path = shard_gfid_to_path(base_gfid)
        if not base_path:
            return None
        return base_path
    else:
        # trash文件、备份文件，回收站文件返回原路径
        return base_path


# 将分片的path转化成base gfid的path
def file_to_base_file_path(path, all_shards):
    if path.startswith('/.vs/shard/'):
        # 分片文件
        # /.vs/shard/f3/f3abe315-063d-48bb-adc0-6551c3de9fa8/f3abe315-063d-48bb-adc0-6551c3de9fa8.00001.shard
        if len(path.split('/')) <= 4:
            return path
        base_gfid = path.split('/')[4]
        file_path = route_gfid_to_file_path(base_gfid, all_shards)
        if not file_path:
            return path
        else:
            return file_path
    elif path.startswith('/vs/snapshot/'):
        # 快照文件
        # /vs/snapshot/f5/f53c8446-cb52-4d63-8fa5-98d5278d3b02/1.qcow2
        if len(path.split('/')) <= 4:
            return path
        base_gfid = path.split('/')[4]
        file_path = route_gfid_to_file_path(base_gfid, all_shards)
        if not file_path:
            return path
        else:
            return file_path
    else:
        return path


def route_gfid_to_file_path(gfid, all_shards):
    for shard in all_shards:
        if gfid == shard['gfid']:
            path = shard['path']
            return file_to_base_file_path(path, all_shards)
    return None


# 提供两种方法，实现分片gfid转文件路径
# 如果all_shards为空，通过shard层gfid转文件路径（适用于少量文件查询）
# 如果all_shards有值，基于内存查询获取文件路径（适用于大量文件查询）
def gfid_to_base_file_path(gfid, all_shards):
    if not all_shards:
        return shard_gfid_to_file_path(gfid)
    else:
        return route_gfid_to_file_path(gfid, all_shards)


# 调用badblock_scan的地址转换接口， 将文件偏移转化成磁盘的偏移
def badblock_scan_change_offset(volume_id, brick_id, gfid, offset, size):
    """
    计算文件偏移转换到磁盘偏移
    :param volume_id: 卷ID
    :param brick_id: 文件所在的brick编号
    :param gfid: 文件gfid
    :param offset: 文件起始偏移
    :param size: 文件转换长度
    :return:
    """
    from libvs.glusterfs import Glusterfs
    from badblock_scan.model.change_offset import ChangeOffset

    glfs_vol = Glusterfs(volume_id)
    brick = glfs_vol.get_brick_by_no(brick_id)
    is_arbiter_on = glfs_vol.is_arbiter_on()
    change_offset = ChangeOffset(brick, is_arbiter_on, gfid, offset, size)
    co_results = change_offset.change_offset()
    if not co_results or len(co_results) != 1:
        # 返回的结果只有一个才是合理的
        common.logger.error('failed to get co_results: {}'.format(co_results))
        return None, None

    return co_results[0].dev, co_results[0].ph_start


# 通过route层接口，列出所有的分片
def route_list_all_shards(volume_id):
    from libvs.glusterfs import Glusterfs
    from vs_rpc.vs_rpc_api import VsRpcApi
    route_tasks = VsRpcApi()
    meta_routes = Glusterfs.get_meta_route(volume_id)

    for route in meta_routes:
        result = route_tasks.vs_list_file(route)
        if result['ret']:  # 成功
            # {'files': [{'route_version': 1, 'prio': u'normal', 'gfid': u'9ceb623d-612a-4023-ab2f-c57449aa5ea0',
            # 'route_info': [10, 6, 13], 'replica': 1, 'file_size': 4294967296, 'tier': 0, 'path': u'/t2.qcow2',
            # 'writecache': 0, 'op_ret': 0, 'api_open': 0, 'file_blocks': 1048576}], 'errno': 0, 'ret': True}
            return result['files']
        else:
            common.logger.error('failed to list shards from route: {}, result: {}'.format(route, result))

    error_msg = 'failed to list shards from meta_routes: {}'.format(meta_routes)
    raise common.CmdError(error_msg)


# 通过route层接口，列出所有脑裂的分片
def route_list_all_err_shards():
    from vs_rpc.vs_rpc_api import VsRpcApi
    route_tasks = VsRpcApi()

    result = route_tasks.vs_lookup_file()
    if not result['ret']:  # False 返回出有问题文件
        if result.get('err_files'):
            return result['err_files']
        else:
            error_msg = 'failed to list err file, result: {}'.format(result)
            raise common.CmdError(error_msg)

    # 返回True, 说明文件都正常
    return []


# 通过route层接口，检查副本状态
# "weak": 2,             # 除了排除副本之外的副本能为源的个数大于等于quorum
# "finish_weak": 3,      # 只需要指定的副本能为源即可
# "host": 4,             # 排除本主机brick其他副本都可作为源
def route_check_brick(brick_id, perfect=False):
    from vs_rpc.vs_rpc_api import VsRpcApi
    route_tasks = VsRpcApi()
    result = route_tasks.vs_check_file_by_no(brick_id, perfect)
    if not result['ret']:  # False 返回出有问题文件
        if result.get('err_files'):
            return result['err_files']
        else:
            error_msg = 'failed to check file, brick_id: {}, perfect: {}, result: {}'.format(brick_id, perfect, result)
            raise common.CmdError(error_msg)
    # 返回True, 说明文件都正常
    return []


# 获取client状态"NORMAL" or "UNNORMAL"
# "{\"brickid\":\"%s:%s\",\"volname\":\"%s\",\"vtype\":\"%s\",\"no\":%d,\"gf_server_version\":%u,\"status\":\"%s\"},"
def get_client_status(hostname=None):
    from vs_rpc.vs_rpc_api import VsRpcApi
    clnt = VsRpcApi()
    result = clnt.vs_get_clnt(hostname)
    dump_obj = json.loads(result)
    return dump_obj.get('brick')

# 通过route层接口，获取文件的信息
# {'route_version': 2, 'prio': u'normal', 'lgst_pgfid': u'', 'gfid': u'7d056531-cf3b-460b-942b-8cb27b021c7b', 'route_info': [9, 7, 11], 'replica': 1, 'file_size': 2281701376, 'tier': 0, 'path': u'/images/cluster/3852565135189.vm/vm-disk-1.qcow2', 'writecache': 0, 'op_ret': 0, 'api_open': 0, 'lgst_meta_type': u'', 'file_blocks': 4456448}
def route_get_file_info(gfid):
    gfids = []
    gfids.append(gfid)
    from vs_rpc.vs_rpc_api import VsRpcApi
    clnt = VsRpcApi()
    result = clnt.vs_get_fileinfo(gfids)
    return result['files'][0]