#!/sf/vs/bin/python
# -*- coding:utf-8 -*-
"""
## 分层元数据恢复工具

### 场景说明
该工具主要用于处理分层元数据损坏导致分层无法启动。技术支持曾经遇到过2个问题

1. 卷创建完成后，一线通过dd命令把分层磁盘元数据写0了，然后系统一直正常运行，知道tierd进程发生重启，重新加载分层元数据
2. 分层SSD元数据区域坏块

针对问题1:
- 如果由于dd损坏的区域不超过元数据区即可通过本工具恢复，因为损坏区域大部分未写过，并且无法通过crc校验的。

针对问题2:
- 如果损坏的是超级块，可以通过超级块备份恢复备份在/sf/cfg/vs/cache/目录下，
- 如果损坏的是brick/inode区，可以通过日志找到对应的brick信息进行恢复

### 命令说明

```shell
vsfire.py tier filter_brick [--log_path]
通过分层日志过滤出有效的brick信息，注意如果有磁盘替换可能会不准确

vsfire.py tier filter_inode [--log_path]
通过分层日志过滤出有效的inode信息，注意如果有磁盘替换可能会不准确

vsfire.py tier restore_brick [--ssd_path|--brickno|--brickid]
通过brickno和brickid恢复brick

vsfire.py tier restore_brick_from_tierlog [--ssd_path|--log_path]
通过分层日志自动解析出brick信息，并恢复这些brick，需要注意的是如果分层日志已经发生过轮转就会不准确

vsfire.py tier restore_brick_batch [--ssd_path|--input_path]
通过input文件批量恢复brick，需要注意input文件内容的格式为no brickid

vsfire.py tier restore_inode [--ssd_path|--brickno|--uuid|--no]
通过no/brickid/uuid/恢复inode信息

vsfire.py tier restore_inode_from_tierlog [--ssd_path|--log_path]
通过分层日志自动解析出inode信息，并恢复这些inode，需要注意的是如果分层日志已经发生过轮转就会不准确

vsfire.py tier restore_inode_batch [--ssd_path|--input_path]
通过input文件批量恢复inode，需要注意input文件内容的格式为no gfid brickno

vsfire.py tier restore_extent [--ssd_path]
自动识别SSD中非法extent，并把其格式化成初始化状态
```
"""

import os
import sys
import subprocess
import pylib.utils.utiltools as common

VS_TIER = u"./tier_source/vs_tier/vs_tier"
TIERD_LOG_PATH = u"/sf/log/vs/tierd/tierd.log"

def dump_super(ssd_path):
    if not ssd_path:
        return -1
    return os.system(VS_TIER + " -c dump_super -p {}".format(ssd_path))

def filter_brick(ssd_path="", path=TIERD_LOG_PATH):
    if path == None:
        return
    bricks = []
    brickno = 0
    with open(path, 'r') as fd:
        while True:
            line = fd.readline().strip()
            if not line:
                break
            brick = {}
            if line.find("tier_ssd_alloc_brick") > 0:
                brick_ssd = line.split("(")[1].split(")")[0]
                brick_ssd_path = u"/dev/" + brick_ssd + u"/" + brick_ssd + u"-tcache"
                if ssd_path and ssd_path != brick_ssd_path:
                    continue
                brick["no"] = brickno
                brick["brickid"] = line[line.index("/sf/data/vs/local/"):]
                brick["ssd"] = brick_ssd_path
                bricks.append(brick)
                brickno = brickno + 1
    return bricks

def filter_inode(ssd_path="", path=TIERD_LOG_PATH):
    if path == None:
        return

    bricks = filter_brick(ssd_path, path)
    if not bricks:
        print "no bricks found"
        return

    brickids = [brick['brickid'] for brick in bricks]
    print "brickids: "
    print brickids

    inodes = []
    with open(path, 'r') as fd:
        while True:
            line = fd.readline().strip()
            if not line:
                break
            if line.find("tier_brick_alloc_inode") > 0:
                splits = line.split(',')
                brickid = splits[1][splits[1].index("brickid = ") + len("brickid = "):]
                if brickids and (brickid not in brickids):
                    #print brickid + " is in bricks"
                    continue
                inode = {}
                inode["gfid"] = splits[0][splits[0].index("gfid = ") + len("gfid = "):]
                inode["brickid"] = brickid
                inode["no"] = splits[2][splits[2].index("no = ") + len("no = "):]
                inodes.append(inode)
            if line.find("tier_brick_del_inode") > 0:
                splits = line.split(',')
                inode = {}
                inode["gfid"] = splits[0][splits[0].index("remove file ") + len("remove file "):]
                inode["brickid"] = splits[1][splits[1].index("brick = ") + len("brick = "):]
                for i in inodes:
                    if i["gfid"] == inode["gfid"] and inode["brickid"] == i["brickid"]:
                        #print inode["gfid"] + " is unlinked."
                        inodes.remove(i)
                        break
    return inodes

def restore_brick(ssd_path, brickno, brickid):
    if not ssd_path or brickno < 0 or not brickid:
        return -1
    cmd = VS_TIER + " --cmd restore_brick --use 1 --unlink 0 "
    cmd = cmd + "--no {} --brickid {} --path {}".format(brickno, brickid, ssd_path)
    print(cmd)
    return os.system(cmd)

def restore_brick_from_tierlog(ssd_path, log_path=TIERD_LOG_PATH):
    if not ssd_path:
        return
    bricks = filter_brick(ssd_path, log_path)
    if not bricks:
        return

    print "bricks:"
    for brick in bricks:
        print("brickid: {}".format(brick))
    restore = raw_input("请输入restore_brick确认: ")
    if restore != "restore_brick":
        return
    for brick in bricks:
        restore_brick(ssd_path, brick["no"], brick["brickid"]);

def restore_brick_batch(ssd_path, input_path):
    if not ssd_path or not input_path:
        return
    bricks = []
    with open(input_path, 'r') as fd:
        while True:
            line = fd.readline().strip()
            if not line:
                break
            splits = line.split(' ')
            if len(splits) != 2:
                return
            brick = {}
            brick["no"] = splits[0].strip()
            brick["brickid"] = splits[1].strip()
            bricks.append(brick)
    if not bricks:
        return
    for brick in bricks:
        print("brickno: {} brickid: {}".format(brick["no"], brick["brickid"]))
    restore = raw_input("请输入restore_brick确认: ")
    if restore != "restore_brick":
        return
    for brick in bricks:
        restore_brick(ssd_path, brick["no"], brick["brickid"]);

def restore_inode(ssd_path, no, uuid, brickno):
    if not ssd_path or no < 0 or not uuid or brickno < 0:
        return -1
    cmd = VS_TIER + " --cmd restore_inode --use 1 --unlink 0 "
    cmd = cmd + "--no {} --uuid {} --brickno {} --path {}".format(no, uuid, brickno, ssd_path)
    print(cmd)
    return os.system(cmd)

def restore_inode_from_tierlog(ssd_path, log_path=TIERD_LOG_PATH):
    if not ssd_path:
        return

    inodes = filter_inode(ssd_path, log_path)
    if not inodes:
        return
    print "inodes:"
    for inode in inodes:
        print("gfid: {} no: {} brickid: {}".format(inode["gfid"], inode["no"], inode["brickid"]))
    restore = raw_input("请输入restore_inode确认:")
    if restore != "restore_inode":
        return
    for inode in inodes:
        restore_inode(ssd_path, inode["no"], inode["gfid"], brick["no"])

#通过输入文件restore inode
#输入文件格式: no uuid brickno
def restore_inode_batch(ssd_path, input_path):
    if not ssd_path or not input_path:
        return
    inodes = []
    with open(input_path, 'r') as fd:
        while True:
            line = fd.readline().strip()
            if not line:
                break
            splits = line.split(' ')
            if len(splits) != 3:
                return
            inode = {}
            inode["no"] = splits[0].strip()
            inode["gfid"] = splits[1].strip()
            inode["brickno"] = splits[2].strip()
            inodes.append(inode)
    for inode in inodes:
        print("gfid: {} no: {} brickno: {}".format(inode["gfid"], inode["no"], inode["brickno"]))
    restore = raw_input("请输入restore_inode确认:")
    if restore != "restore_inode":
        return
    for inode in inodes:
        restore_inode(ssd_path, inode["no"], inode["gfid"], inode["brickno"])

def restore_extent(ssd_path, log_path=TIERD_LOG_PATH):
    if not ssd_path:
        print("ssd_path is null")
        return
    restore = raw_input("请输入restore_inode确认:")
    if restore != "restore_inode":
        return
    cmd = VS_TIER + " --cmd restore_extent --path {}".format(ssd_path)
    os.popen(cmd)

@common.only_support_3x
def expand(op_type=''):
    """
        SSD热扩容
    """
    edsver = ""
    ver = common.get_vs_version()
    edsver_p = ver.split('.')
    if len(edsver_p) >= 4:
        edsver = edsver_p[3]

    if '3.0.6' in ver:
        from modules.tier.hot_add_ssd.expand_630X.vs_hot_add_ssd import ssd_expand
    elif '3.3.0' in ver:
        from modules.tier.hot_add_ssd.expand_670X.vs_hot_add_ssd import ssd_expand
    elif '3.5.0' in ver:
        from modules.tier.hot_add_ssd.expand_680.vs_hot_add_ssd import ssd_expand
    elif '3.6.0' in ver:
        from modules.tier.hot_add_ssd.expand_690.vs_hot_add_ssd import ssd_expand
    elif '3.7.0' in ver:
        from modules.tier.hot_add_ssd.expand_6X0.vs_hot_add_ssd import ssd_expand
    elif edsver == "309" or edsver == "501" or edsver == "502" or edsver == "503":
        from modules.tier.hot_add_ssd.EDS309_501_502_503.vs_hot_add_ssd import ssd_expand
    else:
        print 'This version: {} is not support ssd hot add!'.format(ver)

    ssd_expand(op_type)

    return


@common.only_support_3x
def capleak_fix():
   """
     1. 打分层补丁包之后，执行该工具之前，需重启关联的glusterfsd进程; 若已重启过，则不用重启
     2. 执行工具后，需再次重启glusterfsd进程，保证分层new data立即生效
   """
 
   from modules.tier.vst_prealloc_fiemap_tool import _capacity_leak_fix
   _capacity_leak_fix() 



def tier_flush():
    """分层回刷"""
    shell_script = "modules/tier/tier_flush.sh"
    process = subprocess.Popen(shell_script, shell=False, stdout=sys.stdout, stderr=sys.stderr)
    process.communicate()

    returncode = process.returncode
    if returncode == 0:
        print("Shell 脚本执行成功")
    else:
        print("Shell 脚本执行失败")
    return
 
