#!/sf/vs/bin/python
# -*- coding:utf-8 -*-
import errno
import fcntl
import json
import os
import sys

from cluster_manager.db.mongodb.disk import DiskDB
from cluster_manager.db.mongodb.pool import PoolDB


class SingleProc(object):
    """
    实现单例
    【注意：使用这个类后会残留pid文件】
    支持两种方式使用：
    with  SingleProc('/path/xxx'):
        ...
    或
    lock = SingleProc('/path/xxx')
    lock.lock()
    ...
    lock.unlock()
    建议采用第一种
    """

    def __init__(self, pid_file):
        self._pidfile_path = os.path.realpath(pid_file)
        self.__pidfile = None

    def __del__(self):
        if self._pidfile and self._pidfile != -1:
            self.unlock()

    @property
    def _pidfile(self):
        if self.__pidfile is None:
            self.__pidfile = open(self._pidfile_path, 'w')
        return self.__pidfile

    @_pidfile.setter
    def _pidfile(self, val):
        self.__pidfile = val

    def lock(self, block=False):
        if self._pidfile == -1:
            self._pidfile = None

        try:
            if not block:
                operation = fcntl.LOCK_EX | fcntl.LOCK_NB
            else:
                operation = fcntl.LOCK_EX
            fcntl.lockf(self._pidfile, operation)
        except IOError:
            print("another instance is running...")
            self._pidfile = -1
            return -1
        return 0

    def unlock(self):
        # 先清理文件，再解锁
        if self._pidfile == -1 or self._pidfile is None:
            return 0

        pidfile_path = self._pidfile_path

        try:
            if os.path.exists(pidfile_path):
                os.unlink(pidfile_path)
        except IOError as e:
            if e.errno != errno.ENOENT:
                print("error on clean pidfile {}:{}".format(pidfile_path, e))
                raise
        try:

            fcntl.lockf(self._pidfile, fcntl.LOCK_UN)
            self._pidfile.close()
            self._pidfile = -1

        except IOError:
            print("pidfile unlock fail")
            return -1
        return 0

    def __enter__(self):
        if self.lock() == 0:
            return self
        else:
            raise Exception('lock fail')

    def __exit__(self, exc_type, exc_value, traceback):
        self.unlock()


hostid = ""


def dump_diskdb():
    try:
        disk_db = DiskDB()
        if hasattr(disk_db, 'get_disk_list'):
            disk_list = disk_db.get_disk_list()
        elif hasattr(disk_db, 'list_disk'):
            disk_list = disk_db.list_disk()
        else:
            print("cannot get diskdb disk list")
            return False
        for disk in disk_list:
            with open("/root/vs-env/sf/log/vs/diskdb_{}.json".format(disk['sn']), 'w') as f:
                json.dump(disk, f, indent=4)
        return True
    except Exception as ex:
        print("dump_diskdb failed {}".format(ex))
        return False


def dump_pooldb():
    try:
        pool_db = PoolDB()
        pool_info = pool_db.driver.find_one({'type': "vs"})
        with open("/root/vs-env/sf/log/vs/pooldb.json", 'w') as f:
            json.dump(pool_info['hosts'], f, indent=4)
        return True
    except Exception as ex:
        print("dump_pooldb failed {}".format(ex))
        return False


def back_diskdb():
    try:
        disk_db = DiskDB()
        if hasattr(disk_db, 'get_disk_list'):
            disk_list = disk_db.get_disk_list()
        elif hasattr(disk_db, 'list_disk'):
            disk_list = disk_db.list_disk()
        else:
            print("cannot get diskdb disk list")
            return False
        for disk in disk_list:
            with open("/root/vs-env/sf/log/vs/diskdb_{}.json".format(disk['sn']), 'r') as f:
                disk_update = json.load(f)
            disk_db.update_disk(disk['hostid'], disk['sn'], disk_update)
        return True
    except Exception as ex:
        print("back_diskdb failed {}".format(ex))
        return False


def back_pooldb():
    try:
        with open('/root/vs-env/sf/log/vs/pooldb.json', 'r') as f:
            pool_info = json.load(f)
        update_pool = {}
        update_pool['hosts'] = pool_info
        print("pool_info = {}".format(update_pool))
        pool_db = PoolDB()
        pool_db.driver.update_or_insert({"type": "vs"}, update_pool)
        return True
    except Exception as ex:
        print("back_pooldb failed {}".format(ex))
        return False


def load_config():
    try:

        # 需要移动的数据盘
        with open('/root/vs-env/sf/log/vs/request_disk.json', 'r') as f:
            request_data_disk = json.load(f)
        print("need move data_disks {} = ".format(request_data_disk))

        # 需要添加的缓存盘
        with open('/root/vs-env/sf/log/vs/request_cache_disk.json', 'r') as f:
            request_cache_disk = json.load(f)
        print("need add one cache_disk {}".format(request_cache_disk))

        return request_data_disk, request_cache_disk
    except Exception as ex:
        print("load_config failed {}".format(ex))
        return False


def update_diskdb(request_data_disk, request_cache_disk):
    # 缓存盘要改poolid group
    # 数据盘要改group
    try:
        disk_db = DiskDB()
        cache_disk = None
        if hasattr(disk_db, 'get_disk_list'):
            disk_list = disk_db.get_disk_list()
        elif hasattr(disk_db, 'list_disk'):
            disk_list = disk_db.list_disk()
        else:
            print("cannot get diskdb disk list")
            return False

        global hostid
        for disk in disk_list:
            if disk['poolid']:
                poolid = disk['poolid']
            if disk['sn'] == request_cache_disk['disk_sn']:
                hostid = disk['hostid']

        print("update_diskdb poolid = {}".format(poolid))
        print("update_diskdb hostid = {}".format(hostid))
        for disk in disk_list:
            # 缓存盘
            if disk['sn'] == request_cache_disk['disk_sn']:
                with open("/sf/config/vs/disk/{}.json".format(request_cache_disk['disk']), 'r') as fp:
                    disk_conf = json.load(fp)
                    disk_group_id = disk_conf['disk_group_id']
                print("need updata cache_disk in diskdb = {}".format(disk))
                cache_disk = disk
                disk_update = {
                    "poolid": poolid,
                    "group": disk_group_id
                }
                cache_disk.update(disk_update)
                disk_db.update_disk(hostid, disk['sn'], disk_update)
                continue
            # 数据盘
            for data_disk in request_data_disk:
                if disk['sn'] == data_disk['sn']:
                    with open("/sf/config/vs/disk/{}.json".format(data_disk['disk_id']), 'r') as fp:
                        disk_conf = json.load(fp)
                        disk_group_id = disk_conf['disk_group_id']
                    print("need update data_disk in diskdb = {}".format(disk))
                    disk_update = {
                        "group": disk_group_id
                    }
                    disk_db.update_disk(hostid, disk['sn'], disk_update)
                    continue
        return cache_disk
    except Exception as ex:
        print("update_diskdb failed {}".format(ex))
        return False


def update_pooldb(request_data_disk, request_cache_disk):
    # 缓存盘新增
    # 数据盘要改group
    # {
    #   'category':  'DISK_HDD',
    #   'group': 1,
    #   'uuid':  'SVMware_Virtual_disk_6000c29eb26fd82ca78f1020dd23e5f3',
    #   'parts': [],
    #   'sn':  '6000c29eb26fd82ca78f1020dd23e5f3',
    #   'type':  'STORAGE_DATA'
    # }
    try:
        pool_db = PoolDB()
        pool_info = pool_db.driver.find_one({'type': "vs"})
        global hostid
        print("update_pooldb hostid = {}".format(hostid))
        for host in pool_info.get('hosts', []):
            if host['id'] == hostid:
                for disk in host.get('disks', []):
                    if disk['sn'] == request_cache_disk['sn']:
                        disk['group'] = request_cache_disk['group']
                        break
                else:
                    disk_update = {
                        'category': request_cache_disk['category'],
                        'group': request_cache_disk['group'],
                        'uuid': request_cache_disk['uuid'],
                        'parts': [],
                        'sn': request_cache_disk['sn'],
                        'type': "STORAGE_CACHE"
                    }
                    print("hosts = {} , add cache_disk = {}".format(hostid, disk_update))
                    host['disks'].append(disk_update)
            for disk in host.get('disks', []):
                print("pool disk = {}".format(disk))
                for data_disk in request_data_disk:
                    # 数据盘 改group为target_id,缓存盘还没加进来
                    if disk['sn'] == data_disk['sn']:
                        with open("/sf/config/vs/disk/{}.json".format(data_disk['disk_id']), 'r') as fp:
                            disk_conf = json.load(fp)
                            disk_group_id = disk_conf['disk_group_id']
                        disk['group'] = disk_group_id
                        print("this disk need update group_id to {}".format(disk_group_id))
        update_pool = {
            'hosts': pool_info['hosts']
        }
        print("update pool_info = {}".format(update_pool))
        pool_db.driver.update_or_insert({"type": "vs"}, update_pool)
        return True
    except Exception as ex:
        print("update_pooldb failed {}".format(ex))
        return False


def main():
    proc = SingleProc('/var/lock/update_disk_group.lock')
    if proc.lock() != 0:
        print('another vs_hot_add_ssd process is running')
        return 1

    result = dump_diskdb()
    if not result:
        print("dump_diskdb failed")
        return 1

    result = dump_pooldb()
    if not result:
        print("dump_pooldb failed")
        return 1

    result = load_config()
    if result:
        request_data_disk, request_cache_disk = result
    else:
        print("load_config failed")
        return 1

    cache_disk = update_diskdb(request_data_disk, request_cache_disk)
    if not cache_disk:
        print("update_diskdb failed")
        return 1

    result = update_pooldb(request_data_disk, cache_disk)
    if not result:
        print("update_pooldb failed")
        return 1


if __name__ == '__main__':
    sys.exit(main())
    # sys.exit(unitest())
