#!/sf/vs/bin/python
# -*- coding: utf-8 -*-
import os
import sys
import re
import json
import logging
import logging.handlers
import time
import threading
import pylib.utils.color as color
from pylib.utils.utiltools import get_vs_version, is_vs3x, is_vs2x, is_vs23_vs28, is_vs20_vs22, cli

result = 'log/damage_results'

reload(sys)
sys.setdefaultencoding('utf8')

logger = logging.getLogger(__name__)

BRICK_TYPE_DATA = 0
BRICK_TYPE_ARBITER = 1
BRICK_TYPE_META = 2
BRICK_TYPE_META_ARBITER = 3


REP_TYPE_DATA = 0
REP_TYPE_ARBITER = 1
REP_TYPE_META = 2

HOST_NAME_LEN = 17
VG_NAME_LEN = 38

REPLICATE_CLASS_INVALID = 0
REPLICATE_CLASS_2X_2REP = 1
REPLICATE_CLASS_2X_2REP_ARBITER = 2
REPLICATE_CLASS_2X_3REP = 3
REPLICATE_CLASS_3X_2REP = 4
REPLICATE_CLASS_3X_2REP_ARBITER = 5
REPLICATE_CLASS_3X_3REP_2ARBITER = 6

REP_DAMAGE_TYPE_NORMAL = 0
REP_DAMAGE_TYPE_CHARGE = 1
REP_DAMAGE_TYPE_BAD = 2
REP_DAMAGE_TYPE_DAMAGE = 3

FILE_TYPE_SHARD_FILE = 1
FILE_TYPE_NO_SHARD_FILE = 2


DAMAGE_LEVEL_A = 1      # 所有数据副本都已损坏，只剩下仲裁，或者连仲裁都已损坏
DAMAGE_LEVEL_B = 2      # 所有数据副本都有问题，有的数据副本已损坏，有的数据副本被指控或有BAD（指控和BAD只有一个）
DAMAGE_LEVEL_C = 3      # 至少有1个数据副本，但是超过半数的副本或仲裁已损坏
DAMAGE_LEVEL_D = 4      # 有副本或仲裁损坏，但是超过半数的副本或仲裁正常，分片可正常访问
DAMAGE_LEVEL_NORMAL = 0      # 无损坏

DEV_STATUS_NORMAL = 0
DEV_STATUS_HOST_OFFLINE = 1
DEV_STATUS_DISK_OFFLINE = 2


BREAKDOWN_TYPE_NORMAL = 0
BREAKDOWN_TYPE_DATA = 1
BREAKDOWN_TYPE_CACHE = 2

SUMMARY_HEADER_FMT = '{:12}\t{:20}\t{:20}\t{:20}\t{:20}\t{:32}\t{}\n'
SUMMARY_FMT = '  {:10}\t{:20}\t{:20}\t{:20}\t{:20}\t{:32}\t{}\n'

OK_STR = color.green('[OK]')
DONE_STR = color.green('[done]')


class Error(Exception):
    pass

def perror(str):
    sys.stderr.write("error: {}\n".format(str))

def get_human_size_string(size):
    if size < 0:
        return '未知'
    if size > 1024*1024*1024*1024:
        return '{:.2f}TB'.format(float(size)/float(1024*1024*1024*1024))
    elif size > 1024*1024*1024:
        return '{:.2f}GB'.format(float(size)/float(1024*1024*1024))
    elif size > 1024*1024:
        return '{:.2f}MB'.format(float(size)/float(1024*1024))
    elif size > 1024:
        return '{:.2f}KB'.format(float(size)/float(1024))
    else:
        return '{}B'.format(size)


def check_and_mkdir(dir):
    if os.path.exists(dir):
        if not os.path.isdir(dir):
            perror('已经存在同名文件，无法创建结果文件夹：{}'.format(dir))
            exit(-1)
    else:
        os.mkdir(dir)


def check_output_results_dir(output_dir):
    if not output_dir:
        output_dir = result
    check_and_mkdir(output_dir)
    check_and_mkdir(output_dir + '/A')
    check_and_mkdir(output_dir + '/B')
    check_and_mkdir(output_dir + '/C')
    check_and_mkdir(output_dir + '/D')
    check_and_mkdir(output_dir + '/N')


def get_result_file_level(myfile):
    damage_level = myfile['damage_level']
    if damage_level == DAMAGE_LEVEL_A:
        return 'A'
    if damage_level == DAMAGE_LEVEL_B:
        return 'B'
    if damage_level == DAMAGE_LEVEL_C:
        return 'C'
    if damage_level == DAMAGE_LEVEL_D:
        return 'D'
    return 'N'


def get_gfid_by_local_file(pathname, host):
    if host:
        cmdline = 'ssh root@{} "/sf/vs/bin/getfattr -d -m . -e hex \\\"{}\\\"" |grep "trusted.gfid"'.format(host, pathname)
    else:
        cmdline = '/sf/vs/bin/getfattr -d -m . -e hex "{}" |grep "trusted.gfid"'.format(pathname)
    lines = cli(cmdline, True)
    for line in lines:
        if line[0:15] == 'trusted.gfid=0x':
            gfid_orig = line[15:]
            return gfid_orig[0:8]+'-'+gfid_orig[8:12]+'-'+gfid_orig[12:16]+'-'+gfid_orig[16:20]+'-'+gfid_orig[20:]
    return None


def get_all_hosts_info():
    all_hosts = dict()
    lines = cli("vtpclustat", True)
    for line in lines:
        line = re.sub(", Local", "", line)
        strlist = re.sub("\s+"," ",line).split(" ")
        if len(strlist) == 4 and strlist[0][0:5] == "host-":
            online = 1
            if strlist[2].lower() == 'offline':
                online = 0
            all_hosts[strlist[0]] = {
                'host_name': strlist[0],
                'ip': strlist[3],
                'online': online
            }
    return all_hosts

def parse_brick(cluster_info, line):
    strlist = line.split()
    if len(strlist) != 2 and len(strlist) != 3 or strlist[0][0:5] != 'Brick' or strlist[1][0:5] != 'host-':
        raise Exception('Invalid brick line: {}'.format(line))
    brick_no = int(strlist[0][5:-1])
    brick_type = BRICK_TYPE_DATA
    if len(strlist) == 3:
        if strlist[2] == '(arbiter)':
            brick_type = BRICK_TYPE_ARBITER
        elif strlist[2] != '(meta)':
            brick_type = BRICK_TYPE_META
        elif strlist[2] != '(meta-arbiter)':
            brick_type = BRICK_TYPE_META_ARBITER
        else:
            raise Exception('Invalid brick line: {}'.format(line))
    strlist1 = strlist[1].split(':')
    if len(strlist1) != 2 or len(strlist1[0]) != HOST_NAME_LEN:
        raise Exception('Invalid brick line: {}'.format(line))
    host = strlist1[0]
    strlist2 = strlist1[1].split('/')
    vg_name = strlist2[5]
    brick_dir = strlist2[6]
    brick_name = '{}:Brick{}'.format(cluster_info['all_hosts'][host]['ip'], brick_no)
    brick = {
        'brick_no': brick_no,
        'brick_type': brick_type,
        'brick_name': brick_name,
        'host': host,
        'vg_name': vg_name,
        'brick_dir': brick_dir
    }
    return brick


def init_volume_brick_dev(cluster_info):
    volume_name = cluster_info['volume']['volume_name']
    online_hosts = get_online_volume_hosts(cluster_info)
    if len(online_hosts) == 0:
        raise Exception('All hosts of volume {} is offline!'.format(volume_name))
    cmdline = 'vs_cluster_cmd.sh e "/sf/vs/bin/vs_pvinfo.sh" --hosts {}'.format(','.join(online_hosts))
    lines = cli(cmdline, True)
    volume = cluster_info['volume']
    volume_vg_2_dev = volume['vg_2_dev']
    for line in lines:
        if line[0:5] == '/dev/':
            strlist = line.split()
            if len(strlist) == 2:
                volume_vg_2_dev[strlist[1]] = {'devname': strlist[0], 'status': DEV_STATUS_NORMAL}
    volume_bricks = volume['bricks']
    for brick_no in volume_bricks:
        brick = volume_bricks[brick_no]
        vg_name = brick['vg_name']
        if vg_name not in volume_vg_2_dev:
            if brick['host'] not in online_hosts:
                volume_vg_2_dev[vg_name] = {'devname': 'offline', 'status': DEV_STATUS_HOST_OFFLINE}
            else:
                volume_vg_2_dev[vg_name] = {'devname': 'offline', 'status': DEV_STATUS_DISK_OFFLINE}


def get_volume_info_3x(cluster_info, lines, volume_info):
    for line in lines:
        brick = parse_brick(cluster_info, line)
        volume_info['bricks'][brick['brick_no']] = brick


def get_volume_info_2x_no_arbiter(lines, volume_info):
    i = 0
    n = len(lines) - 1
    while i < n:
        brick1 = parse_brick(lines[i])
        brick2 = parse_brick(lines[i+1])
        i += 2
        if brick1['brick_type'] != BRICK_TYPE_DATA or brick2['brick_type'] != BRICK_TYPE_DATA:
            errstr = 'Unsupport Rep Group [{}:{}, {}:{}]'.format(brick1['brick_no'], brick1['brick_type'],
                                                                 brick2['brick_no'], brick2['brick_type'])
            raise Exception(errstr)
        volume_info['bricks'][brick1['brick_no']] = brick1
        volume_info['bricks'][brick2['brick_no']] = brick2
        volume_info['rep_groups'].append({
            'rep0': brick1['brick_no'],
            'rep1': brick2['brick_no']
        })
        key_bn = brick1['brick_no']
        key_dir = brick1['host'] + ':' + brick1['vg_name'] + '/' + brick1['brick_dir']
        volume_info['brick_2_rg'][key_bn] =  volume_info['dir_2_rg'][key_dir] = {
            'rep0': brick1['brick_no'],
            'rep1': brick2['brick_no'],
            'self': 'rep0'
        }
        key_bn = brick2['brick_no']
        key_dir = brick2['host'] + ':' + brick2['vg_name'] + '/' + brick2['brick_dir']
        volume_info['brick_2_rg'][key_bn] =  volume_info['dir_2_rg'][key_dir] = {
            'rep0': brick1['brick_no'],
            'rep1': brick2['brick_no'],
            'self': 'rep1'
        }


def get_volume_info_2x_with_arbiter(lines, volume_info):
    i = 0
    n = len(lines) - 2
    while i < n:
        brick1 = parse_brick(lines[i])
        brick2 = parse_brick(lines[i+1])
        brick3 = parse_brick(lines[i+2])
        i += 3
        if brick1['brick_type'] != BRICK_TYPE_DATA or brick2['brick_type'] != BRICK_TYPE_DATA\
                or brick3['brick_type'] != BRICK_TYPE_ARBITER:
            errstr = 'Unsupport Rep Group [{}:{}, {}:{}, {}:{}]'.format(brick1['brick_no'], brick1['brick_type'],
                                                                        brick2['brick_no'], brick2['brick_type'],
                                                                        brick3['brick_no'], brick3['brick_type'])
            raise Exception(errstr)
        volume_info['bricks'][brick1['brick_no']] = brick1
        volume_info['bricks'][brick2['brick_no']] = brick2
        volume_info['bricks'][brick3['brick_no']] = brick3
        volume_info['rep_groups'].append({
            'rep0': brick1['brick_no'],
            'rep1': brick2['brick_no'],
            'arbiter0': brick3['brick_no']
        })
        key_bn = brick1['brick_no']
        key_dir = brick1['host'] + ':' + brick1['vg_name'] + '/' + brick1['brick_dir']
        volume_info['brick_2_rg'][key_bn] =  volume_info['dir_2_rg'][key_dir] = {
            'rep0': brick1['brick_no'],
            'rep1': brick2['brick_no'],
            'arbiter0': brick3['brick_no'],
            'self': 'rep0'
        }
        key_bn = brick2['brick_no']
        key_dir = brick2['host'] + ':' + brick2['vg_name'] + '/' + brick2['brick_dir']
        volume_info['brick_2_rg'][key_bn] =  volume_info['dir_2_rg'][key_dir] = {
            'rep0': brick1['brick_no'],
            'rep1': brick2['brick_no'],
            'arbiter0': brick3['brick_no'],
            'self': 'rep1'
        }
        key_bn = brick3['brick_no']
        key_dir = brick3['host'] + ':' + brick3['vg_name'] + '/' + brick3['brick_dir']
        volume_info['brick_2_rg'][key_bn] =  volume_info['dir_2_rg'][key_dir] = {
            'rep0': brick1['brick_no'],
            'rep1': brick2['brick_no'],
            'arbiter0': brick3['brick_no'],
            'self': 'arbiter0'
        }


def get_volume_info(vs_ver, cluster_info, lines):
    t1 = time.time()
    volume_hosts = cluster_info['volume']['volume_hosts']
    for line in lines:
        strlist = line.split()
        if (len(strlist) == 2 or len(strlist) == 3) and strlist[1][0:5] == 'host-':
            strlist1 = strlist[1].split(':')
            if len(strlist1) == 2 and strlist1[0] not in volume_hosts:
                volume_hosts.append(strlist1[0])
    volume_info = cluster_info['volume']
    t2 = time.time()
    if is_vs3x(vs_ver):
        get_volume_info_3x(cluster_info, lines, volume_info)
    elif is_vs20_vs22(vs_ver):
        get_volume_info_2x_no_arbiter(lines, volume_info)
    elif is_vs23_vs28(vs_ver):
        if len(volume_hosts) < 3:
            get_volume_info_2x_no_arbiter(lines, volume_info)
        else:
            get_volume_info_2x_with_arbiter(lines, volume_info)
    else:
        raise Exception('Unsupport vs version: {}'.format(vs_ver))
    t3 = time.time()
    return volume_info


def get_local_host():
    cmdline = 'hostname'
    lines = cli(cmdline, True)
    for line in lines:
        if lines[0][0:5] == 'host-':
            return lines[0]
    return ''

def get_cluster_info():
    vs_ver = get_vs_version()
    print '\t获取当前集群的主机列表...',
    t1 = time.time()
    all_hosts = get_all_hosts_info()
    print '\t\t{}'.format(OK_STR)
    print '\t获取当前存储卷的所有brick列表',
    t2 = time.time()
    cmdline = 'gluster v i'
    lines = cli(cmdline, True)
    t3 = time.time()
    localhost = get_local_host()
    cluster_info = {
        'ver': vs_ver,
        'all_hosts': all_hosts,
        'localhost': localhost,
        'volume': {
            'volume_name': '',
            'volume_hosts': [],
            'bricks': dict(),
            'brick_2_vg': dict(),
            'vg_2_dev': dict(),
            'vm_list': [],
            'shards': {}
        }
    }
    volume_lines = []
    for line in lines:
        if line[0:13] == 'Volume Name: ':
            cluster_info['volume']['volume_name'] = line[13:]
        else:
            strlist = line.split()
            if len(strlist) != 2 and len(strlist) != 3:
                continue
            if len(strlist) == 2 and (strlist[0][0:5] != 'Brick' or strlist[1][0:5] != 'host-'):
                continue
            if len(strlist) == 3 and (strlist[0][0:5] != 'Brick' or strlist[1][0:5] != 'host-'):
                continue
            volume_lines.append(line)
    t4 = time.time()
    get_volume_info(vs_ver, cluster_info, volume_lines)
    print '\t\t{}'.format(OK_STR)
    print '\t初始化brick的设备名\t',
    t5 = time.time()
    init_volume_brick_dev(cluster_info)
    t6 = time.time()
    print '\t\t{}'.format(OK_STR)
    return cluster_info


def get_online_volume_hosts(cluster_info):
    volume = cluster_info['volume']
    all_hosts = cluster_info['all_hosts']
    online_hosts = []
    for hn in volume['volume_hosts']:
        if hn in all_hosts:
            host = all_hosts[hn]
            if host['online'] == 1:
                online_hosts.append(hn)
    return online_hosts


def get_all_online_hosts(cluster_info):
    all_hosts = cluster_info['all_hosts']
    online_hosts = []
    for hn in all_hosts:
        host = all_hosts[hn]
        if host['online'] == 1:
            online_hosts.append(hn)
    return online_hosts


def set_shard_route_info_3x(myshard, route):
    if len(route) == 2:
        myshard['rep0'] = int(route[0])
        myshard['rep1'] = int(route[1])
        myshard['rep_class'] = REPLICATE_CLASS_3X_2REP
        myshard['rep0_damage'] = REP_DAMAGE_TYPE_NORMAL
        myshard['rep1_damage'] = REP_DAMAGE_TYPE_NORMAL
    elif len(route) == 3:
        myshard['rep0'] = int(route[0])
        myshard['rep1'] = int(route[1])
        myshard['arbiter0'] = int(route[2])
        myshard['rep_class'] = REPLICATE_CLASS_3X_2REP_ARBITER
        myshard['rep0_damage'] = REP_DAMAGE_TYPE_NORMAL
        myshard['rep1_damage'] = REP_DAMAGE_TYPE_NORMAL
        myshard['arbiter0_damage'] = REP_DAMAGE_TYPE_NORMAL
    elif len(route) == 5:
        myshard['rep0'] = int(route[0])
        myshard['rep1'] = int(route[1])
        myshard['rep2'] = int(route[2])
        myshard['arbiter0'] = int(route[3])
        myshard['arbiter1'] = int(route[4])
        myshard['rep_class'] = REPLICATE_CLASS_3X_3REP_2ARBITER
        myshard['rep0_damage'] = REP_DAMAGE_TYPE_NORMAL
        myshard['rep1_damage'] = REP_DAMAGE_TYPE_NORMAL
        myshard['rep2_damage'] = REP_DAMAGE_TYPE_NORMAL
        myshard['arbiter0_damage'] = REP_DAMAGE_TYPE_NORMAL
        myshard['arbiter1_damage'] = REP_DAMAGE_TYPE_NORMAL
    else:
        raise Exception('Invalid route len')


def parse_rpc_list_shard_line_3x(all_shard, line):
    strlist = re.sub('\s', '', line).split('|')
    if len(strlist) <= 10:
        return 0
    try:
        shard_size = int(strlist[3])
    except:
        return 0
    route = re.sub('[\[\]]', '', strlist[6]).split(',')
    tmp_list1 = strlist[2].split('/')
    if tmp_list1[1] == "images" or tmp_list1[1] == "backup" or tmp_list1[1] == "template" or tmp_list1[1] == "private":
        gfid = strlist[1]
        file_type = FILE_TYPE_NO_SHARD_FILE
        pathname = strlist[2]
        shard_id = -1
    elif len(tmp_list1) >= 6 and tmp_list1[2] == 'shard':
        tmp_list2 = tmp_list1[5].split('.')
        if len(tmp_list2) < 3:
            return 0
        gfid = tmp_list2[0]
        file_type = FILE_TYPE_SHARD_FILE
        pathname = ''
        shard_id = int(tmp_list2[1])
        if shard_size < 0:
            shard_size = 4294967296
    else:
        return 0
    if gfid not in all_shard:
        myfile = {
            'gfid': gfid,
            'file_type': file_type,
            'pathname': pathname,
            'size': 0,
            'shard_num': 0,
            'status': dict(),
            'damage_level': '',
            'max_shard_id': 0,
            'rep_class': REPLICATE_CLASS_INVALID,
            'shards': dict()
        }
        all_shard[gfid] = myfile
    else:
        myfile = all_shard[gfid]
    if shard_id in myfile['shards']:
        return 0
    if shard_id > myfile['max_shard_id']:
        myfile['max_shard_id'] = shard_id
    myshard = {
        'shard_id': shard_id,
        'size': shard_size,
        'shard_gfid': strlist[1],
        'check_full': False
    }
    set_shard_route_info_3x(myshard, route)
    if myfile['rep_class'] == REPLICATE_CLASS_INVALID:
        myfile['rep_class'] = myshard['rep_class']
    elif myfile['rep_class'] != myshard['rep_class']:
        raise Exception('Replicate class error!')
    myfile['shards'][shard_id] = myshard
    return 1


def scan_brick_shard(all_shards, brick):
    cmdline = 'vs_rpc_tool --cmd list --brickno {}'.format(brick['brick_no'])
    t1 = time.time()
    lines = cli(cmdline, True)
    t2 = time.time()
    shards_cnt = 0
    for line in lines:
        shards_cnt += parse_rpc_list_shard_line_3x(all_shards, line)
    t3 = time.time()
    return shards_cnt


def scan_shards_of_bricks_3x(cluster_info, bricks):
    brick_num = len(bricks)
    finish_num = 0
    shards = cluster_info['volume']['shards']
    shards_cnt = 0
    brick_no_list = bricks.keys()
    print('\t待扫描brick列表：{}'.format(str(brick_no_list)))
    first_print = True
    back_len = 0
    for brick_no in bricks:
        mybrick = bricks[brick_no]
        shards_cnt += scan_brick_shard(shards, mybrick)
        finish_num += 1
        if first_print:
            cnt_str = str(finish_num)
            back_len = len(cnt_str)
            sys.stdout.write('\t共{}个brick，已完成：{}'.format(brick_num, cnt_str))
            first_print = False
        else:
            for i in range(0,back_len):
                sys.stdout.write('\b')
            cnt_str = str(finish_num)
            back_len = len(cnt_str)
            sys.stdout.write(cnt_str)
        sys.stdout.flush()
    sys.stdout.write('\t\t\t{}\n'.format(DONE_STR))
    sys.stdout.flush()
    cluster_info['volume']['shards_cnt'] = shards_cnt


def parse_data_disk_conf(cluster_info, conf):
    breakdown_vgs = cluster_info['volume']['breakdown_disks']['vgs']
    tmp_breakdown_vgs = []
    breakdown_bricks = cluster_info['volume']['breakdown_disks']['bricks']
    for part in conf['part_array']:
        part_uuid = part['part_uuid']
        if len(part_uuid) == VG_NAME_LEN and part_uuid[6] == '-' and part_uuid[11] == '-' \
            and part_uuid[16] == '-' and part_uuid[21] == '-' and part_uuid[26] == '-':
            breakdown_vgs.append(part_uuid)
            tmp_breakdown_vgs.append(part_uuid)
    bricks = cluster_info['volume']['bricks']
    for brick_no in bricks:
        mybrick = bricks[brick_no]
        brick_type = mybrick['brick_type']
        if not (brick_type == BRICK_TYPE_DATA or brick_type == BRICK_TYPE_ARBITER):
            continue
        if mybrick['vg_name'] in tmp_breakdown_vgs:
            mybrick['breakdown_type'] = BREAKDOWN_TYPE_DATA
            breakdown_bricks[brick_no] = mybrick
        elif brick_no not in breakdown_bricks:
            mybrick['breakdown_type'] = BREAKDOWN_TYPE_NORMAL


def get_wcache_vg_of_cache_disk(conf):
    for part in conf['part_array']:
        nodes = part['part_uuid'].split('-')
        if nodes[-1] == 'wcache':
            return part['part_uuid']
    raise Exception('failed to get wcache partition')


def get_brick_by_vg_and_brick_dir(cluster_info, vg_name, brick_dir):
    all_bricks = cluster_info['volume']['bricks']
    for brick_no in all_bricks:
        mybrick = all_bricks[brick_no]
        if mybrick['vg_name'] == vg_name and mybrick['brick_dir'] == brick_dir:
            return mybrick
    return None


def parse_cache_disk_conf(cluster_info, host, conf):
    wcache_vg = get_wcache_vg_of_cache_disk(conf)
    breakdown_vgs = cluster_info['volume']['breakdown_disks']['vgs']
    breakdown_bricks = cluster_info['volume']['breakdown_disks']['bricks']
    bricks = cluster_info['volume']['bricks']
    wcache_conf_pathname = '/sf/cfg/vs/cache/wcache.json'
    cmdline = 'ssh root@{} "cat {}"'.format(host, wcache_conf_pathname)
    conf_content = cli(cmdline, False)
    try:
        conf = json.loads(conf_content)
    except Exception as err:
        perror('加载配置文件{}:{}失败，请检查网络是否连通，配置文件是否完好'.format(host, '/sf/cfg/vs/cache/wcache.conf'))
        exit(-1)
    for cache_disk in conf['maps']:
        if cache_disk['uuid'] == wcache_vg:
            for brick in cache_disk['bricks']:
                brick_path = brick['brickId']
                nodes = brick_path.split('/')
                if len(nodes) != 7:
                    continue
                mybrick = None
                if not (nodes[1] == 'sf' and nodes[2] == 'data' and nodes[3] == 'vs' and nodes[4] == 'local'):
                    continue
                breakdown_vgs.append(nodes[5])
    for brick_no in bricks:
        mybrick = bricks[brick_no]
        if mybrick['vg_name'] in breakdown_vgs and (
                'breakdown_type' not in mybrick or mybrick['breakdown_type'] == BREAKDOWN_TYPE_NORMAL):
            mybrick['breakdown_type'] = BREAKDOWN_TYPE_CACHE
            breakdown_bricks[mybrick['brick_no']] = mybrick


def parse_one_breakdown_disk(cluster_info, disk_conf):
    str_list = disk_conf.split('_')
    if len(str_list) != 2 or len(str_list[0]) != 12 or str_list[1] == '':
        perror('非法磁盘配置文件名: {}，请检查后重新输入！'.format(disk_conf))
        exit(-1)
    host = 'host-' + str_list[0]
    cmdline = 'ssh root@{} "cat /sf/cfg/vs/disk/{}"'.format(host, disk_conf)
    conf_content = cli(cmdline, False)
    try:
        conf = json.loads(conf_content)
    except Exception as err:
        perror('加载配置文件{}失败，请检查网络是否连通，配置文件是否完好'.format(disk_conf))
        exit(-1)
    if conf['storage_type'] == 'STORAGE_DATA':
        parse_data_disk_conf(cluster_info, conf)
    elif conf['storage_type'] == 'STORAGE_CACHE':
        parse_cache_disk_conf(cluster_info, host, conf)


def parse_breakdown_disks(cluster_info, breakdown_disk):
    disk_list = breakdown_disk.split(',')
    cluster_info['volume']['breakdown_disks'] = {
        'vgs': [],
        'bricks': dict()
    }
    for disk_conf in disk_list:
        parse_one_breakdown_disk(cluster_info, disk_conf)


def parse_breakdown_hosts(cluster_info, breakdown_host):
    host_list = breakdown_host.split(',')
    cluster_info['volume']['breakdown_disks'] = {
        'vgs': [],
        'bricks': dict()
    }
    bricks = cluster_info['volume']['bricks']
    breakdown_vgs = cluster_info['volume']['breakdown_disks']['vgs']
    breakdown_bricks = cluster_info['volume']['breakdown_disks']['bricks']
    for brick_no in bricks:
        mybrick = bricks[brick_no]
        if mybrick['host'] not in host_list:
            continue
        brick_type = mybrick['brick_type']
        if brick_type == BRICK_TYPE_DATA or brick_type == BRICK_TYPE_ARBITER:
            mybrick['breakdown_type'] = BREAKDOWN_TYPE_DATA
            breakdown_bricks[brick_no] = mybrick
            if mybrick['vg_name'] not in breakdown_vgs:
                breakdown_vgs.append(mybrick['vg_name'])


def is_shard_damage_A_3x(myshard):
    rep_class = myshard['rep_class']
    if rep_class == REPLICATE_CLASS_3X_2REP or rep_class == REPLICATE_CLASS_3X_2REP_ARBITER:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        return rep0_damage == REP_DAMAGE_TYPE_DAMAGE and rep1_damage == REP_DAMAGE_TYPE_DAMAGE
    elif rep_class == REPLICATE_CLASS_3X_3REP_2ARBITER:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        rep2_damage = myshard['rep2_damage']
        return rep0_damage == REP_DAMAGE_TYPE_DAMAGE and rep1_damage == REP_DAMAGE_TYPE_DAMAGE \
               and rep2_damage == REP_DAMAGE_TYPE_DAMAGE
    else:
        raise Exception('Not support yet')


def is_shard_damage_B_3x(myshard):
    rep_class = myshard['rep_class']
    if rep_class == REPLICATE_CLASS_3X_2REP or rep_class == REPLICATE_CLASS_3X_2REP_ARBITER:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        return rep0_damage != REP_DAMAGE_TYPE_NORMAL and rep1_damage != REP_DAMAGE_TYPE_NORMAL
    elif rep_class == REPLICATE_CLASS_3X_3REP_2ARBITER:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        rep2_damage = myshard['rep2_damage']
        return rep0_damage != REP_DAMAGE_TYPE_NORMAL and rep1_damage != REP_DAMAGE_TYPE_NORMAL \
               and rep2_damage != REP_DAMAGE_TYPE_NORMAL
    else:
        raise Exception('Not support yet')


def is_shard_damage_C_3x(myshard):
    rep_class = myshard['rep_class']
    if rep_class == REPLICATE_CLASS_3X_2REP:
        return False
    elif rep_class == REPLICATE_CLASS_3X_2REP_ARBITER:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        arbiter0_damage = myshard['arbiter0_damage']
        if rep0_damage == REP_DAMAGE_TYPE_NORMAL and rep1_damage != REP_DAMAGE_TYPE_NORMAL and arbiter0_damage != REP_DAMAGE_TYPE_NORMAL:
            return True
        if rep0_damage != REP_DAMAGE_TYPE_NORMAL and rep1_damage == REP_DAMAGE_TYPE_NORMAL and arbiter0_damage != REP_DAMAGE_TYPE_NORMAL:
            return True
        return False
    elif rep_class == REPLICATE_CLASS_3X_3REP_2ARBITER:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        rep2_damage = myshard['rep2_damage']
        arbiter0_damage = myshard['arbiter0_damage']
        arbiter1_damage = myshard['arbiter1_damage']
        damage_cnt = 0
        if rep0_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if rep1_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if rep2_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if arbiter0_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if arbiter1_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        return damage_cnt >= 3 and (rep0_damage == REP_DAMAGE_TYPE_NORMAL 
                                or rep1_damage == REP_DAMAGE_TYPE_NORMAL or rep2_damage == REP_DAMAGE_TYPE_NORMAL)
    else:
        raise Exception('Not support yet')


def is_shard_damage_D_3x(myshard):
    rep_class = myshard['rep_class']
    if rep_class == REPLICATE_CLASS_3X_2REP:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        return (rep0_damage != REP_DAMAGE_TYPE_NORMAL and rep1_damage == REP_DAMAGE_TYPE_NORMAL) \
               or (rep0_damage == REP_DAMAGE_TYPE_NORMAL and rep1_damage != REP_DAMAGE_TYPE_NORMAL)
    elif rep_class == REPLICATE_CLASS_3X_2REP_ARBITER:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        arbiter0_damage = myshard['arbiter0_damage']
        damage_cnt = 0
        if rep0_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if rep1_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if arbiter0_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        return damage_cnt == 1
    elif rep_class == REPLICATE_CLASS_3X_3REP_2ARBITER:
        rep0_damage = myshard['rep0_damage']
        rep1_damage = myshard['rep1_damage']
        rep2_damage = myshard['rep2_damage']
        arbiter0_damage = myshard['arbiter0_damage']
        arbiter1_damage = myshard['arbiter1_damage']
        damage_cnt = 0
        if rep0_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if rep1_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if rep2_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if arbiter0_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        if arbiter1_damage != REP_DAMAGE_TYPE_NORMAL:
            damage_cnt += 1
        return damage_cnt < 3 and damage_cnt > 0
    else:
        raise Exception('Not support yet')


def parse_shard_rep_brick_3x(cluster_info, line):
    if line[0:5] != 'host-':
        logger.error('Invalid rep lines, first line:{}'.format(line))
        return -1
    str_list = line.split(':')
    if len(str_list) != 2:
        logger.error('Invalid rep lines, first line:{}'.format(line))
        return -1
    host = str_list[0]
    brick_path = str_list[1]
    nodes = brick_path.split('/')
    if len(nodes) != 7:
        logger.error('invalid brick path, line: {}'.format(line))
        return -1
    mybrick = None
    if nodes[1] == 'sf' and nodes[2] == 'data' and nodes[3] == 'vs' and nodes[4] == 'local':
        mybrick = get_brick_by_vg_and_brick_dir(cluster_info, nodes[5], nodes[6])
    if not mybrick:
        logger.error('解释分片{}的pathinfo失败，找不到路径{}对应的brick'.format(myshard['gfid'], line))
        return -1
    return mybrick['brick_no']


def parse_shard_rep_pathinfo_3x(cluster_info, rep_lines):
    line = rep_lines[0]
    brick_no = parse_shard_rep_brick_3x(cluster_info, rep_lines[0])
    if brick_no < 0:
        return None
    del(rep_lines[0])
    rep_pathinfo = { 'brick_no': brick_no }
    for line in rep_lines:
        field_list = line.split('=')
        if len(field_list) != 2:
            #logger.error('invalid line: {}'.format(line))
            continue
        rep_pathinfo[field_list[0]] = field_list[1]
    return rep_pathinfo


def parse_shard_pathinfo_3x(cluster_info, lines):
    rep_lines = []
    shard_pathinfo = []
    for line in lines:
        if line[0:5] == 'host-':
            if len(rep_lines) > 0:
                rep_pathinfo = parse_shard_rep_pathinfo_3x(cluster_info, rep_lines)
                if not rep_pathinfo:
                    return None
                shard_pathinfo.append(rep_pathinfo)
                rep_lines = []
            rep_lines.append(line)
        elif len(rep_lines) > 0:
            rep_lines.append(line)
    if len(rep_lines) > 0:
        rep_pathinfo = parse_shard_rep_pathinfo_3x(cluster_info, rep_lines)
        if not rep_pathinfo:
            return None
        shard_pathinfo.append(rep_pathinfo)
    return shard_pathinfo


def mark_shard_rep_damage_by_brickno(myshard, brick_no, damage_level):
    if myshard['rep0'] == brick_no:
        if myshard['rep0_damage'] < damage_level:
            myshard['rep0_damage'] = damage_level
        return
    if myshard['rep1'] == brick_no:
        if myshard['rep1_damage'] < damage_level:
            myshard['rep1_damage'] = damage_level
        return
    if myshard['arbiter0'] == brick_no:
        if myshard['arbiter0_damage'] < damage_level:
            myshard['arbiter0_damage'] = damage_level
        return
    if 'rep2' in myshard and myshard['rep2'] == brick_no:
        if myshard['rep2_damage'] < damage_level:
            myshard['rep2_damage'] = damage_level
        return
    if 'arbiter1' in myshard and myshard['arbiter1'] == brick_no:
        if myshard['arbiter1_damage'] < damage_level:
            myshard['arbiter1_damage'] = damage_level
        return
    logger.error('cannot found rep of shard[{}] by brick no[{}]'.format(myshard['shard_gfid'], brick_no))


def check_shard_damage_full_2rep_arbiter(cluster_info, myshard, shard_pathinfo):
    if len(shard_pathinfo) != 3:
        logger.error('Invalid shard pathinfo, rep count[{}], expect 3!'.format(len(shard_pathinfo)))
        return
    bricks = cluster_info['volume']['bricks']
    tier_status = 'user.glusterfs.tier_status'
    wcache_status = 'user.glusterfs.wcache'
    dirty_status = 'user.afr.dirty'
    normal_charge_value = '0x000000000000000000000000'
    normal_tier_value = '0x0000000000000000'
    rep0_charge_key = 'trusted.afr.{}-vclnt-{}'.format(cluster_info['volume']['volume_name'], 3)
    rep1_charge_key = 'trusted.afr.{}-vclnt-{}'.format(cluster_info['volume']['volume_name'], 4)
    arbiter0_charge_key = 'trusted.afr.{}-vclnt-{}'.format(cluster_info['volume']['volume_name'], 5)
    bad_key = 'trusted.file_status'
    nobad_value = '0x000000'
    for rep_pathinfo in shard_pathinfo:
        brick_no = rep_pathinfo['brick_no']
        mybrick = bricks[brick_no]
        if len(rep_pathinfo) == 1:
            mark_shard_rep_damage_by_brickno(myshard, brick_no, REP_DAMAGE_TYPE_DAMAGE)
            continue
        if 'breakdown_type' in mybrick and mybrick['breakdown_type'] == BREAKDOWN_TYPE_CACHE:
            if tier_status not in rep_pathinfo or rep_pathinfo[tier_status] != normal_tier_value:
                mark_shard_rep_damage_by_brickno(myshard, brick_no, REP_DAMAGE_TYPE_DAMAGE)
            if wcache_status not in rep_pathinfo or rep_pathinfo[wcache_status] != normal_tier_value:
                mark_shard_rep_damage_by_brickno(myshard, brick_no, REP_DAMAGE_TYPE_DAMAGE)
        if myshard['rep0_damage'] == REP_DAMAGE_TYPE_NORMAL and rep0_charge_key in rep_pathinfo \
                and rep_pathinfo[rep0_charge_key] != normal_charge_value:
            myshard['rep0_damage'] = REP_DAMAGE_TYPE_CHARGE
        if myshard['rep1_damage'] == REP_DAMAGE_TYPE_NORMAL and rep1_charge_key in rep_pathinfo \
                and rep_pathinfo[rep1_charge_key] != normal_charge_value:
            myshard['rep1_damage'] = REP_DAMAGE_TYPE_CHARGE
        if myshard['arbiter0_damage'] == REP_DAMAGE_TYPE_NORMAL and arbiter0_charge_key in rep_pathinfo \
                and rep_pathinfo[arbiter0_charge_key] != normal_charge_value:
            myshard['arbiter0_damage'] = REP_DAMAGE_TYPE_CHARGE
        if bad_key in rep_pathinfo and rep_pathinfo[bad_key] != nobad_value:
            mark_shard_rep_damage_by_brickno(myshard, brick_no, REP_DAMAGE_TYPE_BAD)


def check_shard_damage_full_3rep_2arbiter(cluster_info, myshard, shard_pathinfo):
    if len(shard_pathinfo) != 5:
        logger.error('Invalid shard pathinfo, rep count[{}], expect 5!'.format(len(shard_pathinfo)))
        return
    bricks = cluster_info['volume']['bricks']
    charge_field_header = 'trusted.afr.{}-vclnt-'.format(cluster_info['volume']['volume_name'])
    tier_status = 'user.glusterfs.tier_status'
    wcache_status = 'user.glusterfs.wcache'
    dirty_status = 'user.afr.dirty'
    normal_charge_value = '0x000000000000000000000000'
    normal_tier_value = '0x0000000000000000'
    rep0_charge_key = 'trusted.afr.{}-vclnt-{}'.format(cluster_info['volume']['volume_name'], 6)
    rep1_charge_key = 'trusted.afr.{}-vclnt-{}'.format(cluster_info['volume']['volume_name'], 7)
    rep2_charge_key = 'trusted.afr.{}-vclnt-{}'.format(cluster_info['volume']['volume_name'], 8)
    arbiter0_charge_key = 'trusted.afr.{}-vclnt-{}'.format(cluster_info['volume']['volume_name'], 9)
    arbiter1_charge_key = 'trusted.afr.{}-vclnt-{}'.format(cluster_info['volume']['volume_name'], 10)
    bad_key = 'trusted.file_status'
    nobad_value = '0x000000'
    for rep_pathinfo in shard_pathinfo:
        brick_no = rep_pathinfo['brick_no']
        mybrick = bricks[brick_no]
        if len(rep_pathinfo) == 1:
            mark_shard_rep_damage_by_brickno(myshard, brick_no, REP_DAMAGE_TYPE_DAMAGE)
            continue
        if 'breakdown_type' in mybrick and mybrick['breakdown_type'] == BREAKDOWN_TYPE_CACHE:
            if tier_status not in rep_pathinfo or rep_pathinfo[tier_status] != normal_tier_value:
                mark_shard_rep_damage_by_brickno(myshard, brick_no, REP_DAMAGE_TYPE_DAMAGE)
            if wcache_status not in rep_pathinfo or rep_pathinfo[wcache_status] != normal_tier_value:
                mark_shard_rep_damage_by_brickno(myshard, brick_no, REP_DAMAGE_TYPE_DAMAGE)
        if myshard['rep0_damage'] == REP_DAMAGE_TYPE_NORMAL and rep0_charge_key in rep_pathinfo \
                and rep_pathinfo[rep0_charge_key] != normal_charge_value:
            myshard['rep0_damage'] = REP_DAMAGE_TYPE_CHARGE
        if myshard['rep1_damage'] == REP_DAMAGE_TYPE_NORMAL and rep1_charge_key in rep_pathinfo \
                and rep_pathinfo[rep1_charge_key] != normal_charge_value:
            myshard['rep1_damage'] = REP_DAMAGE_TYPE_CHARGE
        if myshard['rep2_damage'] == REP_DAMAGE_TYPE_NORMAL and rep2_charge_key in rep_pathinfo \
                and rep_pathinfo[rep2_charge_key] != normal_charge_value:
            myshard['rep2_damage'] = REP_DAMAGE_TYPE_CHARGE
        if myshard['arbiter0_damage'] == REP_DAMAGE_TYPE_NORMAL and arbiter0_charge_key in rep_pathinfo \
                and rep_pathinfo[arbiter0_charge_key] != normal_charge_value:
            myshard['arbiter0_damage'] = REP_DAMAGE_TYPE_CHARGE
        if myshard['arbiter1_damage'] == REP_DAMAGE_TYPE_NORMAL and arbiter1_charge_key in rep_pathinfo \
                and rep_pathinfo[arbiter1_charge_key] != normal_charge_value:
            myshard['arbiter1_damage'] = REP_DAMAGE_TYPE_CHARGE
        if bad_key in rep_pathinfo and rep_pathinfo[bad_key] != nobad_value:
            mark_shard_rep_damage_by_brickno(myshard, brick_no, REP_DAMAGE_TYPE_BAD)


def mark_shard_damage_level(cluster_info, myfile, rep_class, myshard, breakdown_bricks):
    if myshard['rep0'] in breakdown_bricks and breakdown_bricks[myshard['rep0']]['breakdown_type'] != BREAKDOWN_TYPE_NORMAL:
        myshard['rep0_damage'] = REP_DAMAGE_TYPE_DAMAGE
    if myshard['rep1'] in breakdown_bricks and breakdown_bricks[myshard['rep1']]['breakdown_type'] != BREAKDOWN_TYPE_NORMAL:
        myshard['rep1_damage'] = REP_DAMAGE_TYPE_DAMAGE
    if myshard['arbiter0'] in breakdown_bricks and breakdown_bricks[myshard['arbiter0']]['breakdown_type'] != BREAKDOWN_TYPE_NORMAL:
        myshard['arbiter0_damage'] = REP_DAMAGE_TYPE_DAMAGE
    if rep_class == REPLICATE_CLASS_3X_3REP_2ARBITER:
        rep2_bt = BREAKDOWN_TYPE_NORMAL
        if myshard['rep2'] in breakdown_bricks and breakdown_bricks[myshard['rep2']]['breakdown_type'] != BREAKDOWN_TYPE_NORMAL:
            myshard['rep2_damage'] = REP_DAMAGE_TYPE_DAMAGE
        if myshard['arbiter1'] in breakdown_bricks \
                and breakdown_bricks[myshard['arbiter1']]['breakdown_type'] != BREAKDOWN_TYPE_NORMAL:
            myshard['arbiter1_damage'] = REP_DAMAGE_TYPE_DAMAGE
    elif rep_class != REPLICATE_CLASS_3X_2REP_ARBITER:
        raise Exception('Unsupport replicate class: {}'.format(rep_class))
    if is_shard_damage_A_3x(myshard):
        myshard['damage_level'] = DAMAGE_LEVEL_A
    elif is_shard_damage_B_3x(myshard):
        myshard['damage_level'] = DAMAGE_LEVEL_B
    elif is_shard_damage_C_3x(myshard):
        myshard['damage_level'] = DAMAGE_LEVEL_C
    elif is_shard_damage_D_3x(myshard):
        myshard['damage_level'] = DAMAGE_LEVEL_D
    else:
        myshard['damage_level'] = DAMAGE_LEVEL_NORMAL


def count_file_damage_level(myfile):
    A_count = 0
    B_count = 0
    C_count = 0
    D_count = 0
    NORMAL_count = 0
    A_size = 0
    B_size = 0
    C_size = 0
    D_size = 0
    NORMAL_size = 0
    file_shards = myfile['shards']
    for shard_id in file_shards:
        if shard_id < 0 and len(file_shards) > 1:
            continue
        myshard = file_shards[shard_id]
        damage_level = myshard['damage_level']
        if damage_level == DAMAGE_LEVEL_A:
            if myshard['size'] < 0 or A_size < 0:
                A_size = -1
            else:
                A_size += int(myshard['size'])
            A_count += 1
        elif damage_level == DAMAGE_LEVEL_B:
            if myshard['size'] < 0 or B_size < 0:
                B_size = -1
            else:
                B_size += int(myshard['size'])
            B_count += 1
        elif damage_level == DAMAGE_LEVEL_C:
            if myshard['size'] < 0 or C_size < 0:
                C_size = -1
            else:
                C_size += int(myshard['size'])
            C_count += 1
        elif damage_level == DAMAGE_LEVEL_D:
            if myshard['size'] < 0 or D_size < 0:
                D_size = -1
            else:
                D_size += int(myshard['size'])
            D_count += 1
        else:
            if myshard['size'] < 0 or NORMAL_size < 0:
                NORMAL_size = -1
            else:
                NORMAL_size += myshard['size']
            NORMAL_count += 1
    if A_count > 0:
        myfile['damage_level'] = DAMAGE_LEVEL_A
    elif B_count > 0:
        myfile['damage_level'] = DAMAGE_LEVEL_B
    elif C_count > 0:
        myfile['damage_level'] = DAMAGE_LEVEL_C
    elif D_count > 0:
        myfile['damage_level'] = DAMAGE_LEVEL_D
    else:
        myfile['damage_level'] = DAMAGE_LEVEL_NORMAL
    myfile['status']['A_count'] = A_count
    myfile['status']['A_size'] = A_size
    myfile['status']['B_count'] = B_count
    myfile['status']['B_size'] = B_size
    myfile['status']['C_count'] = C_count
    myfile['status']['C_size'] = C_size
    myfile['status']['D_count'] = D_count
    myfile['status']['D_size'] = D_size
    myfile['status']['NORMAL_count'] = NORMAL_count
    myfile['status']['NORMAL_size'] = NORMAL_size


class fattrThread (threading.Thread):   #继承父类threading.Thread
    def __init__(self, threadID, wait_list, finish_list):
        threading.Thread.__init__(self)
        self.threadID = threadID
        self.wait_list = wait_list
        self.finish_list = finish_list

    def run(self):
        while len(self.wait_list) > 0:
            task = self.wait_list[0]
            del(self.wait_list[0])
            cmdline = '/sf/vs/bin/vs_rpc_tool --cmd file_pathinfo --file {}'.format(task['shard_gfid'])
            task['fattr'] = cli(cmdline, True)
            self.finish_list.append(task)


def check_shard_fattr_3x(cluster_info, thread_num = 20):
    shards = cluster_info['volume']['shards']
    gfid_list = shards.keys()
    total_shard_num = 0
    finish_shard_num = 0
    shard_wait_list = []
    shard_finish_list = []
    for gfid in gfid_list:
        total_shard_num += len(shards[gfid]['shards'])
        myfile = shards[gfid]
        rep_class = myfile['rep_class']
        for shard_id in myfile['shards']:
            shard_wait_list.append({
                'gfid': gfid,
                'shard_id': shard_id,
                'shard_gfid': myfile['shards'][shard_id]['shard_gfid']
            })
    threads = []
    for i in range(0, thread_num):
        mythread = fattrThread(i, shard_wait_list, shard_finish_list)
        threads.append(mythread)
        mythread.start()
    file_progress_output = False
    back_len = 0
    print('\t扫描分片的指控信息')
    t1 = time.time()
    while finish_shard_num < total_shard_num:
        if len(shard_finish_list) > 0:
            shard_charge_info = shard_finish_list[0]
            del(shard_finish_list[0])
            myfile = shards[shard_charge_info['gfid']]
            myshard = myfile['shards'][shard_charge_info['shard_id']]
            rep_class = myfile['rep_class']
            fattr_lines = shard_charge_info['fattr']
            shard_pathinfo = parse_shard_pathinfo_3x(cluster_info, fattr_lines)
            if rep_class == REPLICATE_CLASS_3X_2REP_ARBITER:
                check_shard_damage_full_2rep_arbiter(cluster_info, myshard, shard_pathinfo)
                myshard['check_full'] = True
            elif rep_class == REPLICATE_CLASS_3X_3REP_2ARBITER:
                check_shard_damage_full_3rep_2arbiter(cluster_info, myshard, shard_pathinfo)
                myshard['check_full'] = True
            else:
                logger.error('Invalid rep class: {}'.format(rep_class))
            finish_shard_num += 1
            t2 = time.time()
            if (t2 - t1) > 1 or finish_shard_num == total_shard_num:
                if not file_progress_output:
                    cnt_str = str(finish_shard_num)
                    back_len = len(cnt_str)
                    sys.stdout.write('\t共{}个分片，已完成：{}'.format(total_shard_num, cnt_str))
                    file_progress_output = True
                else:
                    for i in range(0, back_len):
                        sys.stdout.write('\b')
                    cnt_str = str(finish_shard_num)
                    back_len = len(cnt_str)
                    sys.stdout.write(cnt_str)
                sys.stdout.flush()
                t1 = t2
    sys.stdout.write('\t\t\t{}\n'.format(DONE_STR))


def mark_damage_level_3x(cluster_info):
    shards = cluster_info['volume']['shards']
    gfid_list = shards.keys()
    total_num = len(gfid_list)
    total_shard_num = 0
    finish_shard_num = 0

    # 检查 'breakdown_disks' 是否在 'volume' 中
    if 'breakdown_disks' in cluster_info['volume']:
        bricks = cluster_info['volume']['breakdown_disks']['bricks']
    else:
        bricks = cluster_info['volume']['bricks']

    print('\t给分片打上损坏标记')
    for gfid in gfid_list:
        total_shard_num += len(shards[gfid]['shards'])
    file_progress_output = False
    back_len = 0
    for gfid in gfid_list:
        myfile = shards[gfid]
        rep_class = myfile['rep_class']
        for shard_id in myfile['shards']:
            myshard = myfile['shards'][shard_id]
            mark_shard_damage_level(cluster_info, myfile, rep_class, myshard, bricks)
            finish_shard_num += 1
            if not file_progress_output:
                cnt_str = str(finish_shard_num)
                back_len = len(cnt_str)
                sys.stdout.write('\t共{}个分片，已完成：{}'.format(total_shard_num, cnt_str))
                file_progress_output = True
            else:
                for i in range(0, back_len):
                    sys.stdout.write('\b')
                cnt_str = str(finish_shard_num)
                back_len = len(cnt_str)
                sys.stdout.write(cnt_str)
            sys.stdout.flush()
        count_file_damage_level(myfile)
        damage_level = myfile['damage_level']
    sys.stdout.write('\t\t\t{}\n'.format(DONE_STR))


def get_vmname_by_pathname(cluster_info, pathname_nodes):
    vm_list = cluster_info['volume']['vm_list']
    vmkey = pathname_nodes[-2].split('.')[0]
    for vm in vm_list:
        if vm[0] == vmkey or vm[1] == vmkey:
            return vm[1]
    return vmkey


def get_vmname_of_backup(cluster_info, pathname_nodes):
    vm_list = cluster_info['volume']['vm_list']
    vmkey = pathname_nodes[-2]
    for vm in vm_list:
        if vm[0] == vmkey or vm[1] == vmkey:
            return vm[1]
    return vmkey


def get_file_type_str(cluster_info, pathname):
    nodes = pathname.split('/')
    ext_name = nodes[-1].split('.')[-1]
    if nodes[1] == 'images' and nodes[2] == 'cluster':
        vmname = get_vmname_by_pathname(cluster_info, nodes)
        if ext_name == 'qcow2':
            return '虚拟机镜像({})'.format(vmname)
        elif ext_name == 'conf':
            return '虚拟机配置({})'.format(vmname)
        else:
            return '虚拟机文件({})'.format(vmname)
    elif nodes[1] == 'iscsi':
        return 'iSCSI磁盘'
    elif nodes[1] == 'backup':
        vmname = get_vmname_of_backup(cluster_info, nodes)
        if ext_name == 'qcow2':
            return '备份镜像(虚拟机：{})'.format(vmname)
        return '备份文件(虚拟机：{})'.format(vmname)
    elif nodes[1] == 'iso':
        return 'iso镜像'
    elif nodes[1] == 'template':
        ext_name1 = nodes[-1].split('.')[-2]
        vmname = get_vmname_of_backup(cluster_info, nodes)
        if ext_name1 == 'qcow2':
            return '模板虚拟机镜像({})'.format(vmname)
        elif ext_name == 'conf':
            return '模板虚拟机配置({})'.format(vmname)
        else:
            return '模板虚拟机文件({})'.format(vmname)
    else:
        return '其他文件'


def get_pathname_by_gfid_meta(meta_brick, localhost, gfid):
    meta_dir = '/sf/data/vs/local/{}/{}'.format(meta_brick['vg_name'], meta_brick['brick_dir'])
    if meta_brick['host'] == localhost:
        find_cmd = 'find "{}" -type f -samefile "{}/.glusterfs/{}/{}/{}"'
        cmdline = find_cmd.format(meta_dir, meta_dir, gfid[0:2], gfid[2:4], gfid)
    else:
        find_cmd = 'find \\\"{}\\\" -type f -samefile \\\"{}/.glusterfs/{}/{}/{}\\\"'
        find_cmd = find_cmd.format(meta_dir, meta_dir, gfid[0:2], gfid[2:4], gfid)
        cmdline = 'ssh root@{} "{}"'.format(meta_brick['host'], find_cmd)
    lines = cli(cmdline, True)
    for line in lines:
        nodes = line.split('/')
        if len(nodes) > 7 and nodes[7] != '.glusterfs':
            nodes1 = nodes[7:]
            return '/' + '/'.join(nodes1)
    return ''


def get_pathname_by_gfid(meta_bricks, localhost, gfid):
    for mybrick in meta_bricks:
        pathname = get_pathname_by_gfid_meta(mybrick, localhost, gfid)
        if pathname:
            return pathname
    return ''


def get_damage_level_str():
    s1 = '----------------------------- 损坏级别定义 -------------------------------\n'
    s2 = 'A级：所有数据副本都已彻底损坏，无法恢复。 一个副本已损坏，另一个副本在修复中，也处于损坏状态，也归入A级。\n'
    s3 = 'B级：所有数据副本都有问题，有的数据副本已彻底损坏，另外的数据副本被指控或者有BAD，可能也已损坏\n'
    s4 = 'C级：有超过半数的副本和仲裁损坏，分片已不可访问，但是至少有一个数据副本是正常的\n'
    s5 = 'D级：有副本或仲裁损坏，但是少于半数，分片可正常访问\n\n\n'
    return s1 + s2 + s3 + s4 + s5


def damage_level_2_string(level):
    if level == DAMAGE_LEVEL_A:
        return 'A级'
    elif level == DAMAGE_LEVEL_B:
        return 'B级'
    elif level == DAMAGE_LEVEL_C:
        return 'C级'
    elif level == DAMAGE_LEVEL_D:
        return 'D级'
    elif level == DAMAGE_LEVEL_NORMAL:
        return '正常'


def get_rep_damage_type_str(damage_type):
    if damage_type == REP_DAMAGE_TYPE_DAMAGE:
        return '已损坏'
    if damage_type == REP_DAMAGE_TYPE_BAD:
        return 'BAD'
    if damage_type == REP_DAMAGE_TYPE_CHARGE:
        return '被指控'
    return ''


def output_file_damage_result_3x_2rep_arbiter(cluster_info, all_bricks, myfile, result_file):
    header_fmt = '{:8}\t{:8}\t{:12}\t{:36}\t{:36}\t{:36}\t{}\n'
    header = header_fmt.format('分片编号', '分片大小', '损坏级别', '0号副本', '1号副本', '仲裁', '分片gfid')
    result_file.write(header)
    fmt_str = '  {:6}\t{:8}\t{:12}\t{:36}\t{:36}\t{:36}\t{}\n'
    for shard_id in myfile['shards']:
        if shard_id < 0 and len(myfile['shards']) > 1:
            continue
        myshard = myfile['shards'][shard_id]
        damage_str = damage_level_2_string(myshard['damage_level'])
        rep0_loc = all_bricks[myshard['rep0']]['brick_name']
        rep1_loc = all_bricks[myshard['rep1']]['brick_name']
        arbiter0_loc = all_bricks[myshard['arbiter0']]['brick_name']
        if myshard['rep0_damage'] != REP_DAMAGE_TYPE_NORMAL:
            rep0_loc = '{}({})'.format(rep0_loc, get_rep_damage_type_str(myshard['rep0_damage']))
        if myshard['rep1_damage'] != REP_DAMAGE_TYPE_NORMAL:
            rep1_loc = '{}({})'.format(rep1_loc, get_rep_damage_type_str(myshard['rep1_damage']))
        if myshard['arbiter0_damage'] != REP_DAMAGE_TYPE_NORMAL:
            arbiter0_loc = '{}({})'.format(arbiter0_loc, get_rep_damage_type_str(myshard['arbiter0_damage']))
        size_str = get_human_size_string(myshard['size'])
        shard_gfid = myshard['shard_gfid']
        file_result = fmt_str.format(str(shard_id), size_str, damage_str, rep0_loc, rep1_loc, arbiter0_loc, shard_gfid)
        result_file.write(file_result)


def output_file_damage_result_3x_3rep_2arbiter(cluster_info, all_bricks, myfile, result_file):
    header_fmt = '{:8}\t{:8}\t{:12}\t{:36}\t{:36}\t{:36}\t{:36}\t{:36}\t{}\n'
    header = header_fmt.format('分片编号', '分片大小', '损坏级别', '0号副本', '1号副本', '2号副本', '0号仲裁', '1号仲裁', '分片gfid')
    result_file.write(header)
    fmt_str = '  {:6}\t{:8}\t{:12}\t{:36}\t{:36}\t{:36}\t{:36}\t{:36}\t{}\n'
    for shard_id in myfile['shards']:
        if shard_id < 0 and len(myfile['shards']) > 1:
            continue
        myshard = myfile['shards'][shard_id]
        damage_str = damage_level_2_string(myshard['damage_level'])
        rep0_loc = all_bricks[myshard['rep0']]['brick_name']
        rep1_loc = all_bricks[myshard['rep1']]['brick_name']
        rep2_loc = all_bricks[myshard['rep2']]['brick_name']
        arbiter0_loc = all_bricks[myshard['arbiter0']]['brick_name']
        arbiter1_loc = all_bricks[myshard['arbiter1']]['brick_name']
        if myshard['rep0_damage'] != REP_DAMAGE_TYPE_NORMAL:
            rep0_loc = '{}({})'.format(rep0_loc, get_rep_damage_type_str(myshard['rep0_damage']))
        if myshard['rep1_damage'] != REP_DAMAGE_TYPE_NORMAL:
            rep1_loc = '{}({})'.format(rep1_loc, get_rep_damage_type_str(myshard['rep1_damage']))
        if myshard['rep2_damage'] != REP_DAMAGE_TYPE_NORMAL:
            rep2_loc = '{}({})'.format(rep2_loc, get_rep_damage_type_str(myshard['rep2_damage']))
        if myshard['arbiter0_damage'] != REP_DAMAGE_TYPE_NORMAL:
            arbiter0_loc = '{}({})'.format(arbiter0_loc, get_rep_damage_type_str(myshard['arbiter0_damage']))
        if myshard['arbiter1_damage'] != REP_DAMAGE_TYPE_NORMAL:
            arbiter1_loc = '{}({})'.format(arbiter1_loc, get_rep_damage_type_str(myshard['arbiter1_damage']))
        size_str = get_human_size_string(myshard['size'])
        shard_gfid = myshard['shard_gfid']
        file_result = fmt_str.format(str(shard_id), size_str, damage_str, rep0_loc,
                                     rep1_loc, rep2_loc, arbiter0_loc, arbiter1_loc, shard_gfid)
        result_file.write(file_result)


def output_single_file_result_3x(cluster_info, meta_bricks, myfile, result_pathname):
    if myfile['file_type'] == FILE_TYPE_SHARD_FILE:
        myfile['pathname'] = get_pathname_by_gfid(meta_bricks, cluster_info['localhost'], myfile['gfid'])
    myfile['type'] = get_file_type_str(cluster_info, myfile['pathname'])
    result_file = open(result_pathname, "w+")
    damage_level_str = get_damage_level_str()
    result_file.write(damage_level_str)
    all_bricks = cluster_info['volume']['bricks']
    file_base_info_str = '----------------------------- 文件基本信息 -------------------------------\n' \
                         'gfid：{}\n文件：{}\n文件类型：{}\n\n'.format(myfile['gfid'], myfile['pathname'], myfile['type'])
    result_file.write(file_base_info_str)
    file_status = myfile['status']
    seg1_str = '----------------------------- 分片损坏汇总 -------------------------------\n'
    A_str = 'A级分片：{}个分片/{}\n'.format(file_status['A_count'], get_human_size_string(file_status['A_size']))
    B_str = 'B级分片：{}个分片/{}\n'.format(file_status['B_count'], get_human_size_string(file_status['B_size']))
    C_str = 'C级分片：{}个分片/{}\n'.format(file_status['C_count'], get_human_size_string(file_status['C_size']))
    D_str = 'D级分片：{}个分片/{}\n'.format(file_status['D_count'], get_human_size_string(file_status['D_size']))
    Normal_str = '正常分片：{}个分片/{}\n\n'.format(file_status['NORMAL_count'], get_human_size_string(file_status['NORMAL_size']))
    seg2_str = '----------------------------- 文件分片列表 --------------------------------\n'
    result_file.write(seg1_str + A_str + B_str + C_str + D_str + Normal_str + seg2_str)
    if myfile['rep_class'] == REPLICATE_CLASS_3X_2REP_ARBITER:
        output_file_damage_result_3x_2rep_arbiter(cluster_info, all_bricks, myfile, result_file)
    elif myfile['rep_class'] == REPLICATE_CLASS_3X_3REP_2ARBITER:
        output_file_damage_result_3x_3rep_2arbiter(cluster_info, all_bricks, myfile, result_file)
    else:
        raise Exception('Unsupported replicate class')
    result_file.close()


def get_file_damage_summary_info(myfile, output_dir):
    gfid = myfile['gfid']
    damage_str = damage_level_2_string(myfile['damage_level'])
    A_str = '{}/{}'.format(myfile['status']['A_count'], get_human_size_string(myfile['status']['A_size']))
    B_str = '{}/{}'.format(myfile['status']['B_count'], get_human_size_string(myfile['status']['B_size']))
    C_str = '{}/{}'.format(myfile['status']['C_count'], get_human_size_string(myfile['status']['C_size']))
    D_str = '{}/{}'.format(myfile['status']['D_count'], get_human_size_string(myfile['status']['D_size']))
    type_str = myfile['type']
    result_file = '{}/{}.txt'.format(output_dir, gfid)
    return SUMMARY_FMT.format(damage_str, A_str, B_str, C_str, D_str, type_str, result_file)


def get_file_damage_summary_info_csv(myfile, output_dir):
    damage_str = damage_level_2_string(myfile['damage_level'])
    A_cnt = myfile['status']['A_count']
    A_size = get_human_size_string(myfile['status']['A_size'])
    B_cnt = myfile['status']['B_count']
    B_size = get_human_size_string(myfile['status']['B_size'])
    C_cnt = myfile['status']['C_count']
    C_size = get_human_size_string(myfile['status']['C_size'])
    D_cnt = myfile['status']['D_count']
    D_size = get_human_size_string(myfile['status']['D_size'])
    type_str = myfile['type']
    result_file = '{}/{}.txt'.format(output_dir, myfile['gfid'])
    return '{},{},{},{},{},{},{},{},{},{},{},{}\n'.format(damage_str, A_cnt, A_size, B_cnt, B_size, C_cnt, C_size,
                                                    D_cnt, D_size, type_str, myfile['pathname'], result_file)


def output_all_files_results_by_level_3x(cluster_info, output_dir, summary_file, output_info, damage_level):
    cnt = output_info['cur_index']
    shards = cluster_info['volume']['shards']
    all_gfid = shards.keys()
    meta_bricks = output_info['meta_bricks']
    total = output_info['total']
    summary_csv = ''
    for gfid in all_gfid:
        myfile = shards[gfid]
        if myfile['damage_level'] == damage_level:
            result_pathname = '{}/{}.txt'.format(output_dir, gfid)
            cnt += 1
            output_single_file_result_3x(cluster_info, meta_bricks, myfile, result_pathname)
            output_info['summary_info'] += get_file_damage_summary_info(myfile, output_dir)
            summary_csv += get_file_damage_summary_info_csv(myfile, output_dir)
            print('\t输出第{}个文件的结果/共{}个文件：{}   {}'.format(cnt, total, result_pathname, color.green('[OK]')))
            #try:
            #except Exception as e:
            #    logger.error('output result of file {} failed, error: {}'.format(result_pathname, str(e)))
            #    print('\t输出第{}个文件的结果/共{}个文件：{}   {}'.format(cnt, total, result_pathname, color.red('[failed]')))
    output_info['cur_index'] = cnt
    summary_file.write(summary_csv)


def get_all_meta_bricks(cluster_info):
    meta_bricks = []
    all_bricks = cluster_info['volume']['bricks']
    localhost = cluster_info['localhost']
    for brick_no in all_bricks:
        mybrick = all_bricks[brick_no]
        if (mybrick['brick_type'] == BRICK_TYPE_META or mybrick['brick_type'] == BRICK_TYPE_META_ARBITER) \
                and mybrick['host'] == cluster_info['localhost']:
            meta_bricks.append(mybrick)
    for brick_no in all_bricks:
        mybrick = all_bricks[brick_no]
        if (mybrick['brick_type'] == BRICK_TYPE_META or mybrick['brick_type'] == BRICK_TYPE_META_ARBITER) \
                and mybrick['host'] != cluster_info['localhost']:
            meta_bricks.append(mybrick)
    return meta_bricks


def output_all_files_results_3x(cluster_info, output_dir, summary_file, output_all_files):
    if output_all_files:
        total = len(cluster_info['volume']['shards'])
    else:
        total = 0
        shards = cluster_info['volume']['shards']
        for gfid in shards:
            damage_level = shards[gfid]['damage_level']
            if damage_level == DAMAGE_LEVEL_A or damage_level == DAMAGE_LEVEL_B or damage_level == DAMAGE_LEVEL_C:
                total += 1
    output_info = {
        'total': total,
        'cur_index': 0,
        'meta_bricks': get_all_meta_bricks(cluster_info),
        'summary_info': ''
    }
    # 按照文件损坏级别从高到低依次输出结果
    output_all_files_results_by_level_3x(cluster_info, output_dir+'/A', summary_file, output_info, DAMAGE_LEVEL_A)
    output_all_files_results_by_level_3x(cluster_info, output_dir+'/B', summary_file, output_info, DAMAGE_LEVEL_B)
    output_all_files_results_by_level_3x(cluster_info, output_dir+'/C', summary_file, output_info, DAMAGE_LEVEL_C)
    if output_all_files:
        output_all_files_results_by_level_3x(cluster_info, output_dir+'/D', summary_file, output_info, DAMAGE_LEVEL_D)
        output_all_files_results_by_level_3x(cluster_info, output_dir+'/N', summary_file, output_info, DAMAGE_LEVEL_NORMAL)
    return output_info['summary_info']


def get_vm_info(cluster_info):
    vm_list = cluster_info['volume']['vm_list']
    cmdline = 'qm-c list'
    lines = cli(cmdline, True)
    for line in lines:
        str_list = line.split()
        try:
            vmid = int(str_list[0])
        except:
            # 如果第一列不是vmid，就不能转换成int，这一行就不是虚拟机信息
            continue
        vm_list.append([str_list[0], str_list[1]])



def output_damage_result_3x(cluster_info, output_dir, output_all_files=False):
    get_vm_info(cluster_info)
    print '\t获取虚拟机信息    {}'.format(color.green('[OK]'))
    if not output_dir:
        output_dir = result
    damage_level_str = get_damage_level_str()
    summary_pathname = output_dir + '/summary.csv'
    summary_file = open(summary_pathname, "w+")
    summary_file.write(damage_level_str)
    summary_file.write('------------------------ 虚拟存储损坏结果汇总信息 -------------------------------\n')
    summary_file.write('损坏级别,A级分片数,A级分片大小,B级分片数,B级分片大小,C级分片数,C级分片大小,D级分片数,D级分片大小,文件类型,文件路径,结果输出\n')
    summary_info = output_all_files_results_3x(cluster_info, output_dir, summary_file, output_all_files)
    summary_file.close()
    print('\t已输出结果汇总：{}   {}'.format(summary_pathname, color.green('[OK]')))
    summary_str = damage_level_str
    summary_str += '------------------------ 虚拟存储损坏结果汇总信息 -------------------------------\n'
    summary_header = SUMMARY_HEADER_FMT.format('损坏级别','A级(分片数/数据量)','B级(分片数/数据量)', 'C级(分片数/数据量)', 'D级(分片数/数据量)', '文件类型', '结果输出')
    summary_str += summary_header
    summary_str += summary_info
    summary_pathname = output_dir + '/summary.txt'
    summary_file = open(summary_pathname, "w+")
    summary_file.write(summary_str)
    summary_file.close()
    return summary_str


def faultshow(breakdown_disks, breakdown_host, scan_allbricks=False):
    output_dir = result
    check_output_results_dir(output_dir)

    t1 = time.time()
    print('第一步：收集存储卷信息')
    cluster_info = get_cluster_info()
    if breakdown_disks:
        parse_breakdown_disks(cluster_info, breakdown_disks)
    if breakdown_host:
        parse_breakdown_hosts(cluster_info, breakdown_host)

    t2 = time.time()
    print('\t花费时间：{:.3f}秒'.format(t2-t1))

    print('第二步：扫描brick的分片信息...')
    if scan_allbricks:
        scan_shards_of_bricks_3x(cluster_info, cluster_info['volume']['bricks'])
    else:
        scan_shards_of_bricks_3x(cluster_info, cluster_info['volume']['breakdown_disks']['bricks'])
    t3 = time.time()
    print('\t共扫描到{}个分片，{}个文件，花费时间：{:.3f}秒'.format(cluster_info['volume']['shards_cnt'],
                                                  len(cluster_info['volume']['shards']), t3-t2))

    print('第三步：识别损坏分片...')
    check_shard_fattr_3x(cluster_info, 20)
    mark_damage_level_3x(cluster_info)
    t4 = time.time()
    print('\t花费时间：{:.3f}秒'.format(t4-t3))

    print('第四步：输出结果...')
    summary_str = output_damage_result_3x(cluster_info, output_dir, False)
    t5 = time.time()
    print('\t花费时间：{:.3f}秒\n\n'.format(t5 - t4))
    print(summary_str)
    print('总共花费时间：{:.3f}秒'.format(t5 - t1))

    return
