#!/sf/vs/bin/python
# -*- coding:utf-8 -*-

"""
## 资源清理工具
"""

import os
import json
import pylib.utils.utiltools as common


def check_if_tier_file_can_fix(brick_path, gfid, bricks):
    if common.fault_point_result():
        raise common.CmdError('common.fault_point_result')
    cmdline = '/sf/vs/bin/vs_tier_cli.py -c dump -a inode'
    result = common.cli(cmdline, False)
    data = json.loads(result)
    for ssd in data.get('ssd'):
        for brick in ssd.get('brick'):
            if brick.get('bi_brickid') != brick_path:
                continue
            for inode in brick.get('inodes'):
                if gfid == inode.get('uuid'):
                    gfid_path = '{}/.glusterfs/{}/{}/{}'.format(brick_path, gfid[0:2], gfid[2:4], gfid)
                    stat_info = os.stat(gfid_path)
                    if stat_info.st_nlink == 1:
                        # 只支持处理gfid_path存在且硬链接数为1的场景
                        common.logger.error('gfid_path: {} nlink: {}'.format(gfid_path, stat_info.st_nlink))
                        return True

    return False


def get_repid_from_brickpath(brick_path, replicate):
    for repid, bricks in replicate.items():
        for brick in bricks:
            if brick_path == brick['path']:
                return repid
    return -1


def handle_tier_residual_file(brick_path, gfid, replicate):
    repid = get_repid_from_brickpath(brick_path, replicate)
    if repid < 0:
        common.logger.error('failed to get repid from brickpath: {}'.format(brick_path))
        return False

    last_real_path = None
    # 标记远端是否存在gfid_path, 如果不存在，认为副本不存在
    remote_gfid_path_count = 0
    bricks = replicate.get(repid)
    for brick in bricks:
        host = brick['host']
        brickpath = brick['path']

        # 当前主机不获取real_path
        if brick_path == brickpath:
            continue

        real_path = None
        gfid_path = '{}/.glusterfs/{}/{}/{}'.format(brickpath, gfid[0:2], gfid[2:4], gfid)

        try:
            if common.fault_point_result():
                raise common.CmdError('common.fault_point_result')
            # 检查远端gfid_path是否存在
            cmdline = '/bin/ls -al {}'.format(gfid_path)
            common.remote_cli(host, cmdline, False)
            remote_gfid_path_count += 1

            # 如果有gfid_path存在，转化成real_path
            cmdline = '/usr/bin/find {} -samefile {} -not -path */.glusterfs/*'.format(brickpath, gfid_path)
            result = common.remote_cli(host, cmdline, False)
            if result:
                real_path = result.replace(brickpath, '').strip()
                common.logger.info('host: {}, gfid_path: {}, real_path: {}'.format(host, gfid_path, real_path))
        except common.CmdError as e:
            if 'No such file or directory' in str(e):
                continue
            else:
                raise

        if last_real_path and real_path and last_real_path != real_path:
            # 如果多个副本找到real_path，必须相同，否则，不支持处理
            common.logger.error('host: {}, gfid_path: {}, real_path: {}, last_real_path: {}'.
                                format(host, gfid_path, real_path, last_real_path))
            return False

        if real_path:
            last_real_path = real_path

    # 处理当前GFID path
    gfid_path = '{}/.glusterfs/{}/{}/{}'.format(brick_path, gfid[0:2], gfid[2:4], gfid)
    if remote_gfid_path_count == 0:
        # 远程副本找到不到gfid_path,说明当前副本的gfid_path应该是残留的，可以直接删除
        # logger.info('try to remove gfid_path: {}'.format(gfid_path))
        # os.remove(gfid_path)
        readline = '请确认文件: {}，是否可以删除，输入\'y\'确认删除，\'n\'取消退出'.format(gfid_path)
        common.check_terminal_input(readline)

        landfill_path = '{}/.glusterfs/landfill/{}'.format(brick_path, gfid)
        common.logger.info('try to link gfid_path: {}, local_path: {}'.format(gfid_path, landfill_path))
        os.link(gfid_path, landfill_path)

        # 设置垃圾回收标记
        cmdline = '/sf/vs/bin/setfattr -n user.glusterfs.rubbish_allow_delete -v 1 {}'.format(landfill_path)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline, False)
    elif remote_gfid_path_count >= 1 and last_real_path:
        # 远程副本gfid_path都存在，并且找到了real_path, 则需要link创建
        local_path = '{}{}'.format(brick_path, last_real_path)
        common.logger.info('try to link gfid_path: {}, local_path: {}'.format(gfid_path, local_path))
        os.link(gfid_path, local_path)
    else:
        common.logger.error('failed to fix remote_gfid_path_count: {}, last_real_path: {}'.
                            format(remote_gfid_path_count, last_real_path))
        return False

    return True


def tier_file_clean(brick_path, gfid):
    version = common.get_vs_version()
    if not common.is_vs_version_valid(version):
        return -1

    if version >= common.VS_VERSION_3_0:
        # 当前只支持vs2.6/vs2.8
        common.logger.error('failed to supported, version: {}'.format(version))
        return -1

    if not brick_path.startswith('/'):
        common.logger.error('failed to supported, brick_path: {} is not invalid'.format(brick_path))
        return -1

    if brick_path.endswith('/'):
        # 如果brick_path以斜杠结尾，去掉
        brick_path = brick_path[:-1]

    if not isinstance(gfid, str) or not common.is_valid_uuid(gfid):
        # filename表示文件路径，一定是'/'开头
        common.logger.error('failed to supported, gfid: {} is not uuid'.format(gfid))
        return -1

    volume_name, hosts, replicate_num, has_arbiter, bricks, replicate = common.get_vs_cluster_info()
    if not volume_name:
        common.logger.error('failed to supported, cannot find volume name')
        return -1

    online_bricks = common.get_online_bricks(volume_name)
    if common.fault_point_result() or not online_bricks:
        common.logger.error('failed to supported, cannot get online_bricks')
        return -1

    if online_bricks.get(brick_path) != 'y':
        common.logger.error('failed to supported, brick_path: {} is not online'.format(brick_path))
        return -1

    # 确认GFID在分层中存在，但是gfid_path链接数为1
    result = check_if_tier_file_can_fix(brick_path, gfid, bricks)
    if not result:
        common.logger.error('brick_path: {}, gfid: {}, failed to check tier, maybe normal'.format(brick_path, gfid))
        return -1

    result = handle_tier_residual_file(brick_path, gfid, replicate)
    if not result:
        common.logger.error('brick_path: {}, gfid: {}, failed to handle tier'.format(brick_path, gfid))
        return -1

    return 0


def _tier_file_clean(brick_path, gfid):
    lock_file = common.get_vsfire_lock_file()
    with common.VsfireFlock(lock_file) as lock:
        ret = tier_file_clean(brick_path, gfid)
        if ret:
            print common.Colored().red('执行失败')
        else:
            print common.Colored().cyan('执行成功')
        return ret
