#!/bin/bash
# 
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
#
# 功能: 处理非qcow2文件gfid脑裂，保留最新的文件
# 用法：集群中任一主机上执行 
#      ./vsts-fix-brian-split-file.sh   /sf/data/vs/gfs/vs_vol_rep2/backup/metadata/8257806142240/vm-disk-5.bcfg   -- 修复指定的bcfg或者conf文件
#      ./vsts-fix-brian-split-file.sh   -- 脚本单独执行，则修复挂载点/sf/data/vs/gfs/vs_vol_rep2/下所有脑裂的非qcow2文件，如bcfg、conf等文件
# 适用: VS2.8及以下版本，不支持VS3.0版本
# 版本：Ver 1.0
#####################################################################


. /.PATH
. vs_stddir.sh
. vs_logger.sh

#set -e

CL='\033[0;34m'
NC='\033[0m'

IFS=$'\n'

g_volpath=
g_volfile=
g_file_tmp=/root/vsfire_recovery_FEC61AC2_bad_file.list
g_file_list=/root/vsfire_recovery_FEC61AC2_bad_noqcow2_file.list
g_posix_list=/root/vsfire_recovery_FEC61AC2_posixfile.list

function version_check()
{
    local ver
    ver=$(head -n1 /sf/vs/version )
    if [[ "${ver:0:3}" > "2.8" ]];then
        echo "$0 is only used for vs2.8 or low version, current ver is $ver"
        exit 0
    fi 
}

function log()
{
    echo -e "${CL}$1${NC}"
    log_info "$1"
}

function init()
{
    volname=$(gluster vol list)
    g_volpath=$(cat /proc/mounts | grep "$volname" | grep gfs | awk '{print $2}' )
    
    if [ ! -d "$g_volpath" ]; then
        log "vol path is not exit: ${g_volpath}"
        exit 1
    fi
    
    g_volfile=/sf/vs/glusterfs/etc/glusterd/vols/${volname}/info
    
    if [ ! -f "$g_volfile" ]; then
        log "vol file is not exit: ${g_volfile}"
        exit 1
    fi
}

function handle_badfiles()
{
    #找出所有脑裂的文件
    find ${g_volpath}/ -type f  1>/dev/null  2>"$g_file_tmp"
	
    #过滤掉qcow2文件
	cat $g_file_tmp |tr -d "'" | sed 's|find.*`\(/.*\): .*|\1|' | grep -v 'qcow2$' |grep -v 'vslun' >"$g_file_list"
    if [ $? -ne 0 ]; then 
        log "no split brian file, exit.. "
	exit 0
    fi
	
    #for line in $(cat "$g_file_list" | awk  '{print $2}' );
	for file in $(cat "$g_file_list")
    do 
        #file=${line:1:-2}
        log "current file: $file"
        
        handle_one_file "$file"
    done
}

function handle_one_file()
{
    local nfsfile=$1
    local suffix=${nfsfile:0-5}
    
    if [ "$suffix" == "qcow2" -o "$suffix" == "vslun" ]; then
        log "image file, not handle: $nfsfile"
        return
    fi
    
    get_posix_files "$nfsfile"
    
    if [ $? -ne 0 ]; then 
        return
    fi
    
    select_file "$nfsfile"
    
    backup_files "$nfsfile"
    
    recover_file "$nfsfile"
}

# 将所有主机的posix文件路径和IP记录到g_posix_list中
function get_posix_files()
{
    local ret=0
    local nfsfile=$1
    
    # 截取相对路径
    local len=${#g_volpath}
    local file=${nfsfile:$len}
    
    if [ -z "$file" ]; then
        log "get file path failed: ${nfsfile}"
        return 1
    fi
    
    local filedir=$(dirname "$file")
    local filename=$(basename "$file")
    
    truncate -s 0 "$g_posix_list"
    
    for ip in $(cat "$g_volfile" | grep -w host | awk -F= '{print $2}' | awk -F: '{print $1}' | sort | uniq); 
    do 
        ssh $ip "find /sf/data/vs/local/*/*/${filedir} -name ${filename} -exec echo $ip {} \;" >> "$g_posix_list"
        
        if [ $? -ne 0 ]; then 
            log "ssh $ip find /sf/data/vs/local/*/*/${filedir} -name ${filename} -exec echo $ip {} \;, exec failed."
            ret=1
            break
        fi
    done
    
    log "posix file list: "
    log "$(cat $g_posix_list)"
    
    return $ret
}

# 取一个最新的文件
function select_file()
{
    local ret=0
    local nfsfile=$1
    
    local tm1=""
    local tm2=""
    
    local aip=""
    local afile=""

    for line in $(cat "$g_posix_list"); 
    do
        ip=$(echo $line | awk -F' ' '{print $1}')
        file=$(echo $line | awk -F' ' '{print $2}')

        if [ -z "$ip" -o -z "$file" ]; then
            log "select_file: handle file failed: ${line}"
            ret=1
            break
        fi
        
        #log "ssh $ip /sf/bin/busybox/stat -c %s $file  2>/dev/null"
        
        size=$(ssh $ip "/sf/bin/busybox/stat -c %s \"$file\" " 2>/dev/null)
        if (( $size < 1 )); then
            continue
        fi
        
        if [ -z "$tm1" ]; then
            #log "ssh $ip /sf/bin/busybox/stat -c %Y $file  2>/dev/null"
            tm1=$(ssh $ip "/sf/bin/busybox/stat -c '%Y' \"$file\" " 2>/dev/null)
            aip="$ip"
            afile="$file"
            continue
        fi
        
        #log "ssh $ip /sf/bin/busybox/stat -c %Y $file  2>/dev/null"
        tm2=$(ssh $ip "/sf/bin/busybox/stat -c '%Y' \"$file\" " 2>/dev/null)
        if ((tm2 > tm1)); then
            aip="$ip"
            afile="$file"
            tm1=tm2
        fi
    done
    
	[ -z "$aip" ] || [ -z "$afile" ] || [ -z "$nfsfile" ] && log "select_file: aip->$aip, afile->$afile, nfsfile->$nfsfile" && exit 0
	
	log "select_file: ssh $aip cp $afile ${nfsfile}.tmp"
    ssh $aip cp "$afile" "${nfsfile}.tmp"
    
    return $ret
}

function recover_file()
{
    local nfsfile=$1
    
    log "recover_file: mv ${nfsfile}.tmp ${nfsfile}"
    
    mv "${nfsfile}.tmp" "${nfsfile}"
}

# 将所有的posix文件都移动到本机的local目录下
function backup_files()
{
    local ret=0
    local nfsfile=$1

    local tmpworkdir=/sf/data/local/workdir
        
    for line in $(cat "$g_posix_list"); 
    do
        ip=$(echo $line | awk -F' ' '{print $1}')
        file=$(echo $line | awk -F' ' '{print $2}')

        if [ -z "$ip" -o -z "$file" ]; then
            log "backup_files: handle file failed: ${line}"
            ret=1
            break
        fi
        
        posixfile=${file:17}
        posixdir=$(dirname "$posixfile")

        #log "ssh $ip mkdir -p $tmpworkdir/$posixdir; mv -f $file $tmpworkdir/$posixdir"
        ssh root@$ip "mkdir -p \"$tmpworkdir/$posixdir\"; mv -f \"$file\" \"$tmpworkdir/$posixdir\""
    done
    
    return $ret
}


function main()
{
    local param=$1
    
    if [ "$param" == "-h" -o "$param" == "--help" ]; then
        echo "Usage: "
        echo "       $0 /sf/data/vs/gfs/vs_vol_rep2/backup/metadata/8257806142240/vm-disk-5.bcfg  -- 修复指定的的bcfg或者conf文件，只能是非qcow2文件"
        echo "       $0                                                                           -- 修复虚拟存储挂载点下所有脑裂的非qcow2文件，如bcfg、conf等文件"
        exit 1
    fi
	
    version_check
    
    init
    
    if [ -z "$param" ]; then
        handle_badfiles
    else
        handle_one_file "$param"
    fi
}

main $*
