#!/sf/vs/bin/python
# -*- coding:utf-8 -*-

"""
## VG拷贝, 解决segment太多的问题
"""

import os
import re
import time
import socket
import hashlib
import traceback
import pylib.utils.utiltools as common

s_seg_defrag_start = 15000 # 大于1.5万，开始碎片整理
# s_seg_defrag_start = 0
s_seg_defrag_end = 10000 # 小于1万，结束碎片整理

def copy_and_check(src_dev, dst_dev, dev_size, buffer_size=4 * 1024 * 1024, update_interval=3):
    if not os.path.exists(src_dev) or not os.path.exists(dst_dev) or dev_size < 1024*1024:
        common.logger.error('src_dev {} or dst_dev {} not exists, or dev_size: {} invalid'.format(src_dev, dst_dev, dev_size))
        return -1

    copied_size = 0
    flush_data_size = 0
    copied_interval_size = 0
    total_size = dev_size
    start_time = time.time()
    start_interval_time = start_time
    # 执行文件数据拷贝
    with open(src_dev, 'rb') as fsrc, open(dst_dev, 'wb') as fdst:
        while True:
            buf = fsrc.read(buffer_size)
            if not buf:
                # 读到文件尾，对目标执行flush
                fdst.flush()
                break
            fdst.write(buf)
            buf_size = len(buf)
            copied_size += buf_size
            flush_data_size += buf_size
            copied_interval_size += buf_size
            # 达到flush数据量后，执行一次flush
            if flush_data_size >= buffer_size * 4:
                os.fsync(fdst.fileno())
                flush_data_size = 0

            write_interval_time = time.time()
            if write_interval_time - start_interval_time >= update_interval:
                speed = copied_interval_size / (write_interval_time - start_interval_time)
                remaining_time = (total_size - copied_size) / speed if speed > 0 else float('inf')
                common.print_with_clear('总大小: {}，已拷贝: {} ({:.2f} %), 速率: {}/s, 预计剩余时间: {:.2f} 秒'.
                                        format(common.to_human_readable(total_size),
                                               common.to_human_readable(copied_size),
                                               100.0 * copied_size / total_size,
                                               common.to_human_readable(speed), remaining_time))
                # 更新时间
                start_interval_time = write_interval_time
                copied_interval_size = 0

    print '\n'

    # 执行文件数据校验
    src_hasher = hashlib.md5()
    dst_hasher = hashlib.md5()
    with open(src_dev, 'rb') as fsrc, open(dst_dev, 'rb') as fdst:
        if total_size < 2 * buffer_size:
            # 文件大小小于2MB，对全文件计算md5
            src_hasher.update(fsrc.read(total_size))
            dst_hasher.update(fdst.read(total_size))
        else:
            # 文件大小大于2MB，分别对前1MB以及尾部1MB计算md5
            src_hasher.update(fsrc.read(buffer_size))
            dst_hasher.update(fdst.read(buffer_size))
            fsrc.seek(-buffer_size, os.SEEK_END)
            fdst.seek(-buffer_size, os.SEEK_END)
            src_hasher.update(fsrc.read(buffer_size))
            dst_hasher.update(fdst.read(buffer_size))
    if common.fault_point_result() or src_hasher.hexdigest() != dst_hasher.hexdigest():
        common.logger.error('src hash {} not equal dst hash {}'.format(src_hasher.hexdigest(), dst_hasher.hexdigest()))
        return -1

    return 0


def lv_defag(vgname, lv_name, lv_size, work_host, work_brick_path):
    print common.Colored().cyan('执行lv: {} 拷贝，大小: {}'.format(lv_name, common.to_human_readable(lv_size)))
    common.logger.info('try to defag vgname: {}, lv_name: {}'.format(vgname, lv_name))
    
    lv_dev = '/dev/{}/{}'.format(vgname, lv_name)
    lv_vg = lv_name.split('_')[0]
    lv_gfid = lv_name.split('_')[1].split('.')[0][0:16] # 获取的gfid只获取前16位
    lv_index = lv_name.split('_')[1].split('.')[1]
    lv_name_new = '{}_{}_{}.{}'.format(lv_vg, lv_gfid, common.VSFIRE_MAGIC, lv_index)
    lv_dev_new = '/dev/{}/{}'.format(vgname, lv_name_new)

    result = 0
    try:
        if os.path.exists(lv_dev_new):
            # 如果新的lv存在，刚先删除
            cmdline = '/sbin/lvremove -A y -f {}'.format(lv_dev_new)
            common.logger.info('try to cmdline: {}'.format(cmdline))
            common.cli(cmdline)
        
        cmdline = '/sbin/lvcreate -L {}B -n {} {}'.format(lv_size, lv_dev_new, vgname)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline)

        if copy_and_check(lv_dev, lv_dev_new, lv_size):
            result = -1
    except:
        common.logger.error('got except: {}'.format(traceback.format_exc()))
        result = -1

    if result == 0 and common.check_brick_process_alived(work_host, work_brick_path):
        # brick进程误被拉起，不允许删除原LV
        common.logger.error('host: {}, brick: {} process alived'.format(work_host, work_brick_path))
        result = -1

    if result == 0:
        # 成功，删除旧lv，并把新lv重命名
        cmdline = '/sbin/lvremove -A y -f {}'.format(lv_dev)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline)
        cmdline = '/sbin/lvrename {} {} {}'.format(vgname, lv_dev_new, lv_dev)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline)
    else:
        # 失败，删除新lv
        cmdline = '/sbin/lvremove -A y -f {}'.format(lv_dev_new)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline)

    return result


def get_replicate_by_vgname(vgname, replicate):
    for rep_id, replicate_bricks in replicate.items():
        for brick in replicate_bricks:
            # 找到VG对应的brick path且找到的brick不能是仲裁
            if vgname in brick['path'] and not brick['arbiter']:
                return replicate_bricks
    return {}

def get_vg_lv_and_seg_count(vgname):
    cmdline = '/sbin/lvs {}'.format(vgname)
    cmdline += " --units B --noheadings -o seg_count | /usr/bin/awk '{sum += $1} END {print NR, sum}'"
    common.logger.info('try to cmdline: {}'.format(cmdline))
    result = common.cli(cmdline, True)
    for line in result:
        if not line:
            continue
        lv_count, seg_count = line.split()
        return int(lv_count), int(seg_count)
    return 0, 0

def vg_defrag_with_lock(vgname, version, volume_name, hosts, bricks, replicate):
    
    online_bricks = common.get_online_bricks(volume_name)
    if not online_bricks:
        # 至少有一个在线brick
        common.logger.error('failed to supported, cannot find online_bricks')
        return -1
    
    if version >= common.VS_VERSION_3_0:
        replicate_bricks = replicate[common.RIGHT_TREE_INDEX]
    else:
        replicate_bricks = get_replicate_by_vgname(vgname, replicate)
    if not replicate_bricks:
        # 找到VG所在复制组的bricks
        common.logger.error('failed to supported, cannot find replicate_bricks')
        return -1
    
    common.logger.info('try to defrag vgname: {}'.format(vgname))

    work_host = None
    work_brick_id = -1
    work_brick_path = None
    for brick in replicate_bricks:
        if vgname in brick['path'] and not brick['arbiter']:
            work_host = brick['host']
            work_brick_id = brick['id']
            work_brick_path = brick['path']
            # 确保在VG所在主机执行
            if work_host == socket.gethostname():
                break
            else:
                print common.Colored().red('当前主机不是VG所在主机, 请切换到主机: {} 执行'.format(work_host))
                common.logger.error('failed to supported, vg host: {}, current host: {}'.format(work_host, socket.gethostname()))
                return -1
    
    if not work_host or not work_brick_path:
        common.logger.error('failed to supported, host: {} brick_path: {}'.format(work_host, work_brick_path))
        return -1
    
    # 确保除工作主机外，其它主机所有的brick都是在线的
    for host in hosts:
        if work_host == host:
            continue

        for brick in bricks.get(host):
            if online_bricks.get(brick['path']) != 'y':
                common.logger.error('brick: {}, is not online'.format(brick))
                return -1
    
    # 排除当前brick检查，其它brick内的文件正常
    common.logger.info('try to check brick_id: {}'.format(work_brick_id))
    if version >= common.VS_VERSION_3_0:
        import pylib.rpcservice as rpcservice
        if version < common.VS_VERSION_3_1:
            err_shards = rpcservice.route_check_brick(work_brick_id)
        else:
            err_shards = rpcservice.route_check_brick(work_brick_id, 'weak')
        if err_shards:
            common.logger.error('failed to check brick: {}, found err_shards: {}'.format(work_brick_path, err_shards))
            return -1
    else:
        result = common.check_brick_2x(work_brick_id, replicate_bricks)
        if not result:
            common.logger.error('failed to check file status, brick: {}'.format(work_brick_path))
            return -1

    result = 0
    try:
        # 杀brick进程
        common.stop_brick_process(work_host, work_brick_path)
        
        # 遍历VG中，所有的LV执行
        cmdline = '/sbin/lvs {} --units B --noheadings -o lv_name,seg_count,lv_size | sort -k2,2nr'.format(vgname)
        lvs_info = common.cli(cmdline, True)
        for lv_info in lvs_info:
            # 借分片暂时不支持
            if not lv_info or not lv_info.strip().startswith(vgname):
                continue
            lv_name, seg_count, lv_size = lv_info.split()
            seg_count = int(seg_count)
            lv_size = int(lv_size[:-1])  # 去掉结果的单位'B'
            if seg_count > 0:
                result = lv_defag(vgname, lv_name, lv_size, work_host, work_brick_path)
                if result:
                    # 有一个执行失败了，退出不再继续
                    common.logger.error('failed to defrag vgname: {}, lv_name: {}'.format(vgname, lv_name))
                    break

                lv_count, seg_count = get_vg_lv_and_seg_count(vgname)
                # 如果segment数量小于LV数量，或者segment数量小于1万，退出
                if seg_count <= lv_count or seg_count < s_seg_defrag_end:
                    common.logger.info('success to defrag vgname: {}, lv_count: {}, seg_count: {}'.format(vgname, lv_count, seg_count))
                    break
    except:
        common.logger.error('got except: {}'.format(traceback.format_exc()))
        result = -1
    
    # 恢复brick进程
    common.continue_brick_process(work_host, work_brick_path)
    return result


def get_localhost_vgs(current_host, bricks):
    localhost_vgs = []
    for brick in bricks.get(current_host):
        if brick['arbiter']:
            continue
        vg_name = brick['path'].split('/')[-2]
        localhost_vgs.append(vg_name)
    return list(set(localhost_vgs))

def vg_defrag(vgname, version, volume_name, hosts, bricks, replicate):
    lock_file = common.get_vsfire_lock_file(vgname)
    with common.VsfireFlock(lock_file) as lock:
        return vg_defrag_with_lock(vgname, version, volume_name, hosts, bricks, replicate)

def lvm_defrag(vgname):
    version = common.get_vs_version()
    if not common.is_vs_version_valid(version):
        return -1

    volume_name, hosts, replicate_num, has_arbiter, bricks, replicate = common.get_vs_cluster_info()
    if not volume_name:
        common.logger.error('failed to supported, volume_name: {}'.format(volume_name))
        return -1
    
    # 3.x只支持两主机
    if common.vs_has_efs(version, hosts):
        common.logger.error('failed to supported, hosts: {}'.format(hosts))
        return -1
    
    if not vgname:
        # 如果输入参数为空，表示自动扫描当前主机所有VG
        readline = '是否开始扫描当前主机所有VG, 获取碎片, 输入\'y\'继续，\'n\'退出'
        common.check_terminal_input(readline)
    
        current_host = socket.gethostname()
        localhost_vgs = get_localhost_vgs(current_host, bricks)
        if not localhost_vgs:
            common.logger.error('failed to supported, cannot find localhost_vgs')
            return -1
        
        defrag_vgs = []
        for vg in localhost_vgs:
            if not re.match(r"^\w{6}-\w{4}-\w{4}-\w{4}-\w{4}-\w{4}-\w{6}$", vg):
                continue
            lv_count, seg_count = get_vg_lv_and_seg_count(vg)
            # VG的segment数量大于LV数量，并且segment数量大于1.5万，开始碎片整理
            if seg_count >= lv_count and seg_count > s_seg_defrag_start:
                defrag_vgs.append([vg, lv_count, seg_count])
        if not defrag_vgs:
            common.logger.error('failed to supported, cannot find defrag_vgs')
            print common.Colored().cyan('当前主机没有找到足够多的碎片')
            return 0
        
        defrag_vgs = sorted(defrag_vgs, key=lambda x: x[2], reverse=True)
        for vg_info in defrag_vgs:
            print common.Colored().cyan('VG: {}, LV数量: {}, segment数量: {}'.format(vg_info[0], vg_info[1], vg_info[2]))
        
        for vg_info in defrag_vgs:
            vg = vg_info[0]
            readline = '是否开始执行VG: {} 碎片整理，碎片数量: {}, 输入\'y\'继续，\'n\'退出'.format(vg, vg_info[2])
            common.check_terminal_input(readline)
            result = vg_defrag(vg, version, volume_name, hosts, bricks, replicate)
            if result:
                common.logger.error('failed to defrag vgname: {}'.format(vg))
                return -1
        return 0
    elif isinstance(vgname, str) and re.match(r"^\w{6}-\w{4}-\w{4}-\w{4}-\w{4}-\w{4}-\w{6}$", vgname):
        return vg_defrag(vgname, version, volume_name, hosts, bricks, replicate)
    else:
        common.logger.error('failed to supported, vgname: {} invalid'.format(vgname))
    return -1


def _vg_defrag(vgname):
    ret = lvm_defrag(vgname)
    if ret:
        print common.Colored().red('执行失败')
    else:
        print common.Colored().cyan('执行成功')
    return ret
