#!/sf/vs/bin/python
# -*- coding:utf-8 -*-

"""
## 打印快照元数据内容并修改base gfid
"""

import os
import sys
import json
import ctypes
import datetime
import binascii
import pylib.utils.utiltools as common

SNAPSHOT_HEADER_RESERVED_SIZE = 428

class SnapshotMetaHdr(ctypes.Structure):
    _fields_ = [
        ("magic", ctypes.c_char * 8),
        ("checksum", ctypes.c_uint64),
        ("size", ctypes.c_size_t),
        ("curr_generation", ctypes.c_uint64),
        ("curr_offset", ctypes.c_int64),
        ("curr_size", ctypes.c_size_t),
        ("prev_generation", ctypes.c_uint64),
        ("prev_offset", ctypes.c_int64),
        ("prev_size", ctypes.c_size_t),
        ("meta_generation", ctypes.c_uint64),
        ("version", ctypes.c_int32),
        ("reserved", ctypes.c_uint8 * SNAPSHOT_HEADER_RESERVED_SIZE),
    ]

    def to_dict(self):
        """将结构体转换为字典，用于 JSON 序列化"""
        return {
            "magic": self.magic.decode('utf-8', errors='ignore').rstrip('\x00'),  # 字符串处理
            "checksum": '0x{:x}'.format(self.checksum),
            "size": self.size,
            "curr_generation": self.curr_generation,
            "curr_offset": self.curr_offset,
            "curr_size": self.curr_size,
            "prev_generation": self.prev_generation,
            "prev_offset": self.prev_offset,
            "prev_size": self.prev_size,
            "meta_generation": self.meta_generation,
            "version": self.version,
        }
    
def uuid_to_string(gfid):
    """将 UUID 转换为字符串表示"""
    return ''.join("{:x}".format(b) for b in list(gfid))

# 定义 snapshot_node_info 结构体
class SnapshotNodeInfo(ctypes.Structure):
    _fields_ = [
        ("snap_id", ctypes.c_int64),
        ("blocks", ctypes.c_uint64),
        ("create_time", ctypes.c_uint64),
        ("map_id", ctypes.c_int64),
        ("snap_parent_idx", ctypes.c_int),  # 父快照索引
        ("snap_flag", ctypes.c_int),         # 快照标志
    ]

    def snap_flag_str(self):
        """根据 snap_flag 的值返回相应的字符串表示"""
        snap_flag_string = ''
        snap_flag_list = ["VAILD", "NORMAL", "UNLINK", "LINKCLONE", "LINKCLONE_PARENT", "NEED_MERGE",
                          "CONSISTENCY", "MODIFIED", "NOT_REATIN", "FULLCLONE", "INTERNAL", "BASE",
                          "ROLLBACK_CURR", "ROLLBACK", "TIMER", "MERGE"]
        first = True
        for i in range(len(snap_flag_list)):
            if self.snap_flag & (1 << i):
                if first:
                    snap_flag_string += '('
                    first = False
                else:
                    snap_flag_string += ' | '
                snap_flag_string += snap_flag_list[i]
        if snap_flag_string:
            snap_flag_string += ')'
        return snap_flag_string
    
    class SnapGfid(ctypes.Structure):
        _fields_ = [("create_time", ctypes.c_uint64),
                    ("map_id", ctypes.c_int64)]

    def to_dict(self):
        """将 snapshot_node_info 结构体转换为字典，用于 JSON 序列化"""
        if self.snap_flag & 1 << 4:
            snap_gfid = SnapshotNodeInfo.SnapGfid()
            snap_gfid.create_time = self.create_time
            snap_gfid.map_id = self.map_id
            gfid_array = (ctypes.c_ubyte * 16)()
            ctypes.memmove(gfid_array, ctypes.byref(snap_gfid), ctypes.sizeof(snap_gfid))
            return {
                "snap_id": self.snap_id,
                "blocks": self.blocks,
                "gfid": uuid_to_string(gfid_array),
                "snap_parent_idx": self.snap_parent_idx,
                "snap_flag": self.snap_flag_str(),
            }
        else:
            return {
                "snap_id": self.snap_id,
                "blocks": self.blocks,
                "create_time": datetime.datetime.fromtimestamp(self.create_time).strftime("%Y-%m-%d %H:%M:%S"),
                "map_id": self.map_id,
                "snap_parent_idx": self.snap_parent_idx,
                "snap_flag": self.snap_flag_str(),
            }
    
class SnapshotMetaOndisk(ctypes.Structure):
    _fields_ = [
        ("generation", ctypes.c_uint64),
        ("gfid", ctypes.c_ubyte * 16),                         # uuid_t
        ("curr_id", ctypes.c_int64),
        ("base_blocks", ctypes.c_uint64),
        ("total_blocks", ctypes.c_uint64),
        ("size", ctypes.c_size_t),
        ("snap_size", ctypes.c_size_t),
        ("nr_normal_snap", ctypes.c_int),
        ("nr_linkclone", ctypes.c_int),
        ("nr_snap", ctypes.c_int),
        ("level", ctypes.c_int),
        ("parent_gfid", ctypes.c_ubyte * 16),                  # uuid_t
        ("linkclone_root_id", ctypes.c_int64),
        ("split_mode", ctypes.c_int),     # linkclone_split_mode_t
        ("consistency_generation", ctypes.c_uint64),
        ("consistency_uuid", ctypes.c_ubyte * 16),              # uuid_t
        ("next_consistency_file", ctypes.c_ubyte * 16),         # uuid_t
        ("nr_consistency", ctypes.c_uint32),
        ("snap_op", ctypes.c_int),                 # snapshot_op_t
        ("create_time", ctypes.c_uint64),
        ("prev_generation", ctypes.c_uint64),
        ("linkclone_parent_size", ctypes.c_uint64),
        ("linkclone_first_id", ctypes.c_int64),
        ("copy_type", ctypes.c_int),             # linkclone_copy_type_t
        ("copy_res", ctypes.c_uint32),
        ("padding", ctypes.c_uint8 * 312),
        # snapshot_info 数组需要根据实际内容处理
        ("snap_info", SnapshotNodeInfo * 64),    # 占位，后面会调整实际大小
    ]

    def split_mode_str(self):
        """根据 split_mode 的值返回相应的字符串表示"""
        split_mode_list = ["LINKCLONE_SPLIT_NULL", "LINKCLONE_SPLIT_WIDTH", "LINKCLONE_SPLIT_DEPTH", "LINKCLONE_SPLIT_FULL"]
        if self.split_mode in range(len(split_mode_list)):
            return split_mode_list[self.split_mode]
        return "LINKCLONE_SPLIT_UNKNOWN"
        
    def snap_op_str(self):
        """根据 split_mode 的值返回相应的字符串表示"""
        snapshot_op_list = ['SNAPSHOT_OP_NULL', 'SNAPSHOT_OP_CREATE', 'SNAPSHOT_OP_DELETE',
                             'SNAPSHOT_OP_ROLLBACK', 'SNAPSHOT_OP_CONSISTENCY_ROLLBACK',
                             'SNAPSHOT_OP_LINKCLONE', 'SNAPSHOT_OP_LINKCLONE_SPLIT', 'SNAPSHOT_OP_FULLCLONE',
                             'SNAPSHOT_OP_CONSISTENCY_CREATE']
        if self.snap_op in range(len(snapshot_op_list)):
            return snapshot_op_list[self.snap_op]
        return 'SNAPSHOT_OP_UNKNOWN'
    
    def copy_type_str(self):
        """根据 copy_type 的值返回相应的字符串表示"""
        copy_type_list = ["CLONE_COPY_FULL", "CLONE_COPY_SPARSE", "CLONE_COPY_AUTO", "CLONE_COPY_DEFAULT"]
        if self.copy_type in range(len(copy_type_list)):
            return copy_type_list[self.copy_type]
        return 'SNAPSHOT_OP_UNKNOWN'

    def to_dict(self):
        """将结构体转换为字典，用于 JSON 序列化"""
        
        snap_info_list = []
        for i in range(self.nr_snap):
            snap_info = self.snap_info[i]
            snap_info_list.append(snap_info.to_dict())  # 调用 SnapshotNodeInfo 的 to_dict

        return {
            "generation": self.generation,
            "gfid": uuid_to_string(self.gfid),
            "curr_id": self.curr_id,
            "base_blocks": self.base_blocks,
            "total_blocks": self.total_blocks,
            "size": self.size,
            "snap_size": self.snap_size,
            "nr_normal_snap": self.nr_normal_snap,
            "nr_linkclone": self.nr_linkclone,
            "nr_snap": self.nr_snap,
            "level": self.level,
            "parent_gfid": uuid_to_string(self.parent_gfid),
            "linkclone_root_id": self.linkclone_root_id,
            "split_mode": self.split_mode_str(),
            "consistency_generation": self.consistency_generation,
            "consistency_uuid": uuid_to_string(self.consistency_uuid),
            "next_consistency_file": uuid_to_string(self.next_consistency_file),
            "nr_consistency": self.nr_consistency,
            "snap_op": self.snap_op_str(),  # 需要具体定义
            "create_time": datetime.datetime.fromtimestamp(self.create_time).strftime("%Y-%m-%d %H:%M:%S"),
            "prev_generation": self.prev_generation,
            "linkclone_parent_size": self.linkclone_parent_size,
            "linkclone_first_id": self.linkclone_first_id,
            "copy_type": self.copy_type_str(),
            "copy_res": self.copy_res,
            "snap_info": snap_info_list,  # 添加快照信息列表
        }

def read_snapshot_meta(nfs_path):
    meta_hdr = None
    meta_prev = None
    meta_curr = None
    with open(nfs_path, "rb") as f:
        data_hdr = f.read(512)
        if len(data_hdr) != 512:
            common.logger.error('nfs_path: {} failed to read data_hdr: {}'.format(nfs_path, len(data_hdr)))
            return None, None, None  
        meta_hdr = SnapshotMetaHdr.from_buffer_copy(data_hdr)

        if meta_hdr.prev_size > 0:
            f.seek(meta_hdr.prev_offset)
            data_prev = f.read(meta_hdr.prev_size)
            if len(data_prev) != meta_hdr.prev_size:
                common.logger.error('nfs_path: {} failed to read data_prev: {}'.format(nfs_path, len(data_prev)))
                return None, None, None
            meta_prev = SnapshotMetaOndisk.from_buffer_copy(data_prev)

        f.seek(meta_hdr.curr_offset)
        data_curr = f.read(meta_hdr.curr_size)
        if len(data_curr) != meta_hdr.curr_size:
            common.logger.error('nfs_path: {} failed to read data_curr: {}'.format(nfs_path, len(data_hdr)))
            return None, None, None
        # print 'data_curr: {:x}'.format(binascii.crc32(data_curr) & 0xffffffff)
        meta_curr = SnapshotMetaOndisk.from_buffer_copy(data_curr)
    return meta_hdr, meta_prev, meta_curr
    
def snapshot_meta_hdr_dump(hdr):
    hdr_dict = hdr.to_dict()
    print '========== snapshot_meta_hdr =========='
    json_output = json.dumps(hdr_dict, indent=4)
    print(json_output)

def snapshot_meta_prev_dump(meta_curr):
    meta_curr_dict = meta_curr.to_dict()
    print '========== snapshot_meta_prev =========='
    json_output = json.dumps(meta_curr_dict, indent=4)
    print(json_output)

def snapshot_meta_curr_dump(meta_curr):
    meta_curr_dict = meta_curr.to_dict()
    print '========== snapshot_meta_curr =========='
    json_output = json.dumps(meta_curr_dict, indent=4)
    print(json_output)

def snapshot_meta_set_base_gfid(meta_curr, new_gfid):
    # new_gfid = "c6f89255-c9a0-4dc0-b045-55eee9938e66"  # 替换为您的新 UUID
    new_gfid = new_gfid.replace('-', '')
    assert len(new_gfid) == 32
    for i in range(len(new_gfid)/2):
        meta_curr.gfid[i] = int(new_gfid[2*i:2*i+2], 16)

def write_snapshot_meta(nfs_path, meta_hdr, meta_curr):
    with open(nfs_path, 'r+b') as f:
        f.seek(meta_hdr.curr_offset)
        meta_curr_size = ctypes.sizeof(SnapshotMetaOndisk)
        assert meta_curr_size <= meta_hdr.curr_size
        f.write((ctypes.c_ubyte * meta_curr_size).from_buffer_copy(meta_curr))

        f.seek(meta_hdr.curr_offset)
        data_curr = f.read(meta_hdr.curr_size)
        assert len(data_curr) == meta_hdr.curr_size
        meta_curr_crc = binascii.crc32(data_curr) & 0xffffffff
        meta_hdr.checksum = meta_curr_crc
        
        f.seek(0)
        f.write((ctypes.c_ubyte * 512).from_buffer_copy(meta_hdr))

def snapshot_meta_dump(file_path):
    version = common.get_vs_version()
    if not common.is_vs_version_valid(version):
        return -1
    
    volume_name, hosts, replicate_num, has_arbiter, bricks, replicate = common.get_vs_cluster_info()
    if not volume_name:
        common.logger.error('failed to supported, volume_name: {}'.format(volume_name))
        return -1

    nfs_path = os.path.join(common.get_vs_mount_path(volume_name, version), file_path)
    if not os.path.exists(nfs_path):
        common.logger.error('file_path: {} is not exist'.format(nfs_path))
        return -1
    
    meta_hdr, meta_prev, meta_curr = read_snapshot_meta(nfs_path)
    if not meta_hdr or not meta_curr:
        common.logger.error('failed to read snapshot meta')
        return -1
    
    # 打印出快照头部信息
    snapshot_meta_hdr_dump(meta_hdr)

    # 打印快照prev信息
    if meta_prev:
        snapshot_meta_prev_dump(meta_prev)

    # 打印出快照curr信息
    snapshot_meta_curr_dump(meta_curr)

    # 提示是否继续设置base_gfid
    new_gfid = ''
    readline = '是否设置文件快照base gfid, 输入\'n\'退出, 输入新的gfid继续'
    while True:
        print common.Colored().fuchsia(readline)
        new_gfid = sys.stdin.readline().strip('\n')
        if new_gfid.lower() == 'n':
            return 0
        elif common.check_str_is_gfid(new_gfid):
            break
        else:
            print common.Colored().red('输入新的gfid不合法, 请重新输入')

    # 设置base gfid
    snapshot_meta_set_base_gfid(meta_curr, new_gfid)

    # 将修改后的meta_curr写入文件
    write_snapshot_meta(nfs_path, meta_hdr, meta_curr)
    return 0


def _snapshot_meta_dump(filename):
    if not isinstance(filename, str) or not filename.startswith('/'):
        # filename表示文件路径，一定是'/'开头
        common.logger.error('failed to supported, filename: {} is not path'.format(filename))
        print common.Colored().red('执行失败')
        return -1
    # 去掉开头的'/'
    filename = filename[1:]

    lock_file = common.get_vsfire_lock_file(filename)
    with common.VsfireFlock(lock_file) as lock:
        ret = snapshot_meta_dump(filename)
        if ret:
            print common.Colored().red('执行失败')
        else:
            print common.Colored().cyan('执行成功')
        return ret
