#!/sf/vs/bin/python
# -*- coding:utf-8 -*-

"""
## 文件拷贝并自替换
"""

import os
import time
import hashlib
import traceback
import pylib.utils.utiltools as common


def replace_file(src_file_path, dst_file_path):
    if os.path.exists(dst_file_path):
        raise common.CmdError('dst_file_path: {} exist'.format(dst_file_path))
    os.rename(src_file_path, dst_file_path)


def copy_file_and_check(src_file_path, dst_file_path, buffer_size=1024 * 1024, update_interval=3):
    if not os.path.exists(src_file_path):
        common.logger.error('src_file_path {} not exists'.format(src_file_path))
        return False

    if os.path.exists(dst_file_path):
        common.logger.info('try to remove dst_file_path {}'.format(dst_file_path))
        os.remove(dst_file_path)

    copied_size = 0
    flush_data_size = 0
    copied_interval_size = 0
    total_size = os.path.getsize(src_file_path)
    start_time = time.time()
    start_interval_time = start_time
    # 执行文件数据拷贝
    with open(src_file_path, 'rb') as fsrc, open(dst_file_path, 'wb') as fdst:
        while True:
            buf = fsrc.read(buffer_size)
            if not buf:
                # 读到文件尾，对目标执行flush
                fdst.flush()
                break
            fdst.write(buf)
            buf_size = len(buf)
            copied_size += buf_size
            flush_data_size += buf_size
            copied_interval_size += buf_size
            # 达到flush数据量后，执行一次flush
            if flush_data_size >= 1048576:
                os.fsync(fdst.fileno())
                flush_data_size = 0

            write_interval_time = time.time()
            if write_interval_time - start_interval_time >= update_interval:
                speed = copied_interval_size / (write_interval_time - start_interval_time)
                remaining_time = (total_size - copied_size) / speed if speed > 0 else float('inf')
                common.print_with_clear('文件总大小: {}，已拷贝: {} ({:.2f} %), 速率: {}/s, 预计剩余时间: {:.2f} 秒'.
                                        format(common.to_human_readable(total_size),
                                               common.to_human_readable(copied_size),
                                               100.0 * copied_size / total_size,
                                               common.to_human_readable(speed), remaining_time))
                # 更新时间
                start_interval_time = write_interval_time
                copied_interval_size = 0

    print '\n'
    # 执行文件大小校验
    if common.fault_point_result() or os.path.getsize(src_file_path) != os.path.getsize(dst_file_path):
        common.logger.error('src size {} not equal dst size {}'.
                            format(os.path.getsize(src_file_path), os.path.getsize(dst_file_path)))
        os.remove(dst_file_path)
        return False

    # 执行文件数据校验
    src_hasher = hashlib.md5()
    dst_hasher = hashlib.md5()
    with open(src_file_path, 'rb') as fsrc, open(dst_file_path, 'rb') as fdst:
        if total_size < 2 * buffer_size:
            # 文件大小小于2MB，对全文件计算md5
            src_hasher.update(fsrc.read(total_size))
            dst_hasher.update(fdst.read(total_size))
        else:
            # 文件大小大于2MB，分别对前1MB以及尾部1MB计算md5
            src_hasher.update(fsrc.read(buffer_size))
            dst_hasher.update(fdst.read(buffer_size))
            fsrc.seek(-buffer_size, os.SEEK_END)
            fdst.seek(-buffer_size, os.SEEK_END)
            src_hasher.update(fsrc.read(buffer_size))
            dst_hasher.update(fdst.read(buffer_size))
    if common.fault_point_result() or src_hasher.hexdigest() != dst_hasher.hexdigest():
        common.logger.error('src hash {} not equal dst hash {}'.format(src_hasher.hexdigest(), dst_hasher.hexdigest()))
        os.remove(dst_file_path)
        return False

    return True


def copy_and_replace(file_path):
    version = common.get_vs_version()
    if common.fault_point_result() or not common.is_vs_version_valid(version):
        return -1

    volume_name, hosts, replicate_num, has_arbiter, bricks, replicate = common.get_vs_cluster_info()
    if common.fault_point_result() or not volume_name:
        common.logger.error('failed to supported, volume_name: {}'.format(volume_name))
        return -1

    online_bricks = common.get_online_bricks(volume_name)
    if common.fault_point_result() or not online_bricks:
        # 至少有一个在线brick
        common.logger.error('failed to supported, cannot find online_bricks')
        return -1

    nfs_path = os.path.join(common.get_vs_mount_path(volume_name, version), file_path)
    uuid_suffix = common.calculate_str_md5(file_path)
    src_file_path = nfs_path + '_src_{}_{}'.format(common.vsfire_recovery_dir, uuid_suffix)
    if os.path.exists(src_file_path):
        common.logger.info('try to rename file, from: {} to {}'.format(src_file_path, nfs_path))
        replace_file(src_file_path, nfs_path)

    # 校验文件存在
    if not os.path.exists(nfs_path) or not os.path.isfile(nfs_path):
        common.logger.error("nfs_path {}, is not file".format(nfs_path))
        return -1

    if common.vs_has_efs(version, hosts):
        from modules.file_recovery.show_fault_files import get_file_size_3x
        file_size = get_file_size_3x(file_path, replicate, online_bricks)
    else:
        from modules.file_recovery.show_fault_files import get_file_size_2x
        file_size = get_file_size_2x(version, file_path, hosts, replicate, online_bricks)
    statvfs = os.statvfs(common.get_vs_mount_path(volume_name, version))
    available_size = statvfs.f_bavail * statvfs.f_frsize
    # 文件大小需要大于卷大小至少16GB
    if file_size > available_size - (16*1024*1024*1024):
        common.logger.error('failed to recovery, file_size: {}, available_size: {}'.
                            format(common.to_human_readable(file_path), common.to_human_readable(available_size)))
        return -1

    vms_info = common.files_path_to_vms_name([file_path])
    if not vms_info:
        readline = '请确认文件: {} (大小 {})，没有业务访问，\n存储卷: {} (可用 {})，有足够剩余空间，输入\'y\'继续，\'n\'退出'.\
            format(file_path, common.to_human_readable(file_size),
                   volume_name, common.to_human_readable(available_size))
    else:
        vmid, vm_name = next(iter(vms_info.items()))
        if common.check_if_vmid_running(vmid):
            print common.Colored().red('检查到虚拟机vmid: {}，正在运行，退出修复'.format(vmid))
            common.logger.error('failed to supported, vmid: {} is running'.format(vmid))
            return -1
        readline = '请确认文件: {} (大小 {}，虚拟机 {})，没有业务访问，\n存储卷: {} (可用 {})，有足够剩余空间，输入\'y\'继续，\'n\'退出'.\
            format(file_path, common.to_human_readable(file_size), vm_name,
                   volume_name, common.to_human_readable(available_size))
    common.check_terminal_input(readline)

    # 创建vsfire的修复目录
    nfs_recovery_dir = os.path.join(common.get_vs_mount_path(volume_name, version), common.vsfire_recovery_dir)
    if not os.path.exists(nfs_recovery_dir):
        common.logger.info('try to mkdir: {}'.format(nfs_recovery_dir))
        os.mkdir(nfs_recovery_dir)

    dst_file_path = '{}/{}_dst_{}'.format(nfs_recovery_dir, os.path.basename(file_path), uuid_suffix)
    print common.Colored().cyan('开始拷贝文件，备份源文件: {}, \n拷贝目标文件: {}'.format(src_file_path, dst_file_path))
    try:
        common.logger.info('try to copy_file file_path: {}, src_file_path: {}, dst_file_path: {}'.
                           format(file_path, src_file_path, dst_file_path))

        # 将entry rename成src_entry
        common.logger.info('try to rename file, from: {} to {}'.format(nfs_path, src_file_path))
        replace_file(nfs_path, src_file_path)

        # 将src_entry拷贝成dst_entry
        common.logger.info('try to copy file, from: {} to {}'.format(src_file_path, dst_file_path))
        result = copy_file_and_check(src_file_path, dst_file_path)
        if not result:
            raise common.CmdError("failed to copy and check file, src_file_path: {}, dst_file_path: {}".
                                  format(src_file_path, dst_file_path))

        # 将dst_entry rename成entry
        common.logger.info('try to rename file, from: {} to {}'.format(dst_file_path, nfs_path))
        replace_file(dst_file_path, nfs_path)

        # 拷贝完成把src_path删除
        common.logger.info('try to remove file: {}'.format(src_file_path))
        os.remove(src_file_path)
        return 0
    except:
        common.logger.error('failed to copy_file file_path: {}, src_file_path: {}, dst_file_path: {}, got except: {}'.
                            format(file_path, src_file_path, dst_file_path, traceback.format_exc()))

    # 执行过程中出现异常，尝试恢复源文件
    if os.path.exists(src_file_path):
        common.logger.info('try to rename file, from: {} to {}'.format(src_file_path, nfs_path))
        replace_file(src_file_path, nfs_path)

    return -1


def _copy_and_replace(filename):
    if not isinstance(filename, str) or not filename.startswith('/'):
        # filename表示文件路径，一定是'/'开头
        common.logger.error('failed to supported, filename: {} is not path'.format(filename))
        print common.Colored().red('执行失败')
        return -1
    # 去掉开头的'/'
    filename = filename[1:]

    lock_file = common.get_vsfire_lock_file(filename)
    with common.VsfireFlock(lock_file) as lock:
        ret = copy_and_replace(filename)
        if ret:
            print common.Colored().red('执行失败')
        else:
            print common.Colored().cyan('执行成功')
        return ret
