#!/sf/vs/bin/python
# -*- coding:utf-8 -*-

"""
## 目录恢复工具
"""

import os
import re
import uuid
import traceback
import pylib.utils.utiltools as common


def check_if_dir_can_fix(dirname, bricks, support_subdir):
    match_gfid = None
    for host, bricks_info in bricks.items():
        for brick in bricks_info:
            local_path = os.path.join(brick['path'], dirname)
            try:
                if not support_subdir:
                    # 判断local目录下，只存在文件, 如果entry是文件，会抛不存在的异常
                    cmdline = '/usr/bin/find {} -not -type f -mindepth 1 -maxdepth 1'.format(local_path)
                    result = common.remote_cli(host, cmdline)
                    if result:
                        # 目录下存在非文件，暂时不支持处理
                        common.logger.error('cmdline: {}, result: {}, dir contain dir cannot support'.
                                            format(cmdline, result))
                        return False
                else:
                    # 判断是否是目录，如果entry是文件，会抛不存在的异常
                    cmdline = '/usr/bin/test -d {}'.format(local_path)
                    common.remote_cli(host, cmdline)
            except common.CmdError as e:
                common.logger.error('{}'.format(str(e)))
                # 有副本缺失，也可以修复
                if 'No such file or directory' not in str(e):
                    raise common.CmdError

            # 判断local目录下，目录指控是否正常，GFID是否相同
            try:
                cmdline = '/sf/vs/bin/getfattr -d -m. -e hex {}'.format(local_path)
                result = common.remote_cli(host, cmdline)
                if not result:
                    common.logger.error('cmdline: {} has not result'.format(cmdline))
                    return False

                found_gfid = False
                result = result.split('\n')
                for line in result:
                    if re.search(r"trusted.file_status", line):
                        common.logger.error('cmdline: {}, line: {}, has bad'.format(cmdline, line))
                        return True
                    if re.search(r"vs_vol_rep2-client-", line) \
                            and line.split('=')[1] != '0x000000000000000000000000':
                        common.logger.error('cmdline: {}, line: {}, has changelog'.format(cmdline, line))
                        return True
                    if re.search(r"trusted.afr.dirty", line) \
                            and line.split('=')[1] != '0x000000000000000000000000':
                        common.logger.error('cmdline: {}, line: {}, has dirty'.format(cmdline, line))
                        return True
                    if re.search(r"trusted.gfid", line):
                        found_gfid = True
                        if not match_gfid:
                            match_gfid = line
                        else:
                            if match_gfid != line:
                                common.logger.error('cmdline: {}, gfid: {}, line: {}, is not match'.
                                                    format(cmdline, match_gfid, line))
                                return True
                # 保证必须有GFID属性
                if not found_gfid:
                    common.logger.error('cmdline: {}, not found gfid'.format(cmdline, line))
                    return False
            except common.CmdError as e:
                if 'No such file or directory' not in str(e):
                    raise common.CmdError
                else:
                    # 有副本缺失，也可以修复
                    common.logger.error('{}'.format(str(e)))
                    return True

    common.logger.info('dirname: {}, maybe normal, cannot to fix'.format(dirname))
    return False


def check_new_dir_exist(version, volume_name, dirname, new_dirname, bricks):
    try:
        if common.fault_point_result():
            raise common.CmdError('common.fault_point_result')
        # 校验每个副本是否都创建成功
        for host, bricks_info in bricks.items():
            for brick in bricks_info:
                local_path = os.path.join(brick['path'], new_dirname)
                cmdline = '/bin/ls {}'.format(local_path)
                # 执行命令，失败了，抛出异常
                common.remote_cli(host, cmdline)
    except common.CmdError as e:
        common.logger.error('got common.CmdError: {}'.format(str(e)))
        raise


def rename_childs_to_new_dir(dirname, new_dirname, bricks):
    if common.fault_point_result():
        raise common.CmdError('common.fault_point_result')
    for host, bricks_info in bricks.items():
        for brick in bricks_info:
            try:
                local_path = os.path.join(brick['path'], dirname)
                new_local_path = os.path.join(brick['path'], new_dirname)
                cmdline = '/usr/bin/find {} -mindepth 1 -maxdepth 1 -exec {} {} \;'. \
                    format(local_path, '/bin/mv {}', new_local_path)
                common.logger.info('try to host: {} exec cmdline: {}'.format(host, cmdline))
                common.remote_cli(host, cmdline)
            except common.CmdError as e:
                if 'No such file or directory' not in str(e):
                    raise


def rename_dir_to_landfill(dirname, new_dirname, bricks):
    if common.fault_point_result():
        raise common.CmdError('common.fault_point_result')
    for host, bricks_info in bricks.items():
        for brick in bricks_info:
            try:
                local_path = os.path.join(brick['path'], dirname)
                cmdline = '/usr/bin/find {} -mindepth 1'.format(local_path)
                result = common.remote_cli(host, cmdline)
                if result:
                    # 原目录下还存在文件，说明存在并发问题，需要手动处理
                    common.logger.error('cmdline: {}, result: {}, dir remain file'.format(cmdline, result))
                    raise common.CmdError("old dir remain file")

                landfill_path = os.path.join(brick['path'], '.glusterfs/landfill', os.path.basename(new_dirname))
                cmdline = '/bin/mv {} {}'.format(local_path, landfill_path)
                common.logger.info('try to host: {} exec cmdline: {}'.format(host, cmdline))
                common.remote_cli(host, cmdline)
            except common.CmdError as e:
                if 'No such file or directory' not in str(e):
                    raise


def restore_dir(nfs_path, new_nfs_path):
    if os.path.exists(nfs_path):
        common.logger.error('nfs_path: {} is exist'.format(nfs_path))
        raise common.CmdError("old dir exists")

    common.logger.info('try to rename, from {} to {}'.format(new_nfs_path, nfs_path))
    os.rename(new_nfs_path, nfs_path)


def dir_recovery(dirname):
    version = common.get_vs_version()
    if not common.is_vs_version_valid(version):
        return -1

    if version >= common.VS_VERSION_3_0:
        # 当前只支持vs2.6/vs2.8
        common.logger.error('failed to supported, version: {}'.format(version))
        return -1

    if not isinstance(dirname, str) or not dirname.startswith('/'):
        # filename表示文件路径，一定是'/'开头
        common.logger.error('failed to supported, dirname: {} is not path'.format(dirname))
        return -1

    # 根目录不支持修复
    if dirname == '/' or dirname == '//':
        common.logger.error("entry: {} is invalid".format(dirname))
        return -1

    # 如果dirname以斜杠开头，去掉开头的斜杠
    if dirname.startswith('/'):
        dirname = dirname[1:]

    # 如果dirname以斜杠结尾，去掉结尾的斜杠
    if dirname.endswith('/'):
        dirname = dirname[:-1]

    volume_name, hosts, replicate_num, has_arbiter, bricks, replicate = common.get_vs_cluster_info()
    if not volume_name:
        common.logger.error('failed to supported, cannot find volume name')
        return -1

    # 校验entry是目录，但是目录下不存在子目录(默认不能支持子目录的修复，不然会导致修复出来的子目录显示不了文件)
    support_subdir = False
    result = check_if_dir_can_fix(dirname, bricks, support_subdir)
    if not result:
        common.logger.error('volume_name: {}, entry: {}, cannot to fix'.format(volume_name, dirname))
        return -1

    # 获取老目录全路径
    volume_dir_path = common.get_vs_mount_path(volume_name, version)
    nfs_path = os.path.join(volume_dir_path, dirname)
    vms_info = common.files_path_to_vms_name([nfs_path])
    if not vms_info:
        readline = '请确认目录: {}，内所有文件没有业务访问，输入\'y\'继续，\'n\'退出'.format(nfs_path)
    else:
        vmid, vm_name = next(iter(vms_info.items()))
        if common.check_if_vmid_running(vmid):
            print common.Colored().red('检查到虚拟机vmid: {}，正在运行，退出修复'.format(vmid))
            common.logger.error('failed to supported, vmid: {} is running'.format(vmid))
            return -1
        readline = '请确认目录: {} (虚拟机 {})，内所有文件没有业务访问，输入\'y\'继续，\'n\'退出'.format(nfs_path, vm_name)
    common.check_terminal_input(readline)

    # 获取新目录全路径
    new_dirname = dirname + '_' + common.vsfire_recovery_dir + '_' + common.calculate_str_md5(dirname)
    new_nfs_path = os.path.join(volume_dir_path, new_dirname)
    if not os.path.exists(new_nfs_path):
        common.logger.info('try to mkdir {}'.format(new_nfs_path))
        os.mkdir(new_nfs_path)

    if not os.path.exists(new_nfs_path) or not os.path.isdir(new_nfs_path):
        common.logger.error('failed to check new_nfs_path: {} exist'.format(new_nfs_path))
        return -1

    self_heal_has_closed = False
    try:
        common.logger.info('try to fix dir: {}, new_nfs_path: {}'.format(dirname, new_nfs_path))

        # 创建新的entry目录，并校验新的目录在所有local下都创建成功
        check_new_dir_exist(version, volume_name, dirname, new_dirname, bricks)

        # 关闭entry修复功能
        self_heal_has_closed = True
        cmdline = '/sf/vs/glusterfs/sbin/gluster v set {} cluster.entry-self-heal off'.format(volume_name)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline, False)

        # 将所有local下的entry目录下的文件rename到新的目录下
        rename_childs_to_new_dir(dirname, new_dirname, bricks)

        # 人为确认恢复出来的文件，是否正常
        readline = '请确认目录: {}，内所有文件正常，输入\'y\'继续，\'n\'退出'.format(new_nfs_path)
        common.check_terminal_input(readline)

        # 将所有local下的原目录删除（rename到landfill目录）
        rename_dir_to_landfill(dirname, new_dirname, bricks)

        # 将新目录rename成老的目录
        restore_dir(nfs_path, new_nfs_path)

        # 恢复entry修复功能
        cmdline = '/sf/vs/glusterfs/sbin/gluster v reset {} cluster.entry-self-heal'.format(volume_name)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline, False)
        return 0
    except:
        common.logger.error('failed to fix dir: {}, new_nfs_path: {}, got except: {}'.
                            format(dirname, new_nfs_path, traceback.format_exc()))
    if self_heal_has_closed:
        # 异常场景需要无条件，恢复entry修复功能
        cmdline = '/sf/vs/glusterfs/sbin/gluster v reset {} cluster.entry-self-heal'.format(volume_name)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline, False)

    print common.Colored().red('修复目录: {} 失败，请联系研发确认！\n'
                               '恢复entry-self-heal开关，(glusterd异常才可能需要手动关闭)\n'
                               '手动删除创建的临时目录: {} (请确认目录下每个副本不存在文件才可以手动删除)'.
                               format(nfs_path, new_nfs_path))
    return -1


def _dir_recovery(dirname):
    lock_file = common.get_vsfire_lock_file()
    with common.VsfireFlock(lock_file) as lock:
        ret = dir_recovery(dirname)
        if ret:
            print common.Colored().red('执行失败')
        else:
            print common.Colored().cyan('执行成功')
        return ret
