#!/sf/vs/bin/python
# -*- coding:utf-8 -*-

"""
## tier坏道修复工具
"""
import os
import sys
import json
import socket
import traceback
import pylib.utils.utiltools as common

CACHE_CONF = '/sf/cfg/vs/cache/wcache.json'

def try_bad_blocks_repair2x(tier_gfid_offsets, bad_blocks_gfid, bad_block_dev, src_brick, dst_brick, offset_block, block_size, is_forced):
    tier_offset = offset_block * block_size
    gfid_path = '.glusterfs/{}/{}/{}'.format(bad_blocks_gfid[0:2], bad_blocks_gfid[2:4], bad_blocks_gfid)

    file_offset, file_length = get_file_offset_from_tier_offset(tier_gfid_offsets, tier_offset, block_size)
    if file_length != block_size:
        common.logger.error('failed to get tier_offset: {}, file_length: {}, block_size: {}'.format(tier_offset, file_length, block_size))
        print common.Colored().red('块设备: {}, 分层偏移: {} 无法转化该分层偏移到文件的偏移'.format(bad_block_dev, tier_offset))
        return -1
    
    # 确认目的端是坏道才允许修复，不然跳过
    if not common.check_dev_has_badblock(dst_brick['host'], bad_block_dev, tier_offset, block_size):
        common.logger.error('failed to check badblock, dev: {}, offset: {}'.format(bad_block_dev, tier_offset))
        print common.Colored().red('块设备: {}, 分层偏移: {} 无法确认该区间是真坏道, 请重试修复该坏道区间'.format(bad_block_dev, tier_offset))
        return -1
    common.calc_dev_block_md5(dst_brick['host'], bad_block_dev, tier_offset, block_size)

    # 获取源端的对应的LV与LV偏移
    src_lv_dev = ''
    src_file_offset = file_offset
    src_local_path = os.path.join(src_brick['path'], gfid_path)
    cmdline = '/bin/cat {}'.format(src_local_path)
    result = common.remote_cli(src_brick['host'], cmdline, True)
    for line in result:
        if line and line.split(':')[0] == 'SHARD':
            lv_size = int(line.split(':')[5]) * (128*1024*1024)
            if src_file_offset >= lv_size:
                src_file_offset -= lv_size
            else:
                disk_vg = line.split(':')[2]  # 被借空间的VG
                src_vg = src_brick['path'].split('/')[5]  # 源brick所在的VG
                lv_index = int(line.split(':')[3])
                src_lv_dev = '/dev/{}/{}_{}.{}'.format(disk_vg, src_vg, bad_blocks_gfid.replace('-', ''), lv_index)
                break

    # 将源的偏移转化成块对齐
    src_offset_blocks = src_file_offset / block_size
    cmdline = '/usr/bin/ssh root@{} \"/bin/dd if={} bs={} count=1 skip={} iflag=direct conv=notrunc\" | ' \
              '/usr/bin/ssh root@{} \"/bin/dd of={} bs={} count=1 seek={} oflag=direct conv=notrunc\"'.\
        format(common.get_ssh_host(src_brick['host']), src_lv_dev, block_size, src_offset_blocks,
               common.get_ssh_host(dst_brick['host']), bad_block_dev, block_size, offset_block)

    # 判断是否执行坏道修复
    if not is_forced:
        readline = '是否执行坏道修复，修复命令:{}，输入\'y\'继续, \'n\'退出'.format(cmdline)
        common.check_terminal_input(readline)

    common.logger.info('try to cmdline: {}'.format(cmdline))
    common.cli(cmdline, False)
    common.calc_dev_block_md5(dst_brick['host'], bad_block_dev, tier_offset, block_size)
    return 0

def try_bad_blocks_repair3x(tier_gfid_offsets, bad_blocks_gfid, bad_block_dev, src_brick, dst_brick, offset_block, block_size, is_forced):
    tier_offset = offset_block * block_size
    src_efs_mount_path = '/mnt/{}_{}/{}'.format(src_brick['path'].split('/')[5], common.VSFIRE_MAGIC, bad_blocks_gfid)

    file_offset, file_length = get_file_offset_from_tier_offset(tier_gfid_offsets, tier_offset, block_size)
    if file_length != block_size:
        common.logger.error('failed to get tier_offset: {}, file_length: {}, block_size: {}'.format(tier_offset, file_length, block_size))
        print common.Colored().red('块设备: {}, 分层偏移: {} 无法转化该分层偏移到文件的偏移'.format(bad_block_dev, tier_offset))
        return -1

    # 确认目的端是坏道才允许修复，不然跳过
    if not common.check_dev_has_badblock(dst_brick['host'], bad_block_dev, tier_offset, block_size):
        common.logger.error('failed to check badblock, dev: {}, offset: {}'.format(bad_block_dev, tier_offset))
        print common.Colored().red('块设备: {}, 分层偏移: {} 无法确认该区间是真坏道, 请重试修复该坏道区间'.format(bad_block_dev, tier_offset))
        return -1
    common.calc_dev_block_md5(dst_brick['host'], bad_block_dev, tier_offset, block_size)
    
    cmdline = '/usr/bin/ssh root@{} \"/bin/dd if={} bs={} count=1 skip={} iflag=direct conv=notrunc\" | ' \
                '/usr/bin/ssh root@{} \"/bin/dd of={} bs={} count=1 seek={} oflag=direct conv=notrunc\"'. \
        format(common.get_ssh_host(src_brick['host']), src_efs_mount_path, block_size, file_offset/block_size,
                common.get_ssh_host(dst_brick['host']), bad_block_dev, block_size, offset_block)

    # 判断是否执行坏道修复
    if not is_forced:
        readline = '是否执行坏道修复，修复命令:{}，输入\'y\'继续, \'n\'退出'.format(cmdline)
        common.check_terminal_input(readline)

    common.logger.info('try to cmdline: {}'.format(cmdline))
    common.cli(cmdline, False)
    common.calc_dev_block_md5(dst_brick['host'], bad_block_dev, tier_offset, block_size)
    return 0


def get_ssd_uuid_by_brick(brick_path):
    if not os.path.exists(CACHE_CONF) or os.path.getsize(CACHE_CONF) == 0:
        common.logger.error('cache conf: {} invalid'.format(CACHE_CONF))
        return ''
    
    with open(CACHE_CONF, 'r') as f:
        cache_data = json.load(f)
        if not cache_data.get('maps'):
            common.logger.error('failed to get maps in cache_data: {}'.format(cache_data))
            return ''
        
        for cache in cache_data['maps']:
            for brick in cache['bricks']:
                if brick['brickId'] == brick_path:
                    return  cache['uuid'][0:-7] # 返回'waK8Ug-ex2c-H8bK-FRQT-aEFs-Xtlp-RN2l1P'
    return ''

def get_ssd_dump_info(ssd_uuid, host):
    cmdline = '/sf/vs/bin/vs_tier_cli.py -c dump'
    result = common.remote_cli(host, cmdline, False)
    ssd_dump = json.loads(result.decode('utf-8').strip())
    for ssd_info in ssd_dump['ssd']:
        if ssd_info['ssd_uuid'] == ssd_uuid:
            return ssd_info
    return {}

# 获取文件在分层中的偏移信息
def get_tier_gfid_offsets(gfid, brick):
    tier_gfid_offsets = {}
    ssd_uuid = get_ssd_uuid_by_brick(brick['path'])
    if not ssd_uuid:
        common.logger.error('failed to get ssd_uuid')
        return {}

    ssd_info = get_ssd_dump_info(ssd_uuid, brick['host'])
    if not ssd_info:
        common.logger.error('failed to get ssd_info')
        return {}

    tier_gfid_offsets['ssd_uuid'] = ssd_uuid
    tier_gfid_offsets['offsets'] = []
    cmdline = '/sf/vs/bin/vs_tier_cli.py -c file -a brick_id={},gfid={}'.format(brick['path'], gfid)
    result = common.cli(cmdline, True)
    if ssd_info['version'] == '0x10000':
        # V1格式
        # file_no  block_no data     hot                meta
        # 0        2        1048576  7                  ffffffffffffffff ffffffffffffffff ffffffffffffffff ffffffffffffffff
        # 1        3        1048576  8                  ffffffffffffffff ffffffffffffffff ffffffffffffffff ffffffffffffffff
        for line in result:
            offset_info = line.split()
            if len(offset_info) > 2 and offset_info[0].isdigit() and offset_info[1].isdigit():
                offset = {}
                offset['file_offset'] = int(offset_info[0]) * ssd_info['block_size']
                offset['tier_offset'] = int(offset_info[1]) * ssd_info['block_size'] + ssd_info['data_offset']
                offset['length'] = ssd_info['block_size']
                tier_gfid_offsets['offsets'].append(offset)
    elif ssd_info['version'] == '0x20000':
        # V2格式
        # +------------+--------+-------+-------+-----+---------+-------+---------------+
        # |   offset   |  len   | clean | begin | cnt | blockno | mapno |   ssd_offset  |
        # +------------+--------+-------+-------+-----+---------+-------+---------------+
        # | 362283008  | 131072 |   0   |   32  |  32 | 2015770 |   1   | 2127225290752 |
        # | 362414080  | 131072 |   0   |   64  |  32 | 2015770 |   2   | 2127225421824 |
        for line in result:
            offset_info = line.split()
            if len(offset_info) > 6 and offset_info[0] == '|' and offset_info[1].isdigit() and offset_info[2] == '|' and offset_info[3].isdigit():
                offset = {}
                offset['file_offset'] = int(offset_info[1])
                offset['tier_offset'] = int(offset_info[-2])
                offset['length'] = int(offset_info[3])
                tier_gfid_offsets['offsets'].append(offset)
    elif ssd_info['version'] == '0x30000':
        # V3格式
        # +---------+---------------------+-------------+-------+-------+--------+--------+--------+-------+-------------+-----+
        # |   eno   |        offset       |  phy-offset | shard | dirty | in_use | dm_err | demote | sense |    bitmap   | hot |
        # +---------+---------------------+-------------+-------+-------+--------+--------+--------+-------+-------------+-----+
        # | 1382488 | 9223372032828243968 | 23199088640 | 28674 |   1   |   1    |   0    |   0    |   0   | [ 1 0 0 0 ] | 252 |
        # | 1382496 |          0          | 23199219712 | 28673 |   1   |   1    |   0    |   0    |   0   | [ 1 1 1 1 ] | 123 |
        for line in result:
            offset_info = line.split()
            if len(offset_info) > 6 and offset_info[0] == '|' and offset_info[1].isdigit() and offset_info[2] == '|' and offset_info[3].isdigit():
                offset = {}
                offset['file_offset'] = int(offset_info[3])
                offset['tier_offset'] = int(offset_info[5])
                offset['length'] = ssd_info['block_size']
                tier_gfid_offsets['offsets'].append(offset)
    else:
        common.logger.error('unsupport ssd_info version: {}'.format(ssd_info['version']))
        return {}
    
    # 获取不到文件偏移，返回空
    if not tier_gfid_offsets['offsets']:
        common.logger.error('failed to get offsets')
        return {}
    return tier_gfid_offsets

# 从文件偏移转化成分层的偏移
def get_tier_offset_from_file_offset(tier_gfid_offsets, file_offset, file_length):
    tier_offset = 0
    tier_length = 0
    for offset in tier_gfid_offsets.get('offsets'):
        # 为了简化考虑，只处理一个区间匹配的场景
        if file_offset >= offset['file_offset'] and file_offset < offset['file_offset'] + offset['length']:
            common.logger.info('file_offset: {}, file_length: {} match offset: {}'.format(file_offset, file_length, offset))
            tier_offset = offset['tier_offset'] + (file_offset - offset['file_offset'])
            if offset['file_offset'] + offset['length'] - file_offset < file_length:
                tier_length = offset['file_offset'] + offset['length'] - file_offset
            else:
                tier_length = file_length
            break
    return tier_offset, tier_length

# 从分层的偏移转化成文件的偏移
def get_file_offset_from_tier_offset(tier_gfid_offsets, tier_offset, tier_length):
    file_offset = 0
    file_length = 0
    for offset in tier_gfid_offsets.get('offsets'):
        # 为了简化考虑，只处理一个区间匹配的场景
        if tier_offset >= offset['tier_offset'] and tier_offset < offset['tier_offset'] + offset['length']:
            common.logger.info('tier_offset: {}, tier_length: {} match offset: {}'.format(tier_offset, tier_length, offset))
            file_offset = offset['file_offset'] + (tier_offset - offset['tier_offset'])
            if offset['tier_offset'] + offset['length'] - tier_offset < tier_length:
                file_length = offset['tier_offset'] + offset['length'] - tier_offset
            else:
                file_length = tier_length
            break
    return file_offset, file_length

def get_file_path_from_gfid(gfid, hosts):
    cmdline = '/sf/vs/bin/vs_lv2file.sh {}'.format(gfid.replace('-', ''))
    for host in hosts:
        try:
            result = common.remote_cli(host, cmdline, True)
            for line in result:
                # /sf/data/vs/local/fbOgFF-Za3P-034D-rCZn-S08b-jd5t-CnATyR/f02b03fa-c71e-11ef-8de3-005056959419/images/host-005056956533/centos_auto_test.vm/vm-disk-1.qcow2
                if line.startswith('/sf/data/vs/local/'):
                    entries = line.split('/')
                    if len(entries) >= 8:
                        file_path = '/'.join(entries[7:])
                        return file_path
        except common.CmdError as e:
            common.logger.warn('got except: {}'.format(str(e)))
            continue
    return None

def get_file_replicate_bricks(version, gfid, hosts, replicate, online_bricks):
    gfid_path =  '.glusterfs/{}/{}/{}'.format(gfid[0:2], gfid[2:4], gfid)
    if version >= common.VS_VERSION_3_0:
        from modules.file_recovery.show_fault_files import get_file_replicate_3x
        recovery_replicate = get_file_replicate_3x(gfid_path, replicate, online_bricks)
    else:
        from modules.file_recovery.show_fault_files import get_file_replicate_2x
        recovery_replicate = get_file_replicate_2x(gfid_path, hosts, replicate)
    # 获取的数据复制组，只能是一个
    if not recovery_replicate or len(recovery_replicate) != 1:
        common.logger.error('failed to find data replicate: {}'.format(recovery_replicate))
        return {}
    
    if version >= common.VS_VERSION_3_0:
        replicate_gfid, replicate_bricks = next(iter(recovery_replicate.items()))
    else:
        rep_id, replicate_bricks = next(iter(recovery_replicate.items()))
    if not replicate_bricks:
        common.logger.error('failed to get replicate_bricks')
        return {}
    return replicate_bricks

# 基于坏道主机或者坏道brick id 从复制组中，返回源与目的brick
def get_src_and_dst_brick(replicate_bricks, badblock_host, badblock_brick_id):
    src_brick = {}
    dst_brick = {}
    for brick in replicate_bricks:
        if src_brick and dst_brick:
            break

        if brick['arbiter']:
            continue
        
        if (badblock_host and badblock_host == brick['host']) or (badblock_brick_id and badblock_brick_id == brick['id']):
            dst_brick = brick
        else:
            src_brick = brick
    return src_brick, dst_brick

# 提示虚拟机业务没有运行， 输入文件的GFID
def check_vm_is_stopped(version, gfid, volume_name, hosts):
    if version >= common.VS_VERSION_3_0:
        import pylib.rpcservice as rpcservice
        if common.vs_is_two_host(hosts):
            # 两主机环境，需要列出所有分片后，从所有分片中查看到文件名
            all_shards = rpcservice.route_list_all_shards(volume_name)
            base_file_path = rpcservice.route_gfid_to_file_path(gfid, all_shards)
        else:
            # 三主机以上，通过shard层直接转文件名
            base_file_path = rpcservice.gfid_to_base_file_path(gfid, None)
    else:
        base_file_path = get_file_path_from_gfid(gfid, hosts)
    vms_info = common.files_path_to_vms_name([base_file_path])
    if not vms_info:
        readline = '开始修复分片: {} 上的坏道，请确认相关文件: {}, 是否没有业务访问，输入\'y\'继续，\'n\'退出'.format(gfid, base_file_path)
    else:
        vmid, vm_name = next(iter(vms_info.items()))
        # 不判断虚拟机是否已经开机，有可能虚拟机开机场景下需要修坏道，完全交给人工确认？
        # if common.check_if_vmid_running(vmid):
        #     print common.Colored().red('检查到虚拟机vmid: {}，正在运行，退出修复'.format(vmid))
        #     common.logger.error('failed to supported, vmid: {} is running'.format(vmid))
        #     return -1
        vm_status = '正在运行' if common.check_if_vmid_running(vmid) else '已关机'
        readline = '开始修复分片: {} 上的坏道，请确认相关文件: {}，虚拟机: {}, 状态: {}, 是否没有业务访问，输入\'y\'继续，\'n\'退出'.format(gfid, base_file_path, vm_name, vm_status)
    common.check_terminal_input(readline)

def try_scan_or_get_badblock_offset(host, cmdline, bad_blocks_offsets_path, bad_block_size):
    bad_blocks_offsets = []
    while True:
        if host:
            readline = '在主机: {}, 是否自动扫描坏道，输入\'y\'执行（注意扫描坏道可能导致磁盘卡慢）, 否则输入存放坏道区间的文件路径（注意文件路径为绝对路径）'.format(host)
        else:
            readline = '是否自动扫描坏道，输入\'y\'执行（注意扫描坏道可能导致磁盘卡慢）, 否则输入存放坏道区间的文件路径（注意文件路径为绝对路径）'
        print common.Colored().fuchsia('{}'.format(readline))
        step = sys.stdin.readline().strip('\n')
        if step.lower() == 'y':
            if host:
                common.logger.info('try to host: {}, cmdline: {}'.format(host, cmdline))
                common.remote_cli(host, cmdline)
            else:
                common.logger.info('try to cmdline: {}'.format(cmdline))
                common.cli(cmdline)
            break
        elif os.path.exists(step):
            bad_blocks_offsets_path = step
            break
        else:
            print common.Colored().red('输入字符错误，请重新输入')
    
    with open(bad_blocks_offsets_path, 'r') as f:
        for line in f:
            if line.strip().isdigit():
                bad_blocks_offsets.append(int(line.strip()))
    
    if bad_blocks_offsets:
        print common.Colored().cyan('查找到坏道偏移列表: {},  偏移单位为块大小({})'.format(bad_blocks_offsets, bad_block_size))
    return bad_blocks_offsets_path, bad_blocks_offsets

# 坏道拷贝前，准备源brick，清空缓存与挂载EFS
def prepare_src_brick(volume_name, gfid, has_efs, src_brick):
    has_force_clean_wcache = False
    if has_efs:
        if common.is_wcc_dirty_3x(gfid, src_brick['host'], src_brick['path']):
            has_force_clean_wcache = True
            common.force_clean_wcache_3x(volume_name, gfid, src_brick['host'], src_brick['path'])

        if common.is_tier_dirty_3x(gfid, src_brick['host'], src_brick['path']):
            common.force_clean_tier_3x(gfid, src_brick['host'], src_brick['path'])
        
        # 只读挂载出源efs数据
        common.mount_efs_path(src_brick)
    else:
        gfid_path = '.glusterfs/{}/{}/{}'.format(gfid[0:2], gfid[2:4], gfid)
        if common.is_wcc_dirty_2x(src_brick['host'], src_brick['path'], gfid_path):
            has_force_clean_wcache = True
            common.force_clean_wcache_2x(volume_name, src_brick['host'], src_brick['path'], gfid_path)

        if common.is_tier_dirty_2x(src_brick['host'], src_brick['path'], gfid_path):
            common.force_clean_tier_2x(src_brick['host'], src_brick['path'], gfid_path)
    return has_force_clean_wcache

# 坏道拷贝后，清理源brick，恢复缓存与卸载EFS
def post_src_brick(has_force_clean_wcache, volume_name, has_efs, src_brick):
    if has_efs:
        common.umount_efs_path(src_brick)

    if has_force_clean_wcache:
        common.reset_clean_wcache_3x(volume_name)

def tier_data_repair(version, gfid, volume_name, hosts, replicate, online_bricks):
    common.logger.info('try to tier_data_repair, gfid: {}'.format(gfid))
    color = common.Colored()
    has_efs = True if (common.vs_has_efs(version, hosts)) else False
    bad_block_file_offset = 0
    bad_block_file_length = 4*1024*1024 # 最大块大小
    bad_blocks_offsets = []
    bad_block_size = 4096  # 默认坏道块大小为4096
    bad_block_host = socket.gethostname()
    src_brick = {}  # 坏道修复的源副本
    dst_brick = {}  # 坏道修复的目标副本（也就是存在坏道的副本）

    # 获取文件的复制组
    replicate_bricks = get_file_replicate_bricks(version, gfid, hosts, replicate, online_bricks)
    if not replicate_bricks:
        common.logger.error('failed to get replicate_bricks')
        return -1
    
    # 获取坏道修复的源与目的brick
    src_brick, dst_brick = get_src_and_dst_brick(replicate_bricks, bad_block_host, None)
    if not src_brick or not dst_brick:
        common.logger.error('failed to get src_brick: {} or dst_brick: {}'.format(src_brick, dst_brick))
        return -1

    # 提示业务影响
    check_vm_is_stopped(version, gfid, volume_name, hosts)

    # 交互输入坏道所在的文件偏移
    readline = '当前文件:{}, 请输入坏道所在文件的偏移 (偏移单位为字节, 从分层日志中获取)'.format(gfid)
    while True:
        print color.fuchsia('{}'.format(readline))
        step = sys.stdin.readline().strip('\n')
        if step.isdigit():
            bad_block_file_offset = int(step)
            break
        else:
            print color.red('输入字符错误，请重新输入')

    # 确认分层坏道位置
    tier_gfid_offsets = get_tier_gfid_offsets(gfid, dst_brick)
    if not tier_gfid_offsets:
        common.logger.error('failed to get tier_gfid_offsets')
        return -1
    
    ssd_dev = '/dev/{}/{}-tcache'.format(tier_gfid_offsets['ssd_uuid'], tier_gfid_offsets['ssd_uuid'])
    bad_blocks_offsets_path = '/root/{}_badblocks.txt'.format(common.vsfire_recovery_dir)  # 默认坏道扫描文件
    
    tier_offset, tier_length = get_tier_offset_from_file_offset(tier_gfid_offsets, bad_block_file_offset, bad_block_file_length)
    if tier_length == 0:
        common.logger.error('failed to get tier_offset: {}, tier_length: {}'.format(tier_offset, tier_length))
        return -1
    
    cmdline = '/sbin/badblocks -b {} -o {} {} {} {}'. \
        format(bad_block_size, bad_blocks_offsets_path, ssd_dev, (tier_offset + tier_length) / bad_block_size, tier_offset / bad_block_size)
    bad_blocks_offsets_path, bad_blocks_offsets = try_scan_or_get_badblock_offset(None, cmdline, bad_blocks_offsets_path, bad_block_size)
    if not bad_blocks_offsets:
        common.logger.error('failed to get bad_blocks_offsets in file: {}'.format(bad_blocks_offsets_path))
        return -1

    # 执行坏道修复
    result = 0
    has_force_clean_wcache = True
    try:
        # 坏道修复前，准备源brick
        has_force_clean_wcache = prepare_src_brick(volume_name, gfid, has_efs, src_brick)

        # 开始执行坏道修复
        is_forced = False  # 默认第1次执行需要手工确认，主要是自动计算的偏移地址需要确认
        for offset in bad_blocks_offsets:
            if has_efs:
                if try_bad_blocks_repair3x(tier_gfid_offsets, gfid, ssd_dev, src_brick, dst_brick, offset, bad_block_size, is_forced):
                    result = -1
            else:
                if try_bad_blocks_repair2x(tier_gfid_offsets, gfid, ssd_dev, src_brick, dst_brick, offset, bad_block_size, is_forced):
                    result = -1
            
            # 成功过一次，说明后面的坏道修复比较安全，不需要再重复确认了
            if not result:
                is_forced = True

        # 坏道修复后，尝试删除临时文件
        cmdline = '/bin/rm -f {}'.format(bad_blocks_offsets_path)
        common.logger.info('try to cmdline: {}'.format(cmdline))
        common.cli(cmdline)

        # 坏道修复后，回退源brick
        post_src_brick(has_force_clean_wcache, volume_name, has_efs, src_brick)
        return result
    except:
        common.logger.error('failed to tier_data_repair: {}, got except:{}'.format(gfid, traceback.format_exc()))

    # 坏道修复后，尝试删除临时文件
    cmdline = '/bin/rm -f {}'.format(bad_blocks_offsets_path)
    common.logger.info('try to cmdline: {}'.format(cmdline))
    common.cli(cmdline)

    # 异常场景，回退源brick
    post_src_brick(has_force_clean_wcache, volume_name, has_efs, src_brick)
    return -1


# 将坏道扫描的日志路径转化成对应的brick_id
def get_brick_id_from_bricks(badblock_log_path, bricks):
    for host, bricks_info in bricks.items():
        for brick in bricks_info:
            if brick['path'].split('/')[-1] in badblock_log_path and brick['path'].split('/')[-2] in badblock_log_path:
                return brick['id']
    return None


def tier_data_repair_start(gfid):
    version = common.get_vs_version()
    if not common.is_vs_version_valid(version):
        common.logger.error('failed to supported, version: {}'.format(version))
        return -1

    volume_name, hosts, replicate_num, has_arbiter, bricks, replicate = common.get_vs_cluster_info()
    if common.fault_point_result() or not volume_name:
        common.logger.error('failed to supported, cannot find volume name')
        return -1

    online_bricks = common.get_online_bricks(volume_name)
    if common.fault_point_result() or not online_bricks:
        common.logger.error('failed to supported, cannot find online_bricks')
        return -1

    if not common.check_str_is_gfid(gfid):
        common.logger.error('failed to supported, gfid: {}'.format(gfid))
        return -1

    return tier_data_repair(version, gfid, volume_name, hosts, replicate, online_bricks)


def _tier_data_repair(gfid):
    lock_file = common.get_vsfire_lock_file()
    with common.VsfireFlock(lock_file) as lock:
        ret = tier_data_repair_start(gfid)
        if ret:
            print common.Colored().red('执行失败')
        else:
            print common.Colored().cyan('执行成功')
        return ret